/**  @file bta_crop.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Alex Falkensteiner
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include "bta_crop.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifndef BTA_EXCLUDE_FILTERS

static int mymax(int a, int b) {
    if (a >= b) {
        return a;
    }
    else {
        return b;
    }
}

static int mymin(int a, int b) {
    if (a <= b) {
        return a;
    }
    else {
        return b;
    }
}


BTA_Status BFLTcropInit(BTA_FltCropConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    if (config->roiStartX < 0 || config->roiStartY < 0 || config->roiEndX < 0 || config->roiEndY < 0 ||
        (config->roiStartX < 1) ^ (config->roiStartY < 1) ||
        (config->roiStartX < 1) ^ (config->roiEndX < 1) ||
        (config->roiStartX < 1) ^ (config->roiEndY < 1)) {
        return BTA_StatusInvalidParameter;
    }
    BTA_FltCropInst *inst = (BTA_FltCropInst *)calloc(1, sizeof(BTA_FltCropInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
	inst->channelToProcess = config->channelToProcess;
	inst->roiStartX = config->roiStartX;
	inst->roiStartY = config->roiStartY;
	inst->roiEndX = config->roiEndX;
	inst->roiEndY = config->roiEndY;
	inst->channelIdResult = config->channelIdResult;
    inst->infoEventInst = infoEventInst;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTcropClose(BTA_FltHandle *handle) {
	BTA_FltCropInst **inst = (BTA_FltCropInst **)handle;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTcropApply(BTA_FltHandle handle, BTA_Frame **frame) {
	BTA_FltCropInst *inst = (BTA_FltCropInst *)handle;
    if (!inst || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }


	int chInd;
	int chCount = (*frame)->channelsLen;
	for (chInd = 0; chInd < chCount; chInd++) {
		BTA_Channel *channel = (*frame)->channels[chInd];
		if (!inst->channelToProcess || channel->id == inst->channelToProcess) {
            uint16_t roiStartX, roiStartY, roiEndX, roiEndY;
            if (inst->roiStartX < 1) {
                roiStartX = (int)(inst->roiStartX * channel->xRes);
                roiStartY = (int)(inst->roiStartY * channel->yRes);
                roiEndX = (int)(inst->roiEndX * channel->xRes);
                roiEndY = (int)(inst->roiEndY * channel->yRes);
            }
            else {
                roiStartX = (int)inst->roiStartX;
                roiStartY = (int)inst->roiStartY;
                roiEndX = (int)inst->roiEndX;
                roiEndY = (int)inst->roiEndY;
            }
			uint8_t bytesPerPixel = channel->dataFormat & 0xf;
			if (bytesPerPixel == 0) {
				return BTA_StatusNotSupported;
			}
            int x, y, i;
            int xy = 0;
            if (roiStartX > 0) {
                for (y = mymax(0, roiStartY - 1); y <= mymin(channel->yRes - 1, roiEndY + 1); y++) {
                    for (i = 0; i < bytesPerPixel; i++) {
                        channel->data[bytesPerPixel * (roiStartX - 1 + y * channel->xRes) + i] = 0;
                    }
                }
            }
            if (roiEndX < channel->xRes - 1) {
                for (y = mymax(0, roiStartY - 1); y <= mymin(channel->yRes - 1, roiEndY + 1); y++) {
                    for (i = 0; i < bytesPerPixel; i++) {
                        channel->data[bytesPerPixel * (roiEndX + 1 + y * channel->xRes) + i] = 0;
                    }
                }
            }
            if (roiStartY > 0) {
                for (x = mymax(0, roiStartX - 1); x <= mymin(channel->xRes - 1, roiEndX + 1); x++) {
                    for (i = 0; i < bytesPerPixel; i++) {
                        channel->data[bytesPerPixel * (x + (roiStartY - 1) * channel->xRes) + i] = 0;
                    }
                }
            }
            if (roiEndY < channel->yRes - 1) {
                for (x = mymax(0, roiStartX - 1); x <= mymin(channel->xRes - 1, roiEndX + 1); x++) {
                    for (i = 0; i < bytesPerPixel; i++) {
                        channel->data[bytesPerPixel * (x + (roiEndY + 1) * channel->xRes) + i] = 0;
                    }
                }
            }
			if (inst->channelIdResult) {
				// copy data to new channel
				uint16_t xResNew = roiEndX - roiStartX + 1;
				uint16_t yResNew = roiEndY - roiStartY + 1;
				uint8_t *dataNew = (uint8_t *)malloc(xResNew * yResNew * bytesPerPixel);
				for (y = roiStartY; y <= roiEndY; y++) {
					for (x = roiStartX; x <= roiEndX; x++) {
						for (i = 0; i < bytesPerPixel; i++) {
							dataNew[xy++] = channel->data[bytesPerPixel*x + bytesPerPixel*y*channel->xRes + i];
						}
					}
				}
				BTAinsertChannelDataIntoFrame(*frame, inst->channelIdResult, xResNew, yResNew, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, dataNew, xResNew * yResNew * bytesPerPixel);
			}
			else {
				// crop channel in place
				int x, y, xy = 0, i;
				for (y = roiStartY; y <= roiEndY; y++) {
					for (x = roiStartX; x <= roiEndX; x++) {
						for (i = 0; i < bytesPerPixel; i++) {
							channel->data[xy++] = channel->data[bytesPerPixel*x + bytesPerPixel*y*channel->xRes + i];
						}
					}
				}
				channel->xRes = roiEndX - roiStartX + 1;
				channel->yRes = roiEndY - roiStartY + 1;
				channel->dataLen = channel->xRes * channel->yRes * bytesPerPixel;
			}
		}
	}
	return BTA_StatusOk;
}
#endif
