/**  @file bta_intrinsic_undistort.c
*  
*    @brief This file implements the undistort by intrinsic parameters filter
*  
*    BLT_DISCLAIMER
*  
*    @author Alex Falkensteiner
*  
*    @cond svn
*  
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*  
*    @endcond
*/


#include "bta_intrinsic_undistort.h"
#include <stdlib.h>
#include <string.h>


// Local prototypes

BTA_Status BFLTintrinsicUndistortInit(BTA_FltIntrinsicUndistortConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    BTA_FltIntrinsicUndistortInst *inst;
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    inst = (BTA_FltIntrinsicUndistortInst *)calloc(1, sizeof(BTA_FltIntrinsicUndistortInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->infoEventInst = infoEventInst;
    inst->xRes = config->xRes;
    inst->yRes = config->yRes;
    memcpy(inst->cameraMatrixData, config->cameraMatrix, sizeof(config->cameraMatrix));
    cvInitMatHeader(&inst->cameraMatrix, 3, 3, CV_32F, inst->cameraMatrixData, 3 * sizeof(float));
    memcpy(inst->distCoeffsData, config->distCoeffs, sizeof(config->distCoeffs));
    cvInitMatHeader(&inst->distCoeffs, 5, 1, CV_32F, inst->distCoeffsData, sizeof(float));
    inst->width = -1;
    inst->height = -1;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTintrinsicUndistortClose(BTA_FltHandle *handle) {
    BTA_FltIntrinsicUndistortInst **inst = (BTA_FltIntrinsicUndistortInst **)handle;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTintrinsicUndistortApply(BTA_FltHandle handle, BTA_Frame **frame) {
    int chIn;
    BTA_FltIntrinsicUndistortInst *inst = (BTA_FltIntrinsicUndistortInst *)handle;
    if (!inst || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }
    for (chIn = 0; chIn < (*frame)->channelsLen; chIn++) {
        BTA_Channel *channel = (*frame)->channels[chIn];
        if (channel->id == BTA_ChannelIdColor && (!inst->xRes || channel->xRes == inst->xRes) && (!inst->yRes || channel->yRes == inst->yRes)) {
            struct CvMat matSrc, matDst;
            uint8_t *dataDst;
            int inputDataType, rowSize;
            if (channel->xRes == 0 || channel->yRes == 0) {
                // Nothing to undistort
                return BTA_StatusOk;
            }
            if (channel->xRes != inst->width || channel->yRes != inst->height) {
                // maps need initializing, create maps for remap()
	            inst->undistortMapx = cvCreateMat(channel->yRes, channel->xRes, CV_32FC1);
	            if (!inst->undistortMapx || !inst->undistortMapx->data.fl) {
                    return BTA_StatusRuntimeError;
	            }
	            inst->undistortMapy = cvCreateMat(channel->yRes, channel->xRes, CV_32FC1);
	            if (!inst->undistortMapy || !inst->undistortMapy->data.fl) {
                    return BTA_StatusRuntimeError;
	            }
	            cvInitUndistortMap(&inst->cameraMatrix, &inst->distCoeffs, inst->undistortMapx, inst->undistortMapy);
                inst->width = channel->xRes;
                inst->height = channel->yRes;
            }
            if (channel->dataFormat == BTA_DataFormatRgb24) {
                inputDataType = CV_8UC3;
                rowSize = 3 * channel->xRes;
            }
            else if (channel->dataFormat == BTA_DataFormatRgb565) {
                inputDataType = CV_16UC1;
                rowSize = 2 * channel->xRes;
            }
            else {
                return BTA_StatusNotSupported;
            }
            dataDst = (uint8_t *)malloc(channel->dataLen);
            if (!dataDst) {
                return BTA_StatusOutOfMemory;
            }
            cvInitMatHeader(&matSrc, channel->yRes, channel->xRes, inputDataType, channel->data, rowSize);
	        cvInitMatHeader(&matDst, channel->yRes, channel->xRes, inputDataType, dataDst, rowSize);
            cvRemap(&matSrc, &matDst, inst->undistortMapx, inst->undistortMapy, CV_INTER_NN, cvScalarAll(0));
            free(channel->data);
            channel->data = dataDst;
        }
    }
    return BTA_StatusOk;
}

