/**  @file bta_math.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Alex Falkensteiner
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include "bta_math.h"
#include <stdlib.h>
#include <string.h>
#include <float.h>



BTA_Status BFLTmathInit(BTA_FltMathConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    BTA_FltMathInst *inst = (BTA_FltMathInst *)calloc(1, sizeof(BTA_FltMathInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->mathType = config->mathType;
    inst->channelToProcess = config->channelToProcess;
    inst->channelIdResult = config->channelIdResult;
    inst->infoEventInst = infoEventInst;
    switch (inst->mathType) {

    case BTA_FltMathTypeMultFM1:
        if (!config->dataFM1 || !config->xResFM1 || !config->yResFM1) {
            free(inst);
            return BTA_StatusInvalidParameter;
        }
        inst->xResFM1 = config->xResFM1;
        inst->yResFM1 = config->yResFM1;
        inst->dataFM1 = (float *)malloc(inst->xResFM1*inst->yResFM1*sizeof(float));
        if (!inst->dataFM1) {
            free(inst);
            return BTA_StatusOutOfMemory;
        }
        memcpy(inst->dataFM1, config->dataFM1, inst->xResFM1*inst->yResFM1*sizeof(float));
        *handle = inst;
        return BTA_StatusOk;

    case BTA_FltMathTypeAbs:
    case BTA_FltMathTypeMax:
        *handle = inst;
        return BTA_StatusOk;

    default:
        return BTA_StatusNotSupported;
    }
}


BTA_Status BFLTmathClose(BTA_FltHandle *handle) {
    BTA_FltMathInst **inst = (BTA_FltMathInst **)handle;
    switch ((*inst)->mathType) {

    case BTA_FltMathTypeMultFM1:
        free((*inst)->dataFM1);
        break;
        
    default:
        break;
    }
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTmathApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_FltMathInst *inst = (BTA_FltMathInst *)handle;
    if (!inst || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }
    int chInd;
    int pxInd;
    int chCount = (*frame)->channelsLen;
    for (chInd = 0; chInd < chCount; chInd++) {
        BTA_Channel *channel = ((*frame)->channels)[chInd];
        if (channel->id == inst->channelToProcess) {
            switch (inst->mathType) {

            case BTA_FltMathTypeMultFM1:
                if (channel->xRes != inst->xResFM1 || channel->yRes != inst->yResFM1) {
                    return BTA_StatusRuntimeError;
                }
                switch (channel->dataFormat) {
                case BTA_DataFormatFloat32:
                    // TODO: consider channelIdResult
                    for (pxInd = 0; pxInd < inst->xResFM1*inst->yResFM1; pxInd++) {
                        ((float *)(channel->data))[pxInd] *= inst->dataFM1[pxInd];
                    }
                    break;
                default:
                    BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTmathApply: unsupported data format", channel->dataFormat);
                    return BTA_StatusNotSupported;
                }
                break;

            case BTA_FltMathTypeAbs:
                switch (channel->dataFormat) {

				case BTA_DataFormatUInt8:
				case BTA_DataFormatUInt16:
					break;

				case BTA_DataFormatSInt16:
                    // TODO: consider channelIdResult
                    for (pxInd = 0; pxInd < channel->xRes * channel->yRes; pxInd++) {
                        if (((int16_t *)(channel->data))[pxInd] < 0) {
                            ((int16_t *)(channel->data))[pxInd] *= -1;
                        }
                    }
                    break;


                case BTA_DataFormatFloat32:
                    // TODO: consider channelIdResult
                    for (pxInd = 0; pxInd < channel->xRes * channel->yRes; pxInd++) {
                        if (((float *)(channel->data))[pxInd] < 0) {
                            ((float *)(channel->data))[pxInd] *= -1.0f;
                        }
                    }
                    break;

                default:
                    BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTmathApply: unsupported data format", channel->dataFormat);
                    return BTA_StatusNotSupported;
                }
                break;

            case BTA_FltMathTypeMax:
                switch (channel->dataFormat) {

                case BTA_DataFormatUInt8: {
                    uint8_t max = 0;
                    for (pxInd = 0; pxInd < channel->xRes * channel->yRes; pxInd++) {
                        if (((uint8_t *)(channel->data))[pxInd] > max) {
                            max = ((uint8_t *)(channel->data))[pxInd];
                        }
                    }
                    if (inst->channelIdResult) {
                        uint8_t *data = (uint8_t *)malloc(sizeof(uint8_t));
                        *data = max;
                        BTAinsertChannelDataIntoFrame(*frame, inst->channelIdResult, 1, 1, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, data, sizeof(uint8_t));
                    }
                    else {
                        ((uint8_t *)(channel->data))[0] = max;
                        channel->xRes = 1;
                        channel->yRes = 1;
                        channel->dataLen = channel->xRes * channel->yRes * sizeof(uint8_t);
                    }
                    break;
                }

                case BTA_DataFormatUInt16: {
                    uint16_t max = 0;
                    for (pxInd = 0; pxInd < channel->xRes * channel->yRes; pxInd++) {
                        if (((uint16_t *)(channel->data))[pxInd] > max) {
                            max = ((uint16_t *)(channel->data))[pxInd];
                        }
                    }
                    if (inst->channelIdResult) {
                        uint16_t *data = (uint16_t *)malloc(sizeof(uint16_t));
                        *data = max;
                        BTAinsertChannelDataIntoFrame(*frame, inst->channelIdResult, 1, 1, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, (uint8_t *)data, sizeof(uint16_t));
                    }
                    else {
                        ((uint16_t *)(channel->data))[0] = max;
                        channel->xRes = 1;
                        channel->yRes = 1;
                        channel->dataLen = channel->xRes * channel->yRes * sizeof(uint16_t);
                    }
                    break;
                }

                case BTA_DataFormatSInt16: {
                    int16_t max = INT16_MIN;
                    for (pxInd = 0; pxInd < channel->xRes * channel->yRes; pxInd++) {
                        if (((int16_t *)(channel->data))[pxInd] > max) {
                            max = ((int16_t *)(channel->data))[pxInd];
                        }
                    }
                    if (inst->channelIdResult) {
                        int16_t *data = (int16_t *)malloc(sizeof(int16_t));
                        *data = max;
                        BTAinsertChannelDataIntoFrame(*frame, inst->channelIdResult, 1, 1, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, (uint8_t *)data, sizeof(int16_t));
                    }
                    else {
                        ((int16_t *)(channel->data))[0] = max;
                        channel->xRes = 1;
                        channel->yRes = 1;
                        channel->dataLen = channel->xRes * channel->yRes * sizeof(int16_t);
                    }
                    break;
                }

                case BTA_DataFormatFloat32: {
                    float max = FLT_MIN;
                    for (pxInd = 0; pxInd < channel->xRes * channel->yRes; pxInd++) {
                        if (((float *)(channel->data))[pxInd] > max) {
                            max = ((float *)(channel->data))[pxInd];
                        }
                    }
                    if (inst->channelIdResult) {
                        float *data = (float *)malloc(sizeof(float));
                        *data = max;
                        BTAinsertChannelDataIntoFrame(*frame, inst->channelIdResult, 1, 1, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, (uint8_t *)data, sizeof(float));
                    }
                    else {
                        ((float *)(channel->data))[0] = max;
                        channel->xRes = 1;
                        channel->yRes = 1;
                        channel->dataLen = channel->xRes * channel->yRes * sizeof(float);
                    }
                    break;
                }

                default:
                    BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTmathApply: unsupported data format", channel->dataFormat);
                    return BTA_StatusNotSupported;
                }
                break;

            default:
                BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTmathApply: unsupported math type", inst->mathType);
                return BTA_StatusNotSupported;
            }
        }
    }
    return BTA_StatusOk;
}