/**  @file bta_pixel_intrpl.c
*  
*    @brief This file implements the pixel interpolation filter
*  
*    BLT_DISCLAIMER
*  
*    @author Alex Falkensteiner
*  
*    @cond svn
*  
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*  
*    @endcond
*/


#include "bta_orientation.h"
#include <stdlib.h>
#include <string.h>

#ifndef BTA_EXCLUDE_FILTERS

// Local prototypes
static BTA_Status orientationApply(BTA_FltOrientationInst *inst, BTA_Channel **channel);

BTA_Status BFLTorientationInit(BTA_FltOrientationConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    BTA_FltOrientationInst *inst;
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    inst = (BTA_FltOrientationInst *)calloc(1, sizeof(BTA_FltOrientationInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->infoEventInst = infoEventInst;
    inst->orientationType = config->orientationType;
    inst->degrees = config->degrees;    
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTorientationClose(BTA_FltHandle *handle) {
    BTA_FltOrientationInst **inst = (BTA_FltOrientationInst **)handle;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTorientationApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_Status status;
    int chIn;
    if (!handle || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }
    BTA_FltOrientationInst *inst = (BTA_FltOrientationInst *)handle;
    for (chIn = 0; chIn < (*frame)->channelsLen; chIn++) {
        status = orientationApply(inst, &((*frame)->channels[chIn]));
        if (status != BTA_StatusOk) {
            return status;
        }
    }
    return BTA_StatusOk;
}


static BTA_Status orientationApply(BTA_FltOrientationInst *inst, BTA_Channel **channel) {
    int x, y, xy = 0;
    BTA_Channel *channelTemp;
    BTAcloneChannel(*channel, &channelTemp);
    switch (inst->orientationType) {
    case BTA_FltOrientationTypeFlipHor:
        break;

    case BTA_FltOrientationTypeFlipVer:
        break;

    case BTA_FltOrientationTypeRotate:
        switch (inst->degrees) {

        case 0:
            break;
        case 90:

            channelTemp->xRes = (*channel)->yRes;
            channelTemp->yRes = (*channel)->xRes;
            switch (channelTemp->dataFormat) {
            case BTA_DataFormatRgb565:
            case BTA_DataFormatSInt16:
            case BTA_DataFormatUInt16:
                for (x = 0; x < (*channel)->xRes; x++) {
                    for (y = (*channel)->yRes - 1; y >= 0; y--) {
                            ((uint16_t *)channelTemp->data)[xy++] = ((uint16_t *)(*channel)->data)[x + y * (*channel)->xRes];
                    }
                }
                break;
            default:
                BTAinfoEventHelper(inst->infoEventInst, 1, BTA_StatusNotSupported, "orientationApply: unsupported format", 0);
                break;
            }
            break;

        case 270:
            channelTemp->xRes = (*channel)->yRes;
            channelTemp->yRes = (*channel)->xRes;
            switch (channelTemp->dataFormat) {
            case BTA_DataFormatRgb565:
            case BTA_DataFormatSInt16:
            case BTA_DataFormatUInt16:
                for (x = (*channel)->xRes - 1; x >= 0; x--) {
                    for (y = 0; y < (*channel)->yRes; y++) {
                        ((uint16_t *)channelTemp->data)[xy++] = ((uint16_t *)(*channel)->data)[x + y * (*channel)->xRes];
                    }
                }
                break;
            default:
                BTAinfoEventHelper(inst->infoEventInst, 10, BTA_StatusNotSupported, "orientationApply: unsupported format", 0);
                break;
            }
            break;

        default:
            BTAinfoEventHelper(inst->infoEventInst, 10, BTA_StatusNotSupported, "orientationApply: unsupported angle", 0);
            break;
        }
        break;

    default:
        BTAinfoEventHelper(inst->infoEventInst, 10, BTA_StatusNotSupported, "orientationApply: unsupported type", 0);
        break;
    }
    BTAfreeChannel(channel);
    *channel = channelTemp;
    return BTA_StatusOk;
}

#endif
