///  @file example.cpp
///
///  @brief This example shows a common use case of the SDK
///
///  BLT_DISCLAIMER
///
///  @author Alex Falkensteiner


// Before going through this example code, please read the BltTofApi User Manual
// It can be found on https://support.bluetechnix.at/wiki/Bluetechnix_'Time_of_Flight'_API_v2.x


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
    #include <windows.h>
#elif defined PLAT_LINUX
    #include <unistd.h>
    #include <time.h>
#else
    #error "No platform defined"
#endif



// This example works on any platform with any camera in its default configuration.

// If this code is run on the a camera itself, (for example on the multi-ToF platform), the IP endpoints
// must be changed to the loopback interface. 
// Just define this macro:

//#define UseLoopBackInterface


// Local prototypes / little helpers
static void wait(int ms);
static uint32_t BTAgetTickCount();
void readStringFromConsole(char *inputStr, int inputStrLen);
static void errorHandling(BTA_Status status);
int readIntFromConsole();


// Local variables
static void *frameQueue = 0;


#define START_THREAD(x) ((void)(x))  // Not really starting the thread in this example! Only a showcase!


// Example for an implementation of the infoEventEx callback handler
// ***************************************************************************
static void BTA_CALLCONV infoEventEx(BTA_Handle handle, BTA_EventId eventId, int8_t *msg) {
    char statusString[100];
    BTAstatusToString(eventId, statusString, sizeof(statusString));
    printf(" %50.2f: infoEventEx: handle: 0x%p  (%s) %s\n", BTAgetTickCount()/1000.0, handle, statusString, msg);
}


// Example for an implementation of the infoEvent callback handler
// ***************************************************************************
static void BTA_CALLCONV infoEvent(BTA_EventId eventId, int8_t *msg) {
    char statusString[100];
    BTAstatusToString(eventId, statusString, sizeof(statusString));
    printf(" %50.2f: infoEvent:  (%s) %s\n", BTAgetTickCount() / 1000.0, statusString, msg);
}



// Example for an implementation of the frameArrivedEx callback handler
// ***************************************************************************
static void BTA_CALLCONV frameArrivedEx(BTA_Handle handle, BTA_Frame *frame) {
    // Do not free or alter the frame that was passed as a parameter
    // Processing of the frame can be done here, but one must be carefull not to block this thread, as it
    // must get back to its task of parsing frames coming from the camera.
    // For extensive processing, clone the frame like so:
    BTA_Frame *frameClone;
    BTA_Status status = BTAcloneFrame(frame, &frameClone);
    errorHandling(status);
    // The frameClone pointer can now be stored for later processing, in
    // a different thread, but this function should return now.
    // Use the provided frame queue:
    BFQenqueue(frameQueue, frameClone);
    return;

    // See frameProcessingRunFunction for an example
}


// Example showing how to process frames enqueued into frameQueue in frameArrived callback
static void frameProcessingRunFunction() {
    BTA_Frame *frameToProcess;
    while (1) {
        BTA_Status status = BFQdequeue(frameQueue, &frameToProcess, 500);
        if (status == BTA_StatusOk) {
            // Do the processing....................
            BTAfreeFrame(&frameToProcess);
        }
    }
}



// Example for an implementation of the frameArrivedEx callback handler
// ***************************************************************************
static void BTA_CALLCONV frameArrived(BTA_Frame *frame) {
    BTA_Status status;
    BTA_Frame *frameClone;
    status = BTAcloneFrame(frame, &frameClone);
    errorHandling(status);
    // The frameClone pointer can now be stored for later processing, or
    // a new thread can handle that, // but this function should return now.
    // Wherever the processing is done, before loosing the last reference to it,
    // the clone must be free'd
    BTAfreeFrame(&frameClone);
    // (Do not free or alter the frame that was passed as a parameter)
}



#ifdef FOR_EXPERTS_ONLY //****************************************************
// Example for an implementation of the progressReport callback handler
static void BTA_CALLCONV progressReport(BTA_Status status, uint8_t percentage) {
    if (percentage == 0 && status == BTA_StatusOk) {
        printf("Flash update started");
    }
    else if (percentage == 100 && status == BTA_StatusOk) {
        printf("Flash update finished with success");
    }
    else if (status == BTA_StatusOk){
        printf("Flash update progress: %d", percentage);
    }
    else {
        char statusString[100];
        BTAstatusToString(status, statusString, strlen(statusString));
        printf("Flash update failed: %s", statusString);
    }
}
#endif // ********************************************************************




// Example for an implementation of the discovery callback handler
// ***************************************************************************
static void BTA_CALLCONV discoveryCallback(BTA_Handle handle, BTA_DeviceInfo *deviceInfo) {
    printf("--------------------------------------------------------\n");
    printf("discoveryCallback: handle 0x%p device 0x%x, serial %d\n", handle, deviceInfo->deviceType, deviceInfo->serialNumber);
    printf("--------------------------------------------------------\n");
}



int main() {
    BTA_Status status;

    // Initialize frameQueue for later use in frameArrived callback
    BFQinit(5, BTA_QueueModeDropOldest, &frameQueue);

    // Start thread for frame processing
    START_THREAD(&frameProcessingRunFunction);

    // Check version
    //----------------------------------------------------------------------------------------------
    uint32_t verMaj, verMin, verNonFun;
    status = BTAgetVersion(&verMaj, &verMin, &verNonFun, 0, 0, 0, 0);
    errorHandling(status);
    if (verMaj != BTA_VER_MAJ || verMin != BTA_VER_MIN || verNonFun != BTA_VER_NON_FUNC) {
        printf("Please use the correct header files\n");
        printf("Hit <Return> to end the example\n");
        fgetc(stdin);
        exit(0);
    }

    
    // Initialization
    //----------------------------------------------------------------------------------------------
    // First, the library must be configured via the configuration c-structure.
    // The configuration structure must be initialized with standard values using the function
    // BTAinitConfig. The specific implementation of the library defines which parameters are 
    // required and which can be left out. The required parameters must then be set to a valid
    // value before calling BTAopen.

    BTA_Config config;
    printf("BTAinitConfig()\n");
    status = BTAinitConfig(&config);
    errorHandling(status);
    
    // Connection Parameters
    //----------------------------------------------------------------------------------------------
    // Depending on the library and the device, different connection parameters must be set.
    // Redundant connections (like TCP and UDP control connections) are connected sequencially and
    // exclusively
    // (UDP is tried first and only if it fails, TCP is connected)
    // Unnecessary information is ignored (BtaP100Lib ignores ethernet parameters)


    // IMPORTANT
    //----------------------------------------------------------------------------------------------
    // Before compiling and running this example or your own code, please learn the behavior of
    // the library regarding frame capturing.
    // The chapter "BTA_Config parameters" in the document "BltTofApi SUM" explains it shortly.
    //----------------------------------------------------------------------------------------------



    // Specify the interface
    //----------------------------------------------------------------------------------------------
    // The user should specify a device or a group of devices, so the library can select the right interface
    printf(" * Please select the interface to be used:\n");
    printf("   1: Ethernet\n");
    printf("   2: USB\n");
    printf("   3: UART\n");
    int conn = readIntFromConsole();
    switch (conn) {
    case 0:
        printf("Using default Eth\n");
    case 1:
        config.deviceType = BTA_DeviceTypeGenericEth;
        break;
    case 2:
        config.deviceType = BTA_DeviceTypeGenericUsb;
        break;
    case 3: {
        config.deviceType = BTA_DeviceTypeGenericUart;
        printf(" * Please insert the COMXXX number:\n");
        char inputStr[128] = { 0 };
        sprintf(inputStr, "COM%d", readIntFromConsole());
        config.uartPortName = (uint8_t *)inputStr;
        config.uartBaudRate = 115200;
        config.uartDataBits = 8;
        config.uartParity = 0;
        config.uartStopBits = 1;
        config.uartReceiverAddress = 0;
        config.uartTransmitterAddress = 254;
        break;
    }
    default:
        printf("Not a valid input: %d\n", conn);
        printf("Hit <Return> to end the example\n");
        fgetc(stdin);
        exit(0);
        break;
    }



    // Ethernet devices only
    // UDP data connection (normally multicast address)
#   ifndef UseLoopBackInterface
    uint8_t udpDataIpAddr[] = { 224, 0, 0, 1 };
#   else
    uint8_t udpDataIpAddr[] = { 127, 0, 0, 1 };
#   endif
    config.udpDataIpAddr = udpDataIpAddr;
    config.udpDataIpAddrLen = 4;
    config.udpDataPort = 10002;
    // TCP control connection (device's IP address)
#   ifndef UseLoopBackInterface
    uint8_t tcpDeviceIpAddr[] = { 192, 168, 0, 10 };
#   else
    uint8_t tcpDeviceIpAddr[] = { 127, 0, 0, 1 };
#   endif
    config.tcpDeviceIpAddr = tcpDeviceIpAddr;
    config.tcpDeviceIpAddrLen = 4;
    config.tcpControlPort = 10001;
    // UDP control outbound connection (device's IP address)
#   ifndef UseLoopBackInterface
    uint8_t udpControlOutIpAddr[] = { 192, 168, 0, 10 };
#   else
    uint8_t udpControlOutIpAddr[] = { 127, 0, 0, 1 };
#   endif
    config.udpControlOutIpAddr = udpControlOutIpAddr;
    config.udpControlOutIpAddrLen = 4;
    config.udpControlPort = 10003;
    // UDP control inbound connection (normally the host's IP address)
    // For current camera firmwares this is no longer necessary!
    //uint8_t udpControlInIpAddr[] = { 192, 168, 0, 1 };
    //config.udpControlInIpAddr = udpControlInIpAddr;
    //config.udpControlInIpAddrLen = 4;
    //config.udpControlCallbackPort = 10004;
    

    // If you want to receive a log from the library,
    // register a callback function for infoEvents.
    // This is most useful to get detailed error description(s) and we require you
    // to send us this log in order to give you technical support
    config.infoEvent = &infoEvent;
    config.infoEvent = 0;   // Not using this callback in this example
    config.infoEventEx = &infoEventEx;
    // ...and set the verbosity for infoEvents (the higher the more messages).
    // For a support request, please send the log with verbosity >= 9
    config.verbosity = 5;

    // If you want the log to be written directly into a file, then specify this parameter:
    config.infoEventFilename = (uint8_t *)"logfile.txt";

    // If you want to receive the frames immediately when they arrive,
    // register a callback function for incoming frames.
    // Please read chapter "BTA_Config parameters"___ in "BltTofApi SUM"
    config.frameArrived = &frameArrived;
    config.frameArrived = 0;  // Not using this callback in this example
    // The extended version also provides the handle,
    // so we can identify the connection session in case we have several cameras using the same callback function
    config.frameArrivedEx = &frameArrivedEx;
    // If both are defined, only the extendet callback is called!
    // (The same rules apply to infoEvents)

    // Choose whether and how queueing of frames is done.
    // Please read the BltTofApi SUM section "BTA_Config parameters" in order to
    // find out if queueing makes sense for your device and application
    // We choose 'DropOldest', so we always get the most recent frame
    config.frameQueueMode = BTA_QueueModeDropOldest;
    // Set the length of the frame queue.
    config.frameQueueLength = 5;
    
    // The frame mode can be configured in order to get the desired data channels in a frame from
    // the sensor / library:
    config.frameMode = BTA_FrameModeDistAmp;
    config.frameMode = BTA_FrameModeCurrentConfig;  // We don't want to change the frameMode in this example

    // USB (P100) connections only
    // If this parameter is left empty, the default lens configuration will be used.
    // Otherwise, use the name (and path) of a valid lens calibration file encoded in ASCII.
    config.calibFileName = (uint8_t *)"calibFile.bin";
    config.calibFileName = 0; // Not using this in this example

    // Select camera by serial number
    config.serialNumber = 120;  // the SDK accepts the connection only if the serial number matches
    config.serialNumber = 0;    // any

    // Select camera by serial product order number (PON)
    config.pon = (uint8_t *)"150-3001";  // the SDK accepts the connection only if the PON matches
    config.pon = 0;    // any

    
    // Connecting
    //----------------------------------------------------------------------------------------------
    // Now that the configuration structure is filled in, the connection is ready to be opened
    // The first infoEvents should fire while the library is connecting to the sensor.

    BTA_Handle btaHandle;
    printf("BTAopen()\n");
    status = BTAopen(&config, &btaHandle);
    errorHandling(status);


    // Connection Status
    //----------------------------------------------------------------------------------------------
    // It is possible to get the service state and connection state (not all connections support this)
    
    printf("Service running: %d\n", BTAisRunning(btaHandle));
    printf("Connection up: %d\n", BTAisConnected(btaHandle));



    printf("\n");
    // Querying Device Information
    //----------------------------------------------------------------------------------------------
    // The following example shows, how general information on the device can be retrieved.
    // The resulting struct may be only partly filled depending on the device’s capabilities.

    BTA_DeviceInfo *deviceInfo;
    printf("BTAgetDeviceInfo()\n");
    status = BTAgetDeviceInfo(btaHandle, &deviceInfo);
    errorHandling(status);
    printf("Device type: 0x%x\n", deviceInfo->deviceType);
    printf("BTAfreeDeviceInfo()\n");
    BTAfreeDeviceInfo(deviceInfo);
    

    printf("\n");
    // Frame-rate
    //----------------------------------------------------------------------------------------------
    // Read and change the frame rate via these functions

    float frameRate;
    printf("BTAgetFrameRate()\n");
    status = BTAgetFrameRate(btaHandle, &frameRate);
    errorHandling(status);
    printf("Framerate is %f\n", frameRate);
    printf("BTAsetFrameRate(%f)\n", frameRate);
    status = BTAsetFrameRate(btaHandle, frameRate);
    errorHandling(status);
    

    // Integration time
    //----------------------------------------------------------------------------------------------
    // Read and change the main integration time via this functions:

    uint32_t integrationTime;
    printf("BTAgetIntegrationTime()\n");
    status = BTAgetIntegrationTime(btaHandle, &integrationTime);
    errorHandling(status);
    printf("Integration time is %d\n", integrationTime);
    printf("BTAsetIntegrationTime(%d)\n", integrationTime);
    status = BTAsetIntegrationTime(btaHandle, integrationTime);
    errorHandling(status);
    

    // Modulation frequency
    //----------------------------------------------------------------------------------------------
    // Read and change the main modulation frequency via this functions

    uint32_t modulationFrequency;
    printf("BTAgetModulationFrequency()\n");
    status = BTAgetModulationFrequency(btaHandle, &modulationFrequency);
    errorHandling(status);
    printf("Modulation frequency is %d\n", modulationFrequency);
    printf("BTAsetModulationFrequency(%d)\n", modulationFrequency);
    status = BTAsetModulationFrequency(btaHandle, modulationFrequency);
    errorHandling(status);
    

    // Global offset
    //----------------------------------------------------------------------------------------------
    // Global stands for ‘all pixels’, meaning, that this offset is applied to all pixels.
    // It is, for all current devices, valid for the currently set modulation frequency.
    // It can only and should be set for all predefined modulation frequencies (see device’s SUM).
    // When changing the modulation frequency, the global offset reads differently.

    float offset;
    printf("BTAgetGlobalOffset()\n");
    status = BTAgetGlobalOffset(btaHandle, &offset);
    errorHandling(status);
    printf("Global offset is %f\n", offset);
    printf("BTAsetGlobalOffset(%f)\n", offset);
    status = BTAsetGlobalOffset(btaHandle, offset);
    errorHandling(status);
    

    printf("\n");
    // Register read/write
    //----------------------------------------------------------------------------------------------
    // Register operations are done via readRegister and writeRegister.
    // Most devices support mult-read and multi-write. The last parameter can be used to take
    // advantage of that feature.
    // For a normal read/write null can be passed.
    // Example for one register at address 0x20:
    uint32_t regValue;
    printf("BTAreadRegister()\n");
    status = BTAreadRegister(btaHandle, 0x20, &regValue, 0);
    errorHandling(status);
    printf("BTAwriteRegister()\n");
    status = BTAwriteRegister(btaHandle, 0x20, &regValue, 0);
    errorHandling(status);
    //The application must guarantee that register accesses are done exclusively. A read/write is
    // only called when the last read/write returned.
    

    // Frame mode
    //----------------------------------------------------------------------------------------------
    // The frame mode defines what channels the sonsor delivers to the library and/or
    // what data the library puts into a frame
    printf(" * Please select the frame mode to be used:\n");
    printf("    0: CurrentConfig\n");
    printf("    1: DistAmp\n");
    printf("    2: ZAmp\n");
    printf("    3: DistAmpFlags\n");
    printf("    4: XYZ\n");
    printf("    5: XYZAmp\n");
    printf("    6: DistAmpColor\n");
    printf("    7: XYZAmpFlags\n");
    printf("    8: RawPhases\n");
    printf("    9: Intensities\n");
    printf("   10: DistColor\n");
    printf("   11: DistAmpBalance\n");
    printf("   11: XYZColor\n");
    BTA_FrameMode frameMode = (BTA_FrameMode)readIntFromConsole();
    printf("BTAsetFrameMode(%d)\n", (int)frameMode);
    status = BTAsetFrameMode(btaHandle, frameMode);
    errorHandling(status);

    if (frameMode == BTA_FrameModeDistAmpColor || frameMode == BTA_FrameModeDistColor || frameMode == BTA_FrameModeXYZColor) {
        // RGB sensor config changes might take longer..
        printf("waiting...\n");
        wait(4000);
    }
    

    // Flush frame queue
    //----------------------------------------------------------------------------------------------
    // If BTA_Config was set for frame queueing, the queue can be flushed like so
    // This way the frame in the lib's queue is fresh regarding the just written parameters
    BTAflushFrameQueue(btaHandle);


    printf("\n");
    // LibParams
    //----------------------------------------------------------------------------------------------
    // LibParams are a way of interacting with the BltTofApi directly
    // In this example we read information about throughput
    float bytesReceived;
    status = BTAgetLibParam(btaHandle, BTA_LibParamBytesReceivedStream, &bytesReceived);
    errorHandling(status);
    printf("UDP data stream inbound data so far: %1.0f\n", bytesReceived);
    float framesParsed;
    status = BTAgetLibParam(btaHandle, BTA_LibParamFramesParsedCount, &framesParsed);
    errorHandling(status);
    printf("Frames parsed by API so far: %1.0f\n", framesParsed);


    // UDP stream endpoint address
    //----------------------------------------------------------------------------------------------
    // The UDP stream destination address needs to be changed if this example is to be run on the camera itsef
#   ifdef UseLoopBackInterface
    // set stream IP to 192.168.0.10
    //uint32_t eth0UdpStreamIp0 = 0x000a;
    //uint32_t eth0UdpStreamIp1 = 0xc0a8;
    uint32_t eth0UdpStreamIp0 = 0x0001;
    uint32_t eth0UdpStreamIp1 = 0x7f00;
    status = BTAwriteRegister(btaHandle, 0x024c, &eth0UdpStreamIp0, 0);
    errorHandling(status);
    status = BTAwriteRegister(btaHandle, 0x024d, &eth0UdpStreamIp1, 0);
    errorHandling(status);
    // Note: Eth0UdpStreamIp1 must be set in order to apply the changes
    wait(500);
    status = BTAgetLibParam(btaHandle, BTA_LibParamBytesReceivedStream, &bytesReceived);
    errorHandling(status);
    printf("UDP data stream inbound data so far: %1.0f\n", bytesReceived);
    status = BTAgetLibParam(btaHandle, BTA_LibParamFramesParsedCount, &framesParsed);
    errorHandling(status);
    printf("Frames parsed by API so far: %1.0f\n", framesParsed);
#   endif


    // Frame Retrieval
    //----------------------------------------------------------------------------------------------
    // Once the connection is established, the frameArrived callback (if not null) is called
    // whenever a frame is received from the device.
    // But a frame can also be requested actively from the library (if BTA_Config was set properly)
    // Please read the BltTofApi SUM chapter "BTA_Config parameters" for more details.

    BTA_Frame *frame;
    printf("BTAgetFrame()\n");
    status = BTAgetFrame(btaHandle, &frame, 5000);
    errorHandling(status);
    
    // The frame structure contains all the necessary information which can be accessed directly
    // or using the helper functions.
    // Let's dissect a frame:
    if (frame) {
        if (frame->channels) {
            for (int chInd = 0; chInd < frame->channelsLen; chInd++) {
                BTA_ChannelId id = frame->channels[chInd]->id;
                uint16_t xRes = frame->channels[chInd]->xRes;
                uint16_t yRes = frame->channels[chInd]->yRes;
                BTA_DataFormat dataFormat = frame->channels[chInd]->dataFormat;
                BTA_Unit unit = frame->channels[chInd]->unit;
                void* data = frame->channels[chInd]->data;
                double avg = 0;
                for (int y = 0; y < yRes; y++) {
                    for (int x = 0; x < xRes; x++) {
                        switch (dataFormat) {
                        case BTA_DataFormatUInt8:
                            avg += ((uint8_t *)data)[x + y*xRes];
                            break;
                        case BTA_DataFormatUInt16:
                            avg += ((uint16_t *)data)[x + y*xRes];
                            break;
                        case BTA_DataFormatUInt32:
                            avg += ((uint32_t *)data)[x + y*xRes];
                            break;
                        case BTA_DataFormatSInt16:
                            avg += ((int16_t *)data)[x + y*xRes];
                            break;
                        case BTA_DataFormatSInt32:
                            avg += ((int32_t *)data)[x + y*xRes];
                            break;
                        case BTA_DataFormatFloat32:
                            avg += ((float *)data)[x + y*xRes];
                            break;
                        case BTA_DataFormatFloat64:
                            avg += ((double *)data)[x + y*xRes];
                            break;
                        case BTA_DataFormatRgb565:
                            avg += ((uint16_t *)data)[x + y*xRes]; // should separate RGB channels
                            break;
                        case BTA_DataFormatRgb24:
                            avg += ((uint8_t *)data)[3 * (x + y*xRes)] + ((uint8_t *)data)[3 * (x + y*xRes) + 1] + ((uint8_t *)data)[3 * (x + y*xRes) + 2]; // should separate RGB channels
                            break;
                        default:
                            printf("\nunhandled data format\n");
                            break;
                        }
                    }
                }

                printf("\n");
                printf("channel %d has id %d. %dx%d pixels with unit %d\n", chInd, id, xRes, yRes, unit);
                if (xRes != 0 && yRes != 0) {
                    printf("The average over all pixels is %f\n", avg / xRes / yRes);
                }
            }
        }
    }


    // Frame Cleanup
    //----------------------------------------------------------------------------------------------
    // A successful call to getFrame or cloneFrame always demands for a call to BTAfreeFrame

    printf("BTAfreeFrame()\n");
    status = BTAfreeFrame(&frame);
    errorHandling(status);
    
    
    
    uint32_t intDataLen = 10;
    BTA_IntrinsicData *intData = (BTA_IntrinsicData *)malloc(intDataLen * sizeof(BTA_IntrinsicData));
    uint32_t extDataLen = 10;
    BTA_ExtrinsicData *extData = (BTA_ExtrinsicData *)malloc(extDataLen * sizeof(BTA_ExtrinsicData));
    status = BTAgetLensParameters(btaHandle, intData, &intDataLen, extData, &extDataLen);
    errorHandling(status);
    if (status == BTA_StatusOk) {
        for (int i = 0; i < intDataLen; i++) {
            printf(" Intrinsic data for lens %d:\n", i);
            printf("   xRes: %d\n", intData[i].xRes);
            printf("   yRes: %d\n", intData[i].yRes);
            printf("   fx: %f\n", intData[i].fx);
            printf("   fy: %f\n", intData[i].fy);
            printf("   cx: %f\n", intData[i].cx);
            printf("   cy: %f\n", intData[i].cy);
            printf("   k1: %f\n", intData[i].k1);
            printf("   k2: %f\n", intData[i].k2);
            printf("   k3: %f\n", intData[i].k3);
            printf("   k4: %f\n", intData[i].k4);
            printf("   k5: %f\n", intData[i].k5);
            printf("   k6: %f\n", intData[i].k6);
            printf("   p1: %f\n", intData[i].p1);
            printf("   p2: %f\n", intData[i].p2);
            printf("   lensId: %d\n", intData[i].lensId);
            printf("\n");
        }
        for (int i = 0; i < intDataLen; i++) {
            printf(" Extrinsic data for lens %d:\n", i);
            printf("   rotMat: %f  %f  %f\n", extData[i].rot[0], extData[i].rot[1], extData[i].rot[2]);
            printf("           %f  %f  %f\n", extData[i].rot[3], extData[i].rot[4], extData[i].rot[5]);
            printf("           %f  %f  %f\n", extData[i].rot[6], extData[i].rot[7], extData[i].rot[8]);
            printf("   trlVec: %f\n", extData[i].trl[0]);
            printf("           %f\n", extData[i].trl[1]);
            printf("           %f\n", extData[i].trl[2]);
            printf("\n");
        }
        printf("\n");
    }
    else {
        char statusString[100];
        BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
        printf("\nBTAgetLensParameters() returned %s\n\n", statusString);
    }



#   ifdef FOR_EXPERTS_ONLY // ***************************************************
    // Flash update
    //----------------------------------------------------------------------------------------------
    // A transfer of data to the device, typically to be saved in the device’s flash memory can be
    // performed by calling the library’s function BTAflashUpdate. It is mainly used to perform a 
    // firmware update. The struct BTA_FlashUpdateConfig is passed containing all the information
    // needed. The library defines which fields in the structure have to be filled depending on
    // the type of the update and/or data. Thus, the user must know how to configure the update
    // and what data to pass as parameter.

    // An example for updating the pixel list is:
    BTA_FlashUpdateConfig flashUpdateConfig;
    flashUpdateConfig.target = BTA_FlashTargetApplication;
    uint32_t dummydata = 0x12345678;
    flashUpdateConfig.data = (uint8_t *)&dummydata;
    flashUpdateConfig.dataLen = 4;
    printf("BTAflashUpdate()\n");
    status = BTAflashUpdate(btaHandle, &flashUpdateConfig, (FN_BTA_ProgressReport)&progressReport);
    errorHandling(status);

    // This function simplifies the process of a firmware update allowing the user to pass a
    // connection handle and the name of a binary firmware file. Internally it uses BTAflashUpdate()
    printf("BTAfirmwareUpdate()\n");
    status = BTAfirmwareUpdate(btaHandle, (uint8_t *)"fw.bin", (FN_BTA_ProgressReport)&progressReport);
    errorHandling(status);
    #endif // ********************************************************************



#   ifdef THIS_SECTION_COMPROMISES_REGISTER_VALUES_PERMANENTLY // ***************
    // Storing and restoring register settings
    //----------------------------------------------------------------------------------------------
    // The register values can be stored in flash memory in order to be preserved beyond a
    // reboot/power cycle
    printf("BTAwriteCurrentConfigToNvm()\n");
    status = BTAwriteCurrentConfigToNvm(btaHandle);
    errorHandling(status);

    // The restoring of the default configuration typically requires a reboot in order to take
    // immediate effect
    printf("BTArestoreDefaultConfig()\n");
    status = BTArestoreDefaultConfig(btaHandle);
    errorHandling(status);
#   endif // ********************************************************************



    printf("\n");
    // Grabbing the stream
    //----------------------------------------------------------------------------------------------
    // The Blt library is able to write frames to disk. Very conveniently, all the frame data
    // is stored in a *.bltstream file. These files can later be replayed by the BltTofApi as
    // if the same sensor was connected.

    BTA_GrabbingConfig grabbingConfig;
    printf("BTAinitGrabbingConfig()\n");
    status = BTAinitGrabbingConfig(&grabbingConfig);
    errorHandling(status);
    grabbingConfig.filename = (uint8_t *)"streamtest.bltstream"; // (ASCII coded)
    printf("BTAstartGrabbing()\n");
    status = BTAstartGrabbing(btaHandle, &grabbingConfig);
    errorHandling(status);

    // Grabbing is in progress. Let it grab for 3 seconds
    wait(3000);
    
    printf("Stop grabbing by calling BTAstartGrabbing(handle, 0)\n");
    status = BTAstartGrabbing(btaHandle, 0);
    errorHandling(status);


#   ifndef UseLoopBackInterface  // This code runs on the camera, so don't reset the camera, please
    // Device Reset
    //----------------------------------------------------------------------------------------------
    // If the device supports this functionality, a device reset can be performed via

    printf("BTAsendReset()\n");
    status = BTAsendReset(btaHandle);
    if (status == BTA_StatusDeviceUnreachable) {
        // Some devices don't always respond to this command becaus ethey already reset.
        // 
        printf(" * Please make sure the device did actually reset and hit <Return>\n");
        fgetc(stdin);
    }
    else {
        errorHandling(status);
    }
    printf(" * Please make sure the device is back up running from reset and accessible again, the hit <Return>\n");
    fgetc(stdin);
#   endif


    // Disconnecting
    //----------------------------------------------------------------------------------------------
    // When work is done and no other threads need to access the library's functions,
    // disconnect the sensor and stop the service by simply calling BTAclose

    printf("BTAclose()\n");
    status = BTAclose(&btaHandle);
    errorHandling(status);


    printf("\n");
    // Playback of previously grabbed bltstream file
    //----------------------------------------------------------------------------------------------
    BTA_Config config2;
    BTAinitConfig(&config2);
    config2.deviceType = BTA_DeviceTypeBltstream;
    config2.bltstreamFilename = (uint8_t *)"streamtest.bltstream"; // (ASCII coded)
    config2.infoEventEx = &infoEventEx;
    config2.verbosity = 1;
    config2.frameArrivedEx = &frameArrivedEx;
    BTA_Handle handleBltstream;
    status = BTAopen(&config2, &handleBltstream);
    errorHandling(status);
    float totalFrameCount;
    status = BTAgetLibParam(handleBltstream, BTA_LibParamStreamTotalFrameCount, &totalFrameCount);
    errorHandling(status);
    printf("The bltstream beeing played back contains a total of %0.0f frames\n", totalFrameCount);
    printf("Starting playback by setting BTA_LibParamStreamAutoPlaybackSpeed=1\n");
    BTAsetLibParam(handleBltstream, BTA_LibParamStreamAutoPlaybackSpeed, 1);
    // The stream is played back in background and the frames are delivered via the callback.
    // Alternatively, if queueing is enabled, one can also call BTAgetFrame
    wait(1000);
    printf("Setting the playback speed to factor 3 by setting BTA_LibParamStreamAutoPlaybackSpeed=3\n");
    BTAsetLibParam(handleBltstream, BTA_LibParamStreamAutoPlaybackSpeed, 3);
    errorHandling(status);
    wait(1000);
    printf("Done playing around, call BTAclose and finish the example\n");
    status = BTAclose(&handleBltstream);
    errorHandling(status);




    printf("\n");
    // Discovery
    //----------------------------------------------------------------------------------------------
    // The feature of lists connected devices

    BTA_DiscoveryConfig discoveryConfig;
    BTAinitDiscoveryConfig(&discoveryConfig);
    discoveryConfig.deviceType = (BTA_DeviceType)0;             // discover on all interfaces
    //discoveryConfig.deviceType = BTA_DeviceTypeGenericUsb;    // discover only USB
    //discoveryConfig.deviceType = BTA_DeviceTypeGenericEth;    // discover only Ethernet
    uint8_t broadcastIpAddr[] = { 192, 168, 0, 255 };
    discoveryConfig.broadcastIpAddr = broadcastIpAddr;          // the broadcast address of the subnet
    discoveryConfig.broadcastIpAddrLen = 4;
    discoveryConfig.broadcastPort = 0;                          // use standard port

    BTA_Handle discoveryHandle;
    printf("BTAstartDiscovery()\n");
    status = BTAstartDiscovery(&discoveryConfig, &discoveryCallback, &infoEvent, &discoveryHandle);
    errorHandling(status);
    printf("\nWaiting...\n");
    wait(3000);
    printf("BTAstopDiscovery()\n");
    status = BTAstopDiscovery(&discoveryHandle);
    errorHandling(status);


    // Free the frameQueue instance (and frames in it if there are)
    BFQclose(&frameQueue);

    printf("\n\nExample end reached successfully. Hit <Return> to end the example\n");
    fgetc(stdin);
}









////////////////////////////////////////////////////////////////////////////////////////////////////////
// little helpers


static void wait(int ms) {
#   if defined(PLAT_LINUX) || defined(linux)
    usleep(ms * 1000);
#   elif defined(PLAT_WINDOWS) || defined(WIN32) || defined(WIN64)
    Sleep(ms);
#   endif
}


static uint32_t BTAgetTickCount() {
#   ifdef PLAT_WINDOWS
    return GetTickCount();
#   elif defined PLAT_LINUX
    struct timespec ts;
    if (clock_gettime(CLOCK_MONOTONIC, &ts) != 0) {
        // what can I do?
        return 0;
    }
    return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
#   endif
}


void readStringFromConsole(char *inputStr, int inputStrLen) {
    fflush(stdin);
    char *result = fgets(inputStr, inputStrLen, stdin);
    if (!result) {
        inputStr[0] = 0;
        return;
    }
    while (strlen(inputStr) > 0 && (inputStr[strlen(inputStr) - 1] == '\n' || inputStr[strlen(inputStr) - 1] == '\r' || inputStr[strlen(inputStr) - 1] == ' ')) {
        inputStr[strlen(inputStr) - 1] = 0;
    }
    while (strlen(inputStr) > 0 && inputStr[0] == ' ') {
        int l = (int)strlen(inputStr);
        for (int i = 1; i < l - 1; i++) {
            inputStr[i - 1] = inputStr[i];
        }
    }
}


int readIntFromConsole() {
    char inputStr[50];
    readStringFromConsole(inputStr, 50);
    return atoi(inputStr);
}


static void errorHandling(BTA_Status status) {
    if (status != BTA_StatusOk) {
        char statusString[100];
        BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
        printf("error: %s (%d)\n", statusString, status);
        printf("Hit <Return> to end the example\n");
        fgetc(stdin);
        exit(0);
    }
}