///  @file example.cpp
///
///  @brief The examples illustrate commonly used functionalities of the library.
///
///  @author Birgit Hasenberger
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

#include "example_func.h"

// Local prototypes / little helpers
static int readIntFromConsole(); 
static void readStringFromConsole(char *inputStr, int inputStrLen);
static void errorHandling(BTA_Status status);



int main() {

	// Check version
	//----------------------------------------------------------------------------------------------
	BTA_Status status;
	uint32_t verMaj, verMin, verNonFun;
	status = BTAgetVersion(&verMaj, &verMin, &verNonFun, 0, 0, 0, 0);
	errorHandling(status);
	if (verMaj != BTA_VER_MAJ || verMin != BTA_VER_MIN || verNonFun != BTA_VER_NON_FUNC) {
		printf("Please use the correct header files\n");
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}


	// Choose example
	//----------------------------------------------------------------------------------------------
	printf("\n");
	printf("Welcome!\n");
	printf(" * Please select the example you would like to run:\n");
	printf("    1: Quick Start Ethernet - How to connect a device via ethernet, get a frame, and extract data from a frame.\n");
	printf("    2: Quick Start USB - How to connect a device via USB, get a frame, and extract data from a frame.\n");
	printf("    3: Advanced Configuration Callbacks - How to set up callback handlers for info events and arrived frames.\n");
	printf("    4: Advanced Configuration Queueing - How to set up and handle frame queueing.\n");
	printf("    5: Frame Processing - How to change the frame mode and process the resulting frames.\n");
	printf("    6: Register and Info - How to query device information, read/write register values, store/restore register values, query library parameters. \n");
	printf("    7: Bltstream - How to record and play back a bltstream file.\n");
	printf("    8: Update and Reset - How to perform a flash update and device reset, and how to set up the configuration if the code runs on the camera itself.\n");
	printf("    9: Device Discovery - How to set up and perform device discovery.\n");
	int example = readIntFromConsole();
	printf("Chosen example: %d\n", example);
	switch (example) {
	case 1:
		main_subEth();
		break;
	case 2:
		main_subUSB();
		break;
	case 3:
		main_subConfigCallbacks();
		break;
	case 4:
		main_subConfigQueueing();
		break;
	case 5:
		main_subFrame();
		break;
	case 6:
		main_subReg();
		break;
	case 7:
		main_subStream();
		break;
	case 8:
		main_subUpdate();
		break;
	case 9:
		main_subDiscovery();
		break;
	default:
		printf("Not a valid input: %d\n", example);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
		break;
	}

	printf("\n\nExample end reached successfully. Hit <Return> to end the example\n");
	fgetc(stdin);
}


static int readIntFromConsole() {
	char inputStr[50];
	readStringFromConsole(inputStr, 50);
	return atoi(inputStr);
}

static void readStringFromConsole(char* inputStr, int inputStrLen) {
	fflush(stdin);
	char* result = fgets(inputStr, inputStrLen, stdin);
	if (!result) {
		inputStr[0] = 0;
		return;
	}
	while (strlen(inputStr) > 0 && (inputStr[strlen(inputStr) - 1] == '\n' || inputStr[strlen(inputStr) - 1] == '\r' || inputStr[strlen(inputStr) - 1] == ' ')) {
		inputStr[strlen(inputStr) - 1] = 0;
	}
	while (strlen(inputStr) > 0 && inputStr[0] == ' ') {
		int l = (int)strlen(inputStr);
		for (int i = 1; i < l - 1; i++) {
			inputStr[i - 1] = inputStr[i];
		}
	}
}

static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}
