///  @file subConfigCallbacks.cpp
///
///  @brief This example illustrates how to set up callback handlers for info events and 
///  arrived frames.
///
///  @author Birgit Hasenberger, Alex Falkensteiner
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

static void errorHandling(BTA_Status status);
static uint32_t BTAgetTickCount();
static void wait(int ms);
uint32_t hours, min, sec, msec, musec;
uint32_t timeArr;


// Example for an implementation of the frameArrivedEx callback handler
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV frameArrivedEx(BTA_Handle handle, BTA_Frame *frame) {
	// This function is called whenever a frame arrives and can be used to process the frame. 
	// However, the function must return before a new frame can be parsed. This means one must 
	// be careful not to block this thread. For extensive processing, use the queueing 
	// framework of the library (more details in example "Advanced Configuration Queueing") or 
	// clone the frame (via BTAcloneFrame) and manually add it to a queue (via BFQenqueue).
	// Note that the frame that was passed as a parameter should NOT be freed or altered.

	timeArr = frame->timeStamp;
	musec = timeArr % 1000;
	timeArr /= 1000;
	msec = timeArr % 1000;
	timeArr /= 1000;
	sec = timeArr % 60;
	timeArr /= 60;
	min = timeArr % 60;
	hours = timeArr / 60;
	printf("Frame arrived: Frame no. %d at %02d:%02d:%02d.%03d %03d.\n", frame->frameCounter, hours, min, sec, msec, musec);
	return;
}

// Example for an implementation of the infoEventEx callback handler
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV infoEventEx(BTA_Handle handle, BTA_EventId eventId, int8_t *msg) {
	char statusString[100];
	BTAstatusToString(eventId, statusString, sizeof(statusString));
	printf(" %50.2f: infoEventEx: handle: 0x%p  (%s) %s\n", BTAgetTickCount() / 1000.0, handle, statusString, msg);
}


void main_subConfigCallbacks() {

	// Purpose of this example
	//----------------------------------------------------------------------------------------------
	// In this example, the use of callback functions is illustrated. During the configuration of 
	// the library, callback handlers for info events and arrived frames are defined. For the 
	// purpose of this example, the info-event callback function prints information about the event 
	// such as the current status, the callback function for arrived frames prints the number and 
	// timestamp of the frame (see definitions above).

	BTA_Status status;
	BTA_Config config;

	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	errorHandling(status);

	config.deviceType = BTA_DeviceTypeGenericEth;

	config.udpDataAutoConfig = 1;
	config.udpDataIpAddr = 0;
	config.udpDataIpAddrLen = 0;
	config.udpDataPort = 0;

	uint8_t tcpDeviceIpAddr[] = { 192, 168, 0, 10 };
	config.tcpDeviceIpAddr = tcpDeviceIpAddr;
	config.tcpDeviceIpAddrLen = 4;
	config.tcpControlPort = 10001;

	uint8_t udpControlOutIpAddr[] = { 192, 168, 0, 10 };
	config.udpControlOutIpAddr = udpControlOutIpAddr;
	config.udpControlOutIpAddrLen = 4;
	config.udpControlPort = 10003;

	config.frameMode = BTA_FrameModeDistAmp;


	// Registration of callback function for info events
	//----------------------------------------------------------------------------------------------
	// If you want to receive a log from the library, register a callback function for infoEvents.
	// This is most useful to get detailed error description(s) and we require you to send us this 
	// log in order to give you technical support. The verbosity for info events can be set, where 
	// a higher number means more messages. For a support request, please send the log with 
	// verbosity >= 9. A file name can be specified if the log should be written directly into a 
	// file.
	config.infoEventEx = &infoEventEx;
	config.verbosity = 5;
	config.infoEventFilename = (uint8_t *)"logfile.txt";


	// Registration of callback function for arrived frames
	//----------------------------------------------------------------------------------------------
	// If you want to receive the frames immediately when they arrive, register a callback function 
	// for incoming frames. The extended version also provides the handle, so we can identify the 
	// connection session in case several devices use the same callback function. If both the 
	// regular and extended callback are defined, only the extended callback is called.
	config.frameArrivedEx = &frameArrivedEx;


	// The first infoEvents should fire while the library is connecting to the sensor.
	// Once the connection is established, the frameArrived callback (if not null) is called
	// whenever a frame is received from the device.
	BTA_Handle btaHandle;
	printf("BTAopen()\n");
	status = BTAopen(&config, &btaHandle);
	errorHandling(status);

	printf("Service running: %d\n", BTAisRunning(btaHandle));
	printf("Connection up: %d\n", BTAisConnected(btaHandle));


	// For illustration purposes, the number of parsed frames after 100 ms is printed. More
	// details in example "Register and Info".
	float framesParsed; 
	wait(100);
	status = BTAgetLibParam(btaHandle, BTA_LibParamFramesParsedCount, &framesParsed);
	errorHandling(status);
	printf("Frames parsed by API so far: %1.0f\n", framesParsed);

	printf("\n");
	printf("BTAclose()\n");
	status = BTAclose(&btaHandle);
	errorHandling(status);
}


static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}

static uint32_t BTAgetTickCount() {
#   ifdef PLAT_WINDOWS
	return GetTickCount();
#   elif defined PLAT_LINUX
	struct timespec ts;
	if (clock_gettime(CLOCK_MONOTONIC, &ts) != 0) {
		// what can I do?
		return 0;
	}
	return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
#   endif
}

static void wait(int ms) {
#   if defined(PLAT_LINUX) || defined(linux)
	usleep(ms * 1000);
#   elif defined(PLAT_WINDOWS) || defined(WIN32) || defined(WIN64)
	Sleep(ms);
#   endif
}
