///  @file subConfigQueueing.cpp
///
///  @brief This example illustrates how to set up and handle frame queueing.
///
///  @author Birgit Hasenberger, Alex Falkensteiner
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

static uint32_t BTAgetTickCount();
static void errorHandling(BTA_Status status);
static void wait(int ms);


// Example for an implementation of the infoEventEx callback handler (more details in example 
// "Advanced Configuration Callbacks")
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV infoEventEx(BTA_Handle handle, BTA_EventId eventId, int8_t *msg) {
	char statusString[100];
	BTAstatusToString(eventId, statusString, sizeof(statusString));
	printf(" %50.2f: infoEventEx: handle: 0x%p  (%s) %s\n", BTAgetTickCount() / 1000.0, handle, statusString, msg);
}


void main_subConfigQueueing() {

	// Purpose of this example
	//----------------------------------------------------------------------------------------------
	// In this example, the library is configured so that frames received from the device are 
	// stored in a queue. A frame from the queue is accessed via the BTAgetFrame function and the 
	// distance at the central pixel is extracted (similar to examples "Quick Start Ethernet" and 
	// "Quick Start USB".

	BTA_Status status;
	BTA_Config config;

	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	errorHandling(status);

	config.deviceType = BTA_DeviceTypeGenericEth;

	config.udpDataAutoConfig = 1;
	config.udpDataIpAddr = 0;
	config.udpDataIpAddrLen = 0;
	config.udpDataPort = 0;

	uint8_t tcpDeviceIpAddr[] = { 192, 168, 0, 10 };
	config.tcpDeviceIpAddr = tcpDeviceIpAddr;
	config.tcpDeviceIpAddrLen = 4;
	config.tcpControlPort = 10001;

	uint8_t udpControlOutIpAddr[] = { 192, 168, 0, 10 };
	config.udpControlOutIpAddr = udpControlOutIpAddr;
	config.udpControlOutIpAddrLen = 4;
	config.udpControlPort = 10003;


	// Queueing configuration
	//----------------------------------------------------------------------------------------------
	// Choose whether and how frames are queued. For example, when choosing 'DropOldest', the 
	// oldest frame in the queue is removed when the queue is full. The most recent frame is 
	// therefore always available. When choosing 'DropCurrent', the queue remains unchanged when it 
	// is full. The user can set the length of the queue via the frameQueueLength configuration 
	// parameter.
	config.frameQueueMode = BTA_QueueModeDropOldest;
	config.frameQueueLength = 50;

	config.frameMode = BTA_FrameModeDistAmp;

	config.infoEventEx = &infoEventEx;
	config.verbosity = 5;
	config.infoEventFilename = (uint8_t *)"logfile.txt";
	

	BTA_Handle btaHandle;
	printf("BTAopen()\n");
	status = BTAopen(&config, &btaHandle);
	errorHandling(status);

	printf("Service running: %d\n", BTAisRunning(btaHandle));
	printf("Connection up: %d\n", BTAisConnected(btaHandle));

	// For illustration purposes, the number of parsed frames after 100 ms is printed. More
	// details in example "Register and Info".
	float framesParsed;
	wait(100);
	status = BTAgetLibParam(btaHandle, BTA_LibParamFramesParsedCount, &framesParsed);
	errorHandling(status);
	printf("Frames parsed by API so far: %1.0f\n", framesParsed);


	BTA_Frame* frame;
	printf("BTAgetFrame()\n");
	status = BTAgetFrame(btaHandle, &frame, 5000);
	errorHandling(status);

	void* distBuffer;
	BTA_DataFormat dataFormat;
	BTA_Unit unit;
	uint16_t xRes, yRes;

	status = BTAgetDistances(frame, &distBuffer, &dataFormat, &unit, &xRes, &yRes);
	errorHandling(status);

	uint8_t x = xRes / 2;
	uint8_t y = yRes / 2;

	// Here, a commonly used data format and unit for the distance data are assumed to extract the 
	// distance at the central pixel. Some devices provide other data formats and units. Check the 
	// software user manual for the specific device and adapt the code accordingly.
	if ((dataFormat == BTA_DataFormatUInt16) && (unit == BTA_UnitMillimeter)) {
		printf("Distance at pixel [%d,%d] is %d mm.\n", x, y, ((uint16_t*)distBuffer)[x + y * xRes]);
	}
	else {
		printf("Unable to print distance at central pixel. Device provides unexpected data format and/or unit.\n");
		printf("Check the software user manual and adapt the example script accordingly.\n");
	}


	printf("BTAfreeFrame()\n");
	status = BTAfreeFrame(&frame);
	errorHandling(status);


	printf("\n");
	printf("BTAclose()\n");
	status = BTAclose(&btaHandle);
	errorHandling(status);
}



static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}


static uint32_t BTAgetTickCount() {
#   ifdef PLAT_WINDOWS
	return GetTickCount();
#   elif defined PLAT_LINUX
	struct timespec ts;
	if (clock_gettime(CLOCK_MONOTONIC, &ts) != 0) {
		// what can I do?
		return 0;
	}
	return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
#   endif
}

static void wait(int ms) {
#   if defined(PLAT_LINUX) || defined(linux)
	usleep(ms * 1000);
#   elif defined(PLAT_WINDOWS) || defined(WIN32) || defined(WIN64)
	Sleep(ms);
#   endif
}

