///  @file subDiscovery.cpp
///
///  @brief This example illustrates how to set up and perform device discovery.
///
///  @author Birgit Hasenberger, Alex Falkensteiner
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

static void errorHandling(BTA_Status status);
static uint32_t BTAgetTickCount();
static void wait(int ms);

BTA_DeviceType FoundDeviceType;
uint8_t* FoundUdpDataIpAddr;
uint8_t FoundUdpDataIpAddrLen;
uint16_t FoundUdpDataPort;
uint8_t* FoundDeviceIpAddr;
uint8_t FoundDeviceIpAddrLen;
uint16_t FoundTcpControlPort;
uint16_t FoundUdpControlPort;
uint8_t* FoundPon;
uint32_t FoundSerialNumber;
int i;

// Example for an implementation of the infoEventEx callback handler (more details in example 
// "Advanced Configuration Callbacks")
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV infoEvent(BTA_EventId eventId, int8_t *msg) {
	char statusString[100];
	BTAstatusToString(eventId, statusString, sizeof(statusString));
	printf(" %50.2f: infoEvent:  (%s) %s\n", BTAgetTickCount() / 1000.0, statusString, msg);
}

// Example for an implementation of the discovery callback handler
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV discoveryCallback(BTA_Handle handle, BTA_DeviceInfo *deviceInfo) {
	// This function prints information about the found device and writes connection parameters 
	// which makes it possible to connect to the device after discovery.
	printf("--------------------------------------------------------\n");
	printf("discoveryCallback: handle 0x%p device 0x%x, serial %d\n", handle, deviceInfo->deviceType, deviceInfo->serialNumber);
	printf("--------------------------------------------------------\n");

	FoundDeviceType = deviceInfo->deviceType;

	FoundUdpDataIpAddrLen = deviceInfo->udpDataIpAddrLen;
	FoundUdpDataIpAddr = (uint8_t *)malloc(FoundUdpDataIpAddrLen * sizeof(uint8_t));
	for (i = 0; i < FoundUdpDataIpAddrLen; i++) {
		FoundUdpDataIpAddr[i] = deviceInfo->udpDataIpAddr[i];
	}
	FoundUdpDataPort = deviceInfo->udpDataPort;

	FoundDeviceIpAddrLen = deviceInfo->deviceIpAddrLen;
	FoundDeviceIpAddr = (uint8_t*)malloc(FoundDeviceIpAddrLen * sizeof(uint8_t));
	for (i = 0; i < FoundDeviceIpAddrLen; i++) {
		FoundDeviceIpAddr[i] = deviceInfo->deviceIpAddr[i];
	}
	FoundTcpControlPort = deviceInfo->tcpControlPort;
	FoundUdpControlPort = deviceInfo->udpControlPort;

	FoundPon = deviceInfo->productOrderNumber;
	FoundSerialNumber = deviceInfo->serialNumber;
}


void main_subDiscovery() {

	// Purpose of this example
	//----------------------------------------------------------------------------------------------
	// In this example, the use of device discovery is illustrated. The discovery feature produces 
	// a list of available devices that can then be used to establish a connection.

	BTA_Status status;
	

	// Configuration of device discovery
	//----------------------------------------------------------------------------------------------

	printf("\n");
	BTA_DiscoveryConfig discoveryConfig;
	BTAinitDiscoveryConfig(&discoveryConfig);
	discoveryConfig.deviceType = (BTA_DeviceType)0;             // discover on all interfaces
	//discoveryConfig.deviceType = BTA_DeviceTypeGenericUsb;    // discover only USB
	//discoveryConfig.deviceType = BTA_DeviceTypeGenericEth;    // discover only Ethernet
	uint8_t broadcastIpAddr[] = { 192, 168, 0, 255 };
	discoveryConfig.broadcastIpAddr = broadcastIpAddr;          // the broadcast address of the subnet
	discoveryConfig.broadcastIpAddrLen = 4;
	discoveryConfig.broadcastPort = 0;                          // use standard port


	// Running device discovery
	//----------------------------------------------------------------------------------------------

	BTA_Handle discoveryHandle;
	printf("BTAstartDiscovery()\n");
	status = BTAstartDiscovery(&discoveryConfig, &discoveryCallback, &infoEvent, &discoveryHandle);
	errorHandling(status);
	
	printf("\nWaiting...\n");
	wait(3000);

	printf("BTAstopDiscovery()\n");
	status = BTAstopDiscovery(&discoveryHandle);
	errorHandling(status);


	// Connecting to discovered device
	//----------------------------------------------------------------------------------------------

	BTA_Config config;
	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	errorHandling(status);

	if (FoundDeviceType) {
		if (BTAisEthDevice(FoundDeviceType)) {
			config.deviceType = BTA_DeviceTypeGenericEth;
			printf("Found device is Ethernet device.\n");

			config.udpDataIpAddr = FoundUdpDataIpAddr;
			config.udpDataIpAddrLen = FoundUdpDataIpAddrLen;
			config.udpDataPort = FoundUdpDataPort;

			config.tcpDeviceIpAddr = FoundDeviceIpAddr;
			config.tcpDeviceIpAddrLen = FoundDeviceIpAddrLen;
			config.tcpControlPort = FoundTcpControlPort;

			if (FoundUdpControlPort) {
				config.udpControlOutIpAddr = FoundDeviceIpAddr;
				config.udpControlOutIpAddrLen = FoundDeviceIpAddrLen;
				config.udpControlPort = FoundUdpControlPort;
			}
		}
		else if (BTAisUsbDevice(FoundDeviceType)) {
			config.deviceType = BTA_DeviceTypeGenericUsb;
			printf("Found device is USB device.\n");

			config.pon = FoundPon;
			config.serialNumber = FoundSerialNumber;
		}
		else {
			printf("Device type not recognized.\n");
			printf("\n");
		}

		config.frameQueueMode = BTA_QueueModeDropOldest;
		config.frameQueueLength = 5;

		config.frameMode = BTA_FrameModeDistAmp;

		config.infoEvent = &infoEvent;
		config.verbosity = 5;
		config.infoEventFilename = (uint8_t*)"logfile.txt";


		printf("\n");
		BTA_Handle btaHandle;
		printf("BTAopen()\n");
		status = BTAopen(&config, &btaHandle);
		errorHandling(status);

		printf("Service running: %d\n", BTAisRunning(btaHandle));
		printf("Connection up: %d\n", BTAisConnected(btaHandle));

		wait(10);
		printf("BTAclose()\n");
		status = BTAclose(&btaHandle);
		errorHandling(status);

	}

	else {
		printf("No device found.\n");
	}
}


static void wait(int ms) {
#   if defined(PLAT_LINUX) || defined(linux)
	usleep(ms * 1000);
#   elif defined(PLAT_WINDOWS) || defined(WIN32) || defined(WIN64)
	Sleep(ms);
#   endif
}

static uint32_t BTAgetTickCount() {
#   ifdef PLAT_WINDOWS
	return GetTickCount();
#   elif defined PLAT_LINUX
	struct timespec ts;
	if (clock_gettime(CLOCK_MONOTONIC, &ts) != 0) {
		// what can I do?
		return 0;
	}
	return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
#   endif
}

static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}
