///  @file subFrame.cpp
///
///  @brief This example illustrates how to change the frame mode and process the resulting 
///  frames.
///
///  @author Birgit Hasenberger, Alex Falkensteiner
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

static uint32_t BTAgetTickCount();
static void errorHandling(BTA_Status status);
static void wait(int ms);
static void readStringFromConsole(char* inputStr, int inputStrLen);
static int readIntFromConsole();


// Example for an implementation of the infoEventEx callback handler (more details in example 
// "Advanced Configuration Callbacks")
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV infoEventEx(BTA_Handle handle, BTA_EventId eventId, int8_t* msg) {
	char statusString[100];
	BTAstatusToString(eventId, statusString, sizeof(statusString));
	printf(" %50.2f: infoEventEx: handle: 0x%p  (%s) %s\n", BTAgetTickCount() / 1000.0, handle, statusString, msg);
}


void main_subFrame() {

	// Purpose of this example
	//----------------------------------------------------------------------------------------------
	// In this example, the user chooses a frame mode and thus the type of data available for 
	// processing. To illustrate how the data in a frame can be processed, the resolution, the 
	// unit, and the average value is returned for each channel in the frame.

	BTA_Status status;
	BTA_Config config;

	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	errorHandling(status);

	config.deviceType = BTA_DeviceTypeGenericEth;

	config.udpDataAutoConfig = 1;
	config.udpDataIpAddr = 0;
	config.udpDataIpAddrLen = 0;
	config.udpDataPort = 0;

	uint8_t tcpDeviceIpAddr[] = { 192, 168, 0, 10 };
	config.tcpDeviceIpAddr = tcpDeviceIpAddr;
	config.tcpDeviceIpAddrLen = 4;
	config.tcpControlPort = 10001;
	
	uint8_t udpControlOutIpAddr[] = { 192, 168, 0, 10 };
	config.udpControlOutIpAddr = udpControlOutIpAddr;
	config.udpControlOutIpAddrLen = 4;
	config.udpControlPort = 10003;

	config.frameQueueMode = BTA_QueueModeDropOldest;
	config.frameQueueLength = 50;

	config.frameMode = BTA_FrameModeDistAmp;

	config.infoEventEx = &infoEventEx;
	config.verbosity = 5;
	config.infoEventFilename = (uint8_t*)"logfile.txt";


	BTA_Handle btaHandle;
	printf("BTAopen()\n");
	status = BTAopen(&config, &btaHandle);
	errorHandling(status);

	printf("Service running: %d\n", BTAisRunning(btaHandle));
	printf("Connection up: %d\n", BTAisConnected(btaHandle));


	// Frame mode
	//----------------------------------------------------------------------------------------------
	// The frame mode defines which channels the sonsor delivers to the library and/or which data 
	// the library puts into a frame.

	printf(" * Please select the frame mode to be used:\n");
	printf("    0: CurrentConfig\n");
	printf("    1: DistAmp\n");
	printf("    2: ZAmp\n");
	printf("    3: DistAmpFlags\n");
	printf("    4: XYZ\n");
	printf("    5: XYZAmp\n");
	printf("    6: DistAmpColor\n");
	printf("    7: XYZAmpFlags\n");
	printf("    8: RawPhases\n");
	printf("    9: Intensities\n");
	printf("   10: DistColor\n");
	printf("   11: DistAmpBalance\n");
	printf("   12: XYZColor\n");
	BTA_FrameMode frameMode = (BTA_FrameMode)readIntFromConsole();
	printf("BTAsetFrameMode(%d)\n", (int)frameMode);
	status = BTAsetFrameMode(btaHandle, frameMode);
	errorHandling(status);

	// Since configuration changes involving the RGB sensor might take longer, we wait for a few 
	// seconds before moving on.
	if (frameMode == BTA_FrameModeDistAmpColor || frameMode == BTA_FrameModeDistColor || frameMode == BTA_FrameModeXYZColor) {
		printf("waiting...\n");
		wait(4000);
	}


	// Flush frame queue
	//----------------------------------------------------------------------------------------------
	// If BTA_Config was set for frame queueing, the queue can be flushed like so. This way the 
	// frame in the library's queue is fresh regarding the just written parameters.
	
	BTAflushFrameQueue(btaHandle);


	BTA_Frame* frame;
	printf("BTAgetFrame()\n");
	status = BTAgetFrame(btaHandle, &frame, 5000);
	errorHandling(status);


	// Acessing data in a frame
	//----------------------------------------------------------------------------------------------
	// The frame structure contains all the necessary information which can be accessed directly
	// or using the convenience functions (see examples "Quick Start Ethernet" or "Quick Start 
	// USB"). Here, a frame is dissected directly:

	if (frame) {
		if (frame->channels) {
			for (int chInd = 0; chInd < frame->channelsLen; chInd++) {
				BTA_ChannelId id = frame->channels[chInd]->id;
				uint16_t xRes = frame->channels[chInd]->xRes;
				uint16_t yRes = frame->channels[chInd]->yRes;
				BTA_DataFormat dataFormat = frame->channels[chInd]->dataFormat;
				BTA_Unit unit = frame->channels[chInd]->unit;
				void* data = frame->channels[chInd]->data;
				double avg = 0;
				for (int y = 0; y < yRes; y++) {
					for (int x = 0; x < xRes; x++) {
						switch (dataFormat) {
						case BTA_DataFormatUInt8:
							avg += ((uint8_t*)data)[x + y * xRes];
							break;
						case BTA_DataFormatUInt16:
							avg += ((uint16_t*)data)[x + y * xRes];
							break;
						case BTA_DataFormatUInt32:
							avg += ((uint32_t*)data)[x + y * xRes];
							break;
						case BTA_DataFormatSInt16:
							avg += ((int16_t*)data)[x + y * xRes];
							break;
						case BTA_DataFormatSInt32:
							avg += ((int32_t*)data)[x + y * xRes];
							break;
						case BTA_DataFormatFloat32:
							avg += ((float*)data)[x + y * xRes];
							break;
						case BTA_DataFormatFloat64:
							avg += ((double*)data)[x + y * xRes];
							break;
						case BTA_DataFormatRgb565:
							avg += ((uint16_t*)data)[x + y * xRes]; // should separate RGB channels
							break;
						case BTA_DataFormatRgb24:
							avg += ((uint8_t*)data)[3 * (x + y * xRes)] + ((uint8_t*)data)[3 * (x + y * xRes) + 1] + ((uint8_t*)data)[3 * (x + y * xRes) + 2]; // should separate RGB channels
							break;
						default:
							printf("\nunhandled data format\n");
							break;
						}
					}
				}

				printf("\n");
				printf("channel %d has id %d. %dx%d pixels with unit %d\n", chInd, id, xRes, yRes, unit);
				if (xRes != 0 && yRes != 0) {
					printf("The average over all pixels is %f\n", avg / xRes / yRes);
				}
			}
		}
	}


	printf("BTAfreeFrame()\n");
	status = BTAfreeFrame(&frame);
	errorHandling(status);


	printf("\n");
	printf("BTAclose()\n");
	status = BTAclose(&btaHandle);
	errorHandling(status);
}


static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}

static uint32_t BTAgetTickCount() {
#   ifdef PLAT_WINDOWS
	return GetTickCount();
#   elif defined PLAT_LINUX
	struct timespec ts;
	if (clock_gettime(CLOCK_MONOTONIC, &ts) != 0) {
		// what can I do?
		return 0;
	}
	return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
#   endif
}

static void wait(int ms) {
#   if defined(PLAT_LINUX) || defined(linux)
	usleep(ms * 1000);
#   elif defined(PLAT_WINDOWS) || defined(WIN32) || defined(WIN64)
	Sleep(ms);
#   endif
}

static void readStringFromConsole(char* inputStr, int inputStrLen) {
	fflush(stdin);
	char* result = fgets(inputStr, inputStrLen, stdin);
	if (!result) {
		inputStr[0] = 0;
		return;
	}
	while (strlen(inputStr) > 0 && (inputStr[strlen(inputStr) - 1] == '\n' || inputStr[strlen(inputStr) - 1] == '\r' || inputStr[strlen(inputStr) - 1] == ' ')) {
		inputStr[strlen(inputStr) - 1] = 0;
	}
	while (strlen(inputStr) > 0 && inputStr[0] == ' ') {
		int l = (int)strlen(inputStr);
		for (int i = 1; i < l - 1; i++) {
			inputStr[i - 1] = inputStr[i];
		}
	}
}

static int readIntFromConsole() {
	char inputStr[50];
	readStringFromConsole(inputStr, 50);
	return atoi(inputStr);
}
