///  @file subReg.cpp
///
///  @brief This example illustrates how to query device information, read/write register 
///  values, store/restore register values, query library parameters.
///
///  @author Birgit Hasenberger, Alex Falkensteiner
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

static uint32_t BTAgetTickCount();
static void errorHandling(BTA_Status status);


// Example for an implementation of the infoEventEx callback handler (more details in example 
// "Advanced Configuration Callbacks")
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV infoEventEx(BTA_Handle handle, BTA_EventId eventId, int8_t* msg) {
	char statusString[100];
	BTAstatusToString(eventId, statusString, sizeof(statusString));
	printf(" %50.2f: infoEventEx: handle: 0x%p  (%s) %s\n", BTAgetTickCount() / 1000.0, handle, statusString, msg);
}


void main_subReg() {

	// Purpose of this example
	//----------------------------------------------------------------------------------------------
	// In this example, device information is queried. Register values are read and written, and 
	// the register values are stored and restored from memory. Interaction with the library via 
	// library parameters is illustrated. 

	BTA_Status status;
	BTA_Config config;

	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	errorHandling(status);

	config.deviceType = BTA_DeviceTypeGenericEth;

	config.udpDataAutoConfig = 1;
	config.udpDataIpAddr = 0;
	config.udpDataIpAddrLen = 0;
	config.udpDataPort = 0;

	uint8_t tcpDeviceIpAddr[] = { 192, 168, 0, 10 };
	config.tcpDeviceIpAddr = tcpDeviceIpAddr;
	config.tcpDeviceIpAddrLen = 4;
	config.tcpControlPort = 10001;

	uint8_t udpControlOutIpAddr[] = { 192, 168, 0, 10 };
	config.udpControlOutIpAddr = udpControlOutIpAddr;
	config.udpControlOutIpAddrLen = 4;
	config.udpControlPort = 10003;

	config.frameQueueMode = BTA_QueueModeDropOldest;
	config.frameQueueLength = 5;

	config.infoEventEx = &infoEventEx;
	config.verbosity = 5;
	config.infoEventFilename = (uint8_t*)"logfile.txt";

	BTA_Handle btaHandle;
	printf("BTAopen()\n");
	status = BTAopen(&config, &btaHandle);
	errorHandling(status);

	printf("Service running: %d\n", BTAisRunning(btaHandle));
	printf("Connection up: %d\n", BTAisConnected(btaHandle));


	// Querying Device Information
	//----------------------------------------------------------------------------------------------
	// General information on the device can be retrieved using the following functions. The 
	// resulting struct may be only partly filled depending on the devices capabilities.
	
	printf("\n");
	BTA_DeviceInfo *deviceInfo;
	printf("BTAgetDeviceInfo()\n");
	status = BTAgetDeviceInfo(btaHandle, &deviceInfo);
	errorHandling(status);
	printf("Device type: 0x%x\n", deviceInfo->deviceType);
	printf("Serial number: %d\n", deviceInfo->serialNumber);
	printf("Uptime: %d s\n", deviceInfo->uptime);
	printf("BTAfreeDeviceInfo()\n");
	BTAfreeDeviceInfo(deviceInfo);


	// Frame-rate
	//----------------------------------------------------------------------------------------------
	// Read and change the frame rate via these functions:
	
	printf("\n");
	float frameRate;
	printf("BTAgetFrameRate()\n");
	status = BTAgetFrameRate(btaHandle, &frameRate);
	errorHandling(status);
	printf("Framerate is %f\n", frameRate);
	printf("BTAsetFrameRate(%f)\n", frameRate);
	status = BTAsetFrameRate(btaHandle, frameRate);
	errorHandling(status);


	// Integration time
	//----------------------------------------------------------------------------------------------
	// Read and change the main integration time via these functions:
	
	printf("\n");
	uint32_t integrationTime;
	printf("BTAgetIntegrationTime()\n");
	status = BTAgetIntegrationTime(btaHandle, &integrationTime);
	errorHandling(status);
	printf("Integration time is %d\n", integrationTime);
	printf("BTAsetIntegrationTime(%d)\n", integrationTime);
	status = BTAsetIntegrationTime(btaHandle, integrationTime);
	errorHandling(status);


	// Modulation frequency
	//----------------------------------------------------------------------------------------------
	// Read and change the main modulation frequency via these functions:

	printf("\n");
	uint32_t modulationFrequency;
	printf("BTAgetModulationFrequency()\n");
	status = BTAgetModulationFrequency(btaHandle, &modulationFrequency);
	errorHandling(status);
	printf("Modulation frequency is %d\n", modulationFrequency);
	printf("BTAsetModulationFrequency(%d)\n", modulationFrequency);
	status = BTAsetModulationFrequency(btaHandle, modulationFrequency);
	errorHandling(status);


	// Global offset
	//----------------------------------------------------------------------------------------------
	// Global means that this offset is applied to all pixels. It is, for all current devices, 
	// valid for the currently set modulation frequency. It can only and should be set for all 
	// predefined modulation frequencies (see devices software user manual). When changing the 
	// modulation frequency, the global offset reads differently.

	printf("\n");
	float offset;
	printf("BTAgetGlobalOffset()\n");
	status = BTAgetGlobalOffset(btaHandle, &offset);
	errorHandling(status);
	printf("Global offset is %f\n", offset);
	printf("BTAsetGlobalOffset(%f)\n", offset);
	status = BTAsetGlobalOffset(btaHandle, offset);
	errorHandling(status);


	// Register read/write
	//----------------------------------------------------------------------------------------------
	// Register operations are done via the readRegister and writeRegister functions. Most devices 
	// support multi-read and multi-write. The last parameter can be used to take advantage of that 
	// feature. For a normal read/write null can be passed for the last parameter. The application 
	// must guarantee that register accesses are done exclusively. A read/write is only called when 
	// the last read/write returned. Here, one register at address 0x20 is read and written:

	printf("\n");
	uint32_t regValue;
	printf("BTAreadRegister()\n");
	status = BTAreadRegister(btaHandle, 0x20, &regValue, 0);
	errorHandling(status);
	printf("BTAwriteRegister()\n");
	status = BTAwriteRegister(btaHandle, 0x20, &regValue, 0);
	errorHandling(status);


#	ifdef THIS_SECTION_COMPROMISES_REGISTER_VALUES_PERMANENTLY // ***************
   // Storing and restoring register settings
   //----------------------------------------------------------------------------------------------
   // The register values can be stored in flash memory in order to be preserved beyond a
   // reboot/power cycle
	printf("BTAwriteCurrentConfigToNvm()\n");
	status = BTAwriteCurrentConfigToNvm(btaHandle);
	errorHandling(status);

	// The restoring of the default configuration typically requires a reboot in order to take
	// immediate effect
	printf("BTArestoreDefaultConfig()\n");
	status = BTArestoreDefaultConfig(btaHandle);
	errorHandling(status);
#   endif // ********************************************************************


	// LibParams
	//----------------------------------------------------------------------------------------------
	// LibParams are a way of interacting with the BltTofApi directly. Here, information about 
	// throughput is read:

	printf("\n");
	float bytesReceived;
	status = BTAgetLibParam(btaHandle, BTA_LibParamBytesReceivedStream, &bytesReceived);
	errorHandling(status);
	printf("UDP data stream inbound data so far: %1.0f\n", bytesReceived);
	float framesParsed;
	status = BTAgetLibParam(btaHandle, BTA_LibParamFramesParsedCount, &framesParsed);
	errorHandling(status);
	printf("Frames parsed by API so far: %1.0f\n", framesParsed);


	printf("\n");
	printf("BTAclose()\n");
	status = BTAclose(&btaHandle);
	errorHandling(status);
}


static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}


static uint32_t BTAgetTickCount() {
#   ifdef PLAT_WINDOWS
	return GetTickCount();
#   elif defined PLAT_LINUX
	struct timespec ts;
	if (clock_gettime(CLOCK_MONOTONIC, &ts) != 0) {
		// what can I do?
		return 0;
	}
	return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
#   endif
}
