///  @file subUSB.cpp
///
///  @brief This example illustrates how to connect a device via USB, get a frame, and extract 
///  data from a frame.
///
///  @author Birgit Hasenberger, Alex Falkensteiner
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

static void errorHandling(BTA_Status status);

void main_subUSB() {

	// Purpose of this example
	//----------------------------------------------------------------------------------------------
	// In this example, control and data connections to the device are set up, a frame is extracted 
	// and the distance at the central pixel of that frame is returned. The default parameters for 
	// USB connections are used. Refer to the example "Quick Start Ethernet" for an equivalent 
	// procedure using a device connected via ethernet.

	BTA_Status status;
	BTA_Config config;


	// Initialization
	//----------------------------------------------------------------------------------------------
	// First, the library must be configured via the configuration c-structure. The configuration 
	// structure must be initialized with standard values using the function BTAinitConfig. The 
	// specific implementation of the library defines which parameters are required and which can 
	// be left out. The required parameters must then be set to a valid value before calling 
	// BTAopen.

	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	errorHandling(status);


	// Connection Parameters
	//----------------------------------------------------------------------------------------------
	// Depending on the library and the device, different connection parameters must be set.
	// Unnecessary information is ignored (for example, the library ignores ethernet parameters if 
	// a USB interface is selected).

	// The user should specify a device or a group of devices, so the library can select the right 
	// interface.
	config.deviceType = BTA_DeviceTypeUsb;

	// The device to connect to can be selected via its serial number and product order number. 
	// For the purpose of this example, no restriction on those parameters is placed.
	config.serialNumber = 120;  // the SDK accepts the connection only if the serial number matches
	config.serialNumber = 0;    // any
	config.pon = (uint8_t*)"150-3001";  // the SDK accepts the connection only if the PON matches
	config.pon = 0;    // any

	// Choose whether and how queueing of frames is done. More details in examples "Advanced 
	// Configuration Callbacks" and "Advanced Configuration Queueing".
	config.frameQueueMode = BTA_QueueModeDropOldest;
	config.frameQueueLength = 5;

	// The frame mode can be configured in order to get the desired data channels in a frame 
	// from the device. More details in example "Frame Processing".
	config.frameMode = BTA_FrameModeDistAmp;


	// Connecting
	//----------------------------------------------------------------------------------------------
	// Now that the configuration structure is filled in, the connection is ready to be opened.

	printf("\n");
	BTA_Handle btaHandle;
	printf("BTAopen()\n");
	status = BTAopen(&config, &btaHandle);
	errorHandling(status);


	// Connection Status
	//----------------------------------------------------------------------------------------------
	// It is possible to return the service state and connection state (not all connections support 
	// this).

	printf("Service running: %d\n", BTAisRunning(btaHandle));
	printf("Connection up: %d\n", BTAisConnected(btaHandle));


	// Frame Retrieval
	//----------------------------------------------------------------------------------------------
	// Once the connection is established, frames can be retrieved via the frameArrived callback or 
	// they can be requested actively from the library via the BTAgetFrame function depending on 
	// the library configuration set earlier. For simplicity, a frame is obtained via BTAgetFrame 
	// from the frame queue in this example. More details in examples "Advanced Configuration 
	// Callbacks" and "Advanced Configuration Queueing".

	printf("\n");
	BTA_Frame* frame;
	printf("BTAgetFrame()\n");
	status = BTAgetFrame(btaHandle, &frame, 5000);
	errorHandling(status);


	// Distance Extraction
	//----------------------------------------------------------------------------------------------
	// Frames can contain several channels and various types of data. For easy data access, 
	// convenience functions obtain the relevant channel data and parameters from a frame. In this 
	// example, the channel containing distance data is extracted via the BTAgetDistances function. 
	// This type of convenience function is also available to obtain amplitudes, colors, flags, and 
	// x, y, and z coordinates. For more general methods of frame processing, refer to example 
	// "Frame Processing".

	void* distBuffer;
	BTA_DataFormat dataFormat;
	BTA_Unit unit;
	uint16_t xRes, yRes;

	status = BTAgetDistances(frame, &distBuffer, &dataFormat, &unit, &xRes, &yRes);
	errorHandling(status);

	uint8_t x = xRes / 2;
	uint8_t y = yRes / 2;

	// Here, a commonly used data format and unit for the distance data are assumed to extract the 
	// distance at the central pixel. Some devices provide other data formats and units. Check the 
	// software user manual for the specific device and adapt the code accordingly.
	if ((dataFormat == BTA_DataFormatUInt16) && (unit == BTA_UnitMillimeter)) {
		printf("Distance at pixel [%d,%d] is %d mm.\n", x, y, ((uint16_t*)distBuffer)[x + y * xRes]);
	}
	else {
		printf("Unable to print distance at central pixel. Device provides unexpected data format and/or unit.\n");
		printf("Check the software user manual and adapt the example script accordingly.\n");
	}


	// Frame Cleanup
	//----------------------------------------------------------------------------------------------
	// A successful call to getFrame or cloneFrame always demands for a call to BTAfreeFrame.

	printf("BTAfreeFrame()\n");
	status = BTAfreeFrame(&frame);
	errorHandling(status);


	// Disconnecting
	//----------------------------------------------------------------------------------------------
	// When work is done and no other threads need to access the library's functions, disconnect 
	// the device and stop the service by simply calling BTAclose.

	printf("\n");
	printf("BTAclose()\n");
	status = BTAclose(&btaHandle);
	errorHandling(status);
}


static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}
