///  @file subUpdate.cpp
///
///  @brief This example illustrates how to perform a flash update and device reset, and how to 
///  set up the configuration if the code runs on the camera itself.
///
///  @author Birgit Hasenberger, Alex Falkensteiner
///
///  For the API User Manual, the API Reference Manual, and further support, visit 
///  http://support.becom-group.com/wiki/Blt_ToF_API.


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bta.h>

#ifdef PLAT_WINDOWS
#include <windows.h>
#elif defined PLAT_LINUX
#include <unistd.h>
#include <time.h>
#else
#error "No platform defined"
#endif

// If this code is run on the a camera itself, (for example on the multi-ToF platform), the IP 
// endpoints must be changed to the loopback interface. To do this, define this macro:
//#define UseLoopBackInterface


static uint32_t BTAgetTickCount();
static void errorHandling(BTA_Status status);
static void readStringFromConsole(char* inputStr, int inputStrLen);
static int readIntFromConsole();


// Example for an implementation of the infoEventEx callback handler (more details in example 
// "Advanced Configuration Callbacks")
//----------------------------------------------------------------------------------------------
static void BTA_CALLCONV infoEventEx(BTA_Handle handle, BTA_EventId eventId, int8_t *msg) {
	char statusString[100];
	BTAstatusToString(eventId, statusString, sizeof(statusString));
	printf(" %50.2f: infoEventEx: handle: 0x%p  (%s) %s\n", BTAgetTickCount() / 1000.0, handle, statusString, msg);
}

#ifdef FOR_EXPERTS_ONLY //****************************************************
// Example for an implementation of the progressReport callback handler
static void BTA_CALLCONV progressReport(BTA_Status status, uint8_t percentage) {
	if (percentage == 0 && status == BTA_StatusOk) {
		printf("Flash update started");
	}
	else if (percentage == 100 && status == BTA_StatusOk) {
		printf("Flash update finished with success");
	}
	else if (status == BTA_StatusOk) {
		printf("Flash update progress: %d", percentage);
	}
	else {
		char statusString[100];
		BTAstatusToString(status, statusString, strlen(statusString));
		printf("Flash update failed: %s", statusString);
	}
}
#endif // ********************************************************************


void main_subUpdate() {

	// Purpose of this example
	//----------------------------------------------------------------------------------------------
	// In this example, the connected device can be reset and a firmware upgrade can be performed. 
	// For the latter, the macro FOR_EXPERTS_ONLY must be defined. This example also illustrates 
	// how the library must be configured if the code runs on the device itself (via the macro 
	// UseLoopBackInterface).

	BTA_Status status;
	BTA_Config config;

	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	errorHandling(status);

	config.deviceType = BTA_DeviceTypeGenericEth;

#   ifndef UseLoopBackInterface
	config.udpDataAutoConfig = 1;
	config.udpDataIpAddr = 0;
	config.udpDataIpAddrLen = 0;
	config.udpDataPort = 0;
#   else
	uint8_t udpDataIpAddr[] = { 127, 0, 0, 1 };
	config.udpDataIpAddr = udpDataIpAddr;
	config.udpDataIpAddrLen = 4;
	config.udpDataPort = 10002;
#   endif

#   ifndef UseLoopBackInterface
	uint8_t tcpDeviceIpAddr[] = { 192, 168, 0, 10 };
#   else
	uint8_t tcpDeviceIpAddr[] = { 127, 0, 0, 1 };
#   endif
	config.tcpDeviceIpAddr = tcpDeviceIpAddr;
	config.tcpDeviceIpAddrLen = 4;
	config.tcpControlPort = 10001;

#   ifndef UseLoopBackInterface
	uint8_t udpControlOutIpAddr[] = { 192, 168, 0, 10 };
#   else
	uint8_t udpControlOutIpAddr[] = { 127, 0, 0, 1 };
#   endif
	config.udpControlOutIpAddr = udpControlOutIpAddr;
	config.udpControlOutIpAddrLen = 4;
	config.udpControlPort = 10003;

	config.infoEventEx = &infoEventEx;
	config.verbosity = 5;
	config.infoEventFilename = (uint8_t *)"logfile.txt";

	config.frameQueueMode = BTA_QueueModeDropOldest;
	config.frameQueueLength = 5;

	BTA_Handle btaHandle;
	printf("BTAopen()\n");
	status = BTAopen(&config, &btaHandle);
	errorHandling(status);

	printf("Service running: %d\n", BTAisRunning(btaHandle));
	printf("Connection up: %d\n", BTAisConnected(btaHandle));


#   ifndef UseLoopBackInterface  // If this code runs on the camera, do not reset the device.

	printf(" * Would you like to reset the device?\n");
	printf("    0: No\n");
	printf("    1: Yes\n");
	if (readIntFromConsole() == 1) {


		// Device Reset
		//----------------------------------------------------------------------------------------------
		// If the device supports this functionality, a device reset can be performed via the function 
		// BTAsendReset.

		printf("BTAsendReset()\n");
		status = BTAsendReset(btaHandle);
		if (status == BTA_StatusDeviceUnreachable) {
			// Some devices do not always respond to this command because they already reset.
			
			printf(" * Please make sure the device did actually reset and hit <Return>\n");
			fgetc(stdin);
		}
		else {
			errorHandling(status);
		}
		printf(" * Please make sure the device is back up running from reset and accessible again, the hit <Return>\n");
		fgetc(stdin);
	}
	else {
		printf("Device not reset.\n");
	}

#   endif


#   ifdef FOR_EXPERTS_ONLY // ***************************************************
	// Flash update
	//----------------------------------------------------------------------------------------------
	// A transfer of data to the device, typically to be saved in the devices flash memory can be
	// performed by calling the librarys function BTAflashUpdate. It is mainly used to perform a 
	// firmware update. The struct BTA_FlashUpdateConfig is passed containing all the information
	// needed. The library defines which fields in the structure have to be filled depending on
	// the type of the update and/or data. Thus, the user must know how to configure the update
	// and what data to pass as parameter.

	// An example for updating the pixel list is:
	BTA_FlashUpdateConfig flashUpdateConfig;
	flashUpdateConfig.target = BTA_FlashTargetApplication;
	uint32_t dummydata = 0x12345678;
	flashUpdateConfig.data = (uint8_t *)&dummydata;
	flashUpdateConfig.dataLen = 4;
	printf("BTAflashUpdate()\n");
	status = BTAflashUpdate(btaHandle, &flashUpdateConfig, (FN_BTA_ProgressReport)&progressReport);
	errorHandling(status);

	// This function simplifies the process of a firmware update allowing the user to pass a
	// connection handle and the name of a binary firmware file. Internally it uses BTAflashUpdate()
	printf("BTAfirmwareUpdate()\n");
	status = BTAfirmwareUpdate(btaHandle, (uint8_t *)"fw.bin", (FN_BTA_ProgressReport)&progressReport);
	errorHandling(status);
#endif // ********************************************************************


	printf("BTAclose()\n");
	status = BTAclose(&btaHandle);
	errorHandling(status);
}


static void errorHandling(BTA_Status status) {
	if (status != BTA_StatusOk) {
		char statusString[100];
		BTAstatusToString(status, statusString, (uint16_t)sizeof(statusString));
		printf("error: %s (%d)\n", statusString, status);
		printf("Hit <Return> to end the example\n");
		fgetc(stdin);
		exit(0);
	}
}

static uint32_t BTAgetTickCount() {
#   ifdef PLAT_WINDOWS
	return GetTickCount();
#   elif defined PLAT_LINUX
	struct timespec ts;
	if (clock_gettime(CLOCK_MONOTONIC, &ts) != 0) {
		// what can I do?
		return 0;
	}
	return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
#   endif
}

static void readStringFromConsole(char* inputStr, int inputStrLen) {
	fflush(stdin);
	char* result = fgets(inputStr, inputStrLen, stdin);
	if (!result) {
		inputStr[0] = 0;
		return;
	}
	while (strlen(inputStr) > 0 && (inputStr[strlen(inputStr) - 1] == '\n' || inputStr[strlen(inputStr) - 1] == '\r' || inputStr[strlen(inputStr) - 1] == ' ')) {
		inputStr[strlen(inputStr) - 1] = 0;
	}
	while (strlen(inputStr) > 0 && inputStr[0] == ' ') {
		int l = (int)strlen(inputStr);
		for (int i = 1; i < l - 1; i++) {
			inputStr[i - 1] = inputStr[i];
		}
	}
}

static int readIntFromConsole() {
	char inputStr[50];
	readStringFromConsole(inputStr, 50);
	return atoi(inputStr);
}
