/**
 *    @file bta_eth_example.c
 *    @version 1.0.0
 *  
 *    @brief This example shows a common use case of the SDK
 *  
 *    BLT_DISCLAIMER
 *  
 *    @author Alex Falkensteiner, Harald Krapfenbauer
 */


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <assert.h>

#include <bta.h> /* BltTofApi */
#include <blt-tof-devs.h>


/* Externals */
extern int gpuInit(void); /* defined in gpu_opencl_example.c */
extern int gpuCreateAmplitudeMask(uint16_t *, uint16_t, uint16_t, uint16_t, uint16_t *); /* defined in gpu_opencl_example.c */
extern void gpuExit(void); /* defined in gpu_opencl_example.c */
extern int neonCreateAmplitudeMask(uint16_t *, uint16_t, uint16_t, uint16_t, uint16_t *);


static void BTA_CALLCONV infoEvent(BTA_EventId eventId, int8_t *msg)
{
	printf("   Callback: infoEvent (%d) %s\n", eventId, msg);

	/* This demonstrates the usage of the system log.
	   Log messages are kept in /mnt/logs/ directory.
	*/
	syslog(LOG_DEBUG, "Callback: infoEvent (%d) %s\n", eventId, msg);
}


static void BTA_CALLCONV frameArrived(BTA_Frame *frame)
{
	printf("   Callback: frameArrived FrameCounter %d\n", frame->frameCounter);

	syslog(LOG_DEBUG, "Callback: frameArrived FrameCounter %d\n", frame->frameCounter);

	// BTA_Status status;
	// BTA_Frame *frameClone;
	// // status = BTAcloneFrame(frame, &frameClone);
	// if (status != BTA_StatusOk) {
	// 	// failed to clone the frame
	// 	return;
	// }
	// // The frameClone pointer can now be stored for later processing, but this function should return now
	// // .....
	// // after you're done with frameClone, don't forget to free it
	// BTAfreeFrame(&frameClone);
}


int main()
{
	BTA_Status status;
	BTA_Config config;
	char statusString[100];
    
	/* Logging init */
	openlog("USERAPP", 0, LOG_USER);

	syslog(LOG_INFO, "Starting\n");

	/* The config struct must be initialized. */
	printf("BTAinitConfig()\n");
	status = BTAinitConfig(&config);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		fprintf(stderr, "Error: %s\n", statusString);
		return -1;
	}
    
	/* Ethernet devices need connection information. */
	/* UDP data connection */
	uint8_t udpDataIpAddr[] = { 127, 0, 0, 1 }; /* We use the loopback interface */
	config.udpDataIpAddr = udpDataIpAddr;
	config.udpDataIpAddrLen = 4;
	config.udpDataPort = 10002;
	/* TCP control connection (device's IP address) */
	uint8_t tcpDeviceIpAddr[] = { 127, 0, 0, 1 }; /* Connect to loopback interface */
	config.tcpDeviceIpAddr = tcpDeviceIpAddr;
	config.tcpDeviceIpAddrLen = 4;
	config.tcpControlPort = 10001;
    
	/* Optional settings for advanced functionalities */
	/*---------------------------------------------------------------- */
	/* Select the frame mode to fit your needs. */
	config.frameMode = BTA_FrameModeXYZAmp;
	/* Register a callback function for informative events. */
	config.infoEvent = &infoEvent;
	/* Set the verbosity for informative events. */
	config.verbosity = 10;

	/* If you want to receive the frames immediately when they arrive over UDP, */
	/* register a callback function for incoming frames. */
	config.frameArrived = &frameArrived;

	/* Choose whether and how queueing of frames is done. */
	/* Queueing does not affect the frameArrived callback at all */
	/* If you don't specify frame queuing, you can still use the frameArrived callback, */
	/* but you will get an error on BTAgetFrame() */
	config.frameQueueMode = BTA_QueueModeDropOldest;
	/* Set the length of the frame queue. */
	config.frameQueueLength = 1;
	/*---------------------------------------------------------------- */

	BTA_Handle btaHandle;
	printf("BTAopen()\n");
	int openRetries = 10;
	while (openRetries) {
		openRetries--;
		printf("BTAopen()\n");
		status = BTAopen(&config, &btaHandle);
		if (status == BTA_StatusOk) {
			break;
		} else {
			BTAstatusToString(status, statusString, 50);
			printf("Error: %s\n", statusString);
			printf("Retrying open for another %d times...\n", openRetries);
		}
	}
	if (openRetries < 0) {
		fprintf(stderr, "Error: Giving up on open. Quitting.\n");
		return -1;
	}

	BTA_DeviceInfo *deviceInfo;
	printf("BTAgetDeviceInfo()\n");
	status = BTAgetDeviceInfo(btaHandle, &deviceInfo);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		fprintf(stderr, "Error: %s\n", statusString);
		return -1;
	}
	uint16_t deviceType = deviceInfo->deviceType;
	printf("  Device type: 0x%x\n", deviceType);
	printf("BTAfreeDeviceInfo()\n");
	BTAfreeDeviceInfo(deviceInfo);
	struct bltTofDevice *connectedDevice = NULL;
	for (int i=0; i<sizeof(bltTofDevices)/sizeof(bltTofDevices[0]); i++) {
		if (bltTofDevices[i].deviceType == deviceType) {
			connectedDevice = &bltTofDevices[i];
		}
	}
	if (!connectedDevice) {
		fprintf(stderr, "Error: Unknown device type!\n");
		return -1;
	}

	/* GPU init */
	if (connectedDevice->gpuAvail) {
		if (gpuInit() < 0) {
			fprintf(stderr, "Failed to initialize the GPU\n");
			syslog(LOG_ERR, "Failed to initialize the GPU\n");
			return -1;
		}
	}
    
	printf("Service running: %s\n", BTAisRunning(btaHandle) ? "true" : "false");
	printf("Connection up: %s\n", BTAisConnected(btaHandle) ? "true" : "false");
    
	uint32_t regValue;
	printf("BTAreadRegister()\n");
	status = BTAreadRegister(btaHandle, 5, &regValue, 0);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		printf("error: %s\n", statusString);
		return -1;
	}
    
	printf("BTAwriteRegister()\n");
	status = BTAwriteRegister(btaHandle, 5, &regValue, 0);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		printf("error: %s\n", statusString);
		return -1;
	}

	printf("Configuring camera to use loopback interface\n");
	uint32_t camLocalConfig[2][2] = { /* Address */ /* Value */
		{0x024c, 1}, /* UDP Stream IP: 127.0.0.1 */
		{0x024d, (127<<8)},
	};
	for (int i=0; i<2; i++) {
		status = BTAwriteRegister(btaHandle, camLocalConfig[i][0], &camLocalConfig[i][1], 0);
		if (status != BTA_StatusOk) {
			BTAstatusToString(status, statusString, 50);
			printf("error: %s\n", statusString);
			return -1;
		}
	}
	if (connectedDevice->supports2d) {
		uint32_t camLocalConfig2d[2][2] = { /* Address */ /* Value */
			{0x0252, 1}, /* 2D RTP Stream IP: 127.0.0.1 */
			{0x0253, (127<<8)},
		};
		for (int i=0; i<2; i++) {
			status = BTAwriteRegister(btaHandle, camLocalConfig2d[i][0], &camLocalConfig2d[i][1], 0);
			if (status != BTA_StatusOk) {
				BTAstatusToString(status, statusString, 50);
				printf("error: %s\n", statusString);
				return -1;
			}
		}
	}

	syslog(LOG_INFO, "Application initialized and camera configured\n");

	/* Actively get a frame */
	BTA_Frame *frame;
	printf("BTAgetFrame()\n");
	status = BTAgetFrame(btaHandle, &frame, 3000);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		printf("error: %s\n", statusString);
		return -1;
	}
    
	uint16_t *amplitudes;
	BTA_DataFormat dataFormat;
	BTA_Unit unit;
	uint16_t xRes, yRes;
	printf("BTAgetAmplitudes()\n");
	status = BTAgetAmplitudes(frame, (void **)&amplitudes, &dataFormat, &unit, &xRes, &yRes);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		printf("error: %s\n", statusString);
		return -1;
	}
	if (dataFormat == BTA_DataFormatUInt16) {
		if (unit == BTA_UnitUnitLess) {
			printf("Got amplitude data, first line follows...\n");
			/* as expected -> process amplitude data: amplitudes[i] */
			for (int y = 0; y < 1; y++) {
				for (int x = 0; x < xRes; x++) {
					printf("%u, ", amplitudes[x + y*xRes]);
				}
			}
			printf("\n\n");
		}
	}

	uint16_t ampMask[xRes * yRes];
	if (connectedDevice->gpuAvail) {
		/* Here we demonstrate the usage of OpenCL 1.1 on the i.MX6Q GPU
		   The demo program reads the amplitude array and calculates a mask:
		   If the amplitude is greater than the threshold (300),
		   the value is set to 0xFFFF. 0x0 otherwise.
		*/
		if (gpuCreateAmplitudeMask(amplitudes, xRes, yRes, 300 /*threshold*/, ampMask) < 0) {
			fprintf(stderr, "Error on GPU calculation\n");
			syslog(LOG_ERR, "Error on GPU calculation\n");
		} else {
			printf("Calculated mask from amplitude data using GPU, first line follows...\n");
			for (int y = 0; y < 1; y++) {
				for (int x = 0; x < xRes; x++) {
					printf("%u, ", ampMask[x + y*xRes]);
				}
			}
			printf("\n\n");
		}
	}

	/* Here we demonstrate the usage of the NEON instruction set of the i.MX6Q CPUs
	   The demo program creates the same mask as above
	*/
	if (neonCreateAmplitudeMask(amplitudes, xRes, yRes, 300 /* threshold*/, ampMask) < 0) {
		fprintf(stderr, "Error on NEON calculation\n");
		syslog(LOG_ERR, "Error on NEON calculation\n");
	} else {
		printf("Calculated mask from amplitude data using NEON, first line follows...\n");
		for (int y = 0; y < 1; y++) {
			for (int x = 0; x < xRes; x++) {
				printf("%u, ", ampMask[x + y*xRes]);
			}
		}
		printf("\n\n");
	}

	void *xCoordinates, *yCoordinates, *zCoordinates;
	printf("BTAgetXYZcoordinates()\n");
	status = BTAgetXYZcoordinates(frame, &xCoordinates, &yCoordinates, &zCoordinates, &dataFormat, &unit, &xRes, &yRes);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		printf("error: %s\n", statusString);
		return -1;
	}
	if (dataFormat == BTA_DataFormatSInt16) {
		/* This dataformat tells us that it's ok to cast (void *) to (int16_t *) */
		if (unit == BTA_UnitMillimeter) {
			printf("Got 3D data, first line follows...\n");
			/* as expected -> process point in space: ( xCoordinates[i], yCoordinates[i], zCoordinates[i] ) */
			for (int y = 0; y < 1; y++) {
				for (int x = 0; x < xRes; x++) {
					printf("(%d, %d, %d), ", ((int16_t *)xCoordinates)[x + y*xRes], ((int16_t *)yCoordinates)[x + y*xRes], ((int16_t *)zCoordinates)[x + y*xRes]);
				}
			}
			printf("\n\n");
		}
	}

	printf("BTAfreeFrame()\n");
	status = BTAfreeFrame(&frame);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		printf("error: %s\n", statusString);
		return -1;
	}

	/* During this time, the frameArrived() callback will be called regularly */
	printf("\nSleeping for 1 second\n");
	sleep(1);

	printf("\nBTAclose()\n");
	status = BTAclose(&btaHandle);
	if (status != BTA_StatusOk) {
		BTAstatusToString(status, statusString, 50);
		printf("error: %s\n", statusString);
		return -1;
	}

	/* Release GPU resources */
	gpuExit();

	syslog(LOG_INFO, "Quitting\n");
	printf("\n\n*** bta_eth_example is finished ***\n\n\n");

	return 0;
}
