/**
*   @file           buffer_management.c
*   @ingroup 
*   
*   @brief 
*   
*   BLT_DISCLAIMER
*   
*   @author     Alexander Froemel
*   @version    1.0.0.0
*   @date       28.03.2013
*   
*   @cond svn
*   
*   Information of last commit
*   $Rev::               $:  Revision of last commit
*   $Author::            $:  Author of last commit
*   $Date::              $:  Date of last commit
*   
*   @endcond
**/

/**
 * Copyright (c) 2013 Bluetechnix GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE
 **/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "buffer_management.h"
#ifdef _USE_KERNEL_
    #include <sched.h>
#endif //



void BMTsetup(void) {
    // currently nothing to do
}

void BMTcleanup(void) {
    // currently nothing to do
}

T_ERROR_CODE BMTinitConfigMngr(T_BMT_MNGR_CONFIG *pa_ptConf) {
    T_ERROR_CODE tErr = ERR_NONE;
    if(pa_ptConf) {
        memset(pa_ptConf, 0, sizeof(T_BMT_MNGR_CONFIG));
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTinitConfigClient(T_BMT_CLIENT_CONFIG *pa_ptConf) {
    T_ERROR_CODE tErr = ERR_NONE;
    if(pa_ptConf) {
        memset(pa_ptConf, 0, sizeof(T_BMT_CLIENT_CONFIG));
        //pa_ptConf->ucMDMAchannel = MDMAM_CH_UNSPECIFIED;
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

static void BMTlock(volatile bool *pa_pbLocked) {
    while(*pa_pbLocked) {
#ifdef _USE_KERNEL_
        sched_yield();
#endif //_USE_KERNEL_
    }
    *pa_pbLocked = true;
}

static void BMTunlock(volatile bool *pa_pbLocked) {
    *pa_pbLocked = false;
}

T_BMT_MNGR_HANDLE BMTopen(T_BMT_MNGR_CONFIG *pa_ptConfig, T_ERROR_CODE *pa_ptErr) {
    *pa_ptErr = ERR_NONE;
    T_BMT_MNGR_HANDLE tHndl = 0;
    T_BMT_MNGR_INST *pInst = (T_BMT_MNGR_INST *)calloc(sizeof(T_BMT_MNGR_INST), 1);
    
    if(pInst){
        pInst->ulMaxNofClients = pa_ptConfig->ulMaxNofClients;
        pInst->ptClients = (T_BMT_CLIENT_HANDLE *)calloc(sizeof(T_BMT_CLIENT_HANDLE) * pInst->ulMaxNofClients, 1);
        if(pInst->ptClients) {
            tHndl = (T_BMT_MNGR_HANDLE)pInst;
        } else {
            *pa_ptErr = ERR_BMT_NOT_ENOUGH_MEMORY;
            free(pInst);
        }
    } else {
        *pa_ptErr = ERR_BMT_NOT_ENOUGH_MEMORY;
    }
    return tHndl;
}

T_BMT_CLIENT_HANDLE BMTaddClient(T_BMT_MNGR_HANDLE pa_tMngrHndl, T_BMT_CLIENT_CONFIG *pa_ptConfig, T_ERROR_CODE *pa_ptErr) {
    *pa_ptErr = ERR_NONE;
    T_BMT_CLIENT_HANDLE tHndl = 0;
    if(pa_tMngrHndl) {
        T_BMT_MNGR_INST *pMngrInst = (T_BMT_MNGR_INST *)pa_tMngrHndl;
        if(pMngrInst->ulNofClients < pMngrInst->ulMaxNofClients) {
            T_BMT_CLIENT_INST *pInst = (T_BMT_CLIENT_INST *)calloc(sizeof(T_BMT_CLIENT_INST), 1);
            int i=0;
            if(pInst) {
                do {
                    pInst->bBufferProvidedExtern = pa_ptConfig->bBufferProvidedExtern;
                    //pInst->ucMDMAchannel = pa_ptConfig->ucMDMAchannel;
                    //pInst->bDMAcopy = pa_ptConfig->bDMAcopy;
                    pInst->ulNofBuffers = pa_ptConfig->ulNofBuffers;
                    pInst->ulSizePerBuffer = pa_ptConfig->ulSizePerBuffer;
                    pInst->ulCopyOffsetInBuffer = pa_ptConfig->ulCopyOffsetInBuffer;
                    pInst->tBufferList = (T_BMT_BUFFER_DESC *)calloc(sizeof(T_BMT_BUFFER_DESC) * pInst->ulNofBuffers, 1);
                    if(!pInst->tBufferList) {
                        *pa_ptErr = ERR_BMT_NOT_ENOUGH_MEMORY;
                        break;
                    }
                    if(pInst->bBufferProvidedExtern) {
                        for(i=0; i<pInst->ulNofBuffers; i++) {
                            pInst->tBufferList[i].pucBuf = pa_ptConfig->tBufferList[i].pucBuf;
                            pInst->tBufferList[i].ulCopyOffsetInBuffer = pInst->ulCopyOffsetInBuffer;
                        }
                    } else {
                        unsigned char *pucTemp = (unsigned char *)malloc(pInst->ulSizePerBuffer * pInst->ulNofBuffers);
                        if(pucTemp) {
                            for(i=0; i<pInst->ulNofBuffers; i++) {
                                pInst->tBufferList[i].pucBuf = &pucTemp[i * pInst->ulNofBuffers];
                                pInst->tBufferList[i].ulCopyOffsetInBuffer = pInst->ulCopyOffsetInBuffer;
                            }
                        } else {
                            *pa_ptErr = ERR_BMT_NOT_ENOUGH_MEMORY;
                            free(pInst->tBufferList);
                            break;
                        }
                    }
                    pInst->ptMngr = (T_BMT_MNGR_HANDLE)pMngrInst;
                    // add client to manager's client list
                    for(i=0; i<pMngrInst->ulMaxNofClients; i++) {
                        if(!(pMngrInst->ptClients[i])) {
                            pMngrInst->ptClients[i] = (T_BMT_CLIENT_HANDLE)pInst;
                            pMngrInst->ulNofClients++;
                            break;
                        }
                    }
                } while(0);
                
                if(*pa_ptErr != ERR_NONE) {
                    free(pInst);
                } else {
                    tHndl = (T_BMT_CLIENT_HANDLE)pInst;
                }
            }
        } else {
            *pa_ptErr = ERR_BMT_TOO_MANY_CLIENTS;
        }
    } else {
        *pa_ptErr = ERR_BMT_INVALID_HANDLE;
    }
    return tHndl;
}

T_ERROR_CODE BMTclose(T_BMT_MNGR_HANDLE pa_tHndl) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_MNGR_INST *pInst = (T_BMT_MNGR_INST *) pa_tHndl;
    if(pInst) {
        BMTlock(&pInst->bLocked);
        free(pInst);
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTremoveClient(T_BMT_CLIENT_HANDLE pa_tHndl) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_CLIENT_INST *pInst = (T_BMT_CLIENT_INST *) pa_tHndl;
    if(pInst) {
        T_BMT_MNGR_INST *pMngrInst = (T_BMT_MNGR_INST *)pInst->ptMngr;
        
        BMTlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
        
        int i=0;
        for(i=0; i<pMngrInst->ulMaxNofClients; i++) {
            if(pMngrInst->ptClients[i] == pa_tHndl) {
                pMngrInst->ptClients[i] = 0;
                break;
            }
        }
        if(!pInst->bBufferProvidedExtern) {
            free(pInst->tBufferList[0].pucBuf);
        }
        free(pInst->tBufferList);
        free(pInst);
        
        BMTunlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTgetNextEnqueueBuffer(T_BMT_MNGR_HANDLE pa_tHndl, T_BMT_BUFFER_DESC **pa_ptBuffer) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_MNGR_INST *pInst = (T_BMT_MNGR_INST *) pa_tHndl;
    int i=0;
    if(pInst) {
        BMTlock(&(pInst->bLocked));
        
        for(i=0; i<pInst->ulMaxNofClients; i++) {
            T_BMT_CLIENT_INST *pCurrClient = (T_BMT_CLIENT_INST *)pInst->ptClients[i];
            if((pInst->ptClients[i] != 0) && (pCurrClient->ulFillSize < pCurrClient->ulNofBuffers)) {
                *pa_ptBuffer = &pCurrClient->tBufferList[pCurrClient->ulCurrIn];
                pInst->tLastRequestedEnqueueBufferClient = (T_BMT_CLIENT_HANDLE)pCurrClient;
                break;
            }
        }
        if(i >= pInst->ulMaxNofClients) {
            tErr = ERR_BMT_BUFFER_FULL;
        }
        BMTunlock(&(pInst->bLocked));
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTbufferProcessed(T_BMT_MNGR_HANDLE pa_tHndl) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_MNGR_INST *pInst = (T_BMT_MNGR_INST *) pa_tHndl;
    if(pInst) {
        BMTlock(&(pInst->bLocked));
        T_BMT_CLIENT_INST *pClient = (T_BMT_CLIENT_INST *)pInst->tLastRequestedEnqueueBufferClient;
        if(pClient->ulFillSize < pClient->ulNofBuffers) {
            unsigned char *pucSrc = &pClient->tBufferList[pClient->ulCurrIn].pucBuf[pClient->ulCopyOffsetInBuffer];
            unsigned char *pucDst = NULL;
            unsigned long ulBytesToCopy = 0;
            int i=0;
            for(i=0; i<pInst->ulMaxNofClients; i++) {
                T_BMT_CLIENT_INST *pCurrClient = (T_BMT_CLIENT_INST *)pInst->ptClients[i];
                if((pCurrClient != NULL)  
                        && (pCurrClient != pClient) 
                        && (pCurrClient->ulFillSize < pCurrClient->ulNofBuffers)) {
                    pucDst = pCurrClient->tBufferList[pCurrClient->ulCurrIn].pucBuf;
                    if(pCurrClient->ulSizePerBuffer > pClient->ulSizePerBuffer) {
                        ulBytesToCopy = pClient->tBufferList[pClient->ulCurrIn].ulCurrSize;
                    } else {
                        ulBytesToCopy = pCurrClient->tBufferList[pCurrClient->ulCurrIn].ulCurrSize;
                    }
                    memcpy(&pucDst[pCurrClient->ulCopyOffsetInBuffer], pucSrc, ulBytesToCopy);
                    
                    pCurrClient->tBufferList[pCurrClient->ulCurrIn].ulCurrSize = pClient->tBufferList[pClient->ulCurrIn].ulCurrSize;
                    pCurrClient->tBufferList[pCurrClient->ulCurrIn].pArg = pClient->tBufferList[pClient->ulCurrIn].pArg;
                    pCurrClient->ulCurrIn++;
                    pCurrClient->ulFillSize++;
                    if(pCurrClient->ulCurrIn >= pCurrClient->ulNofBuffers) {
                        pCurrClient->ulCurrIn = 0;
                    }
                }
            }

            pClient->ulCurrIn++;
            pClient->ulFillSize++;
            if(pClient->ulCurrIn >= pClient->ulNofBuffers) {
                pClient->ulCurrIn = 0;
            }
        } else {
            tErr = ERR_BMT_BUFFER_FULL;
        }
        BMTunlock(&(pInst->bLocked));
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTenqueue(T_BMT_MNGR_HANDLE pa_tHndl, unsigned char *pa_pucBuffer, unsigned long pa_ulBufferLength, void *pa_pArg, unsigned long pa_ulFlags) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_MNGR_INST *pInst = (T_BMT_MNGR_INST *) pa_tHndl;
    if (pInst) {
        BMTunlock(&pInst->bLocked);
        unsigned char *pucDst = NULL;
        int i=0;
        unsigned long ulBytesToCopy = 0;
        for(i=0; i<pInst->ulMaxNofClients; i++) {
            T_BMT_CLIENT_INST *pCurrClient = (T_BMT_CLIENT_INST *)pInst->ptClients[i];
            if((pCurrClient != NULL) && (pCurrClient->ulFillSize < pCurrClient->ulNofBuffers)) {
                pucDst = pCurrClient->tBufferList[pCurrClient->ulCurrIn].pucBuf;
                if(pCurrClient->ulSizePerBuffer > pa_ulBufferLength) {
                    ulBytesToCopy = pa_ulBufferLength;
                } else {
                    ulBytesToCopy = pCurrClient->ulSizePerBuffer;
                }
                memcpy(&pucDst[pCurrClient->ulCopyOffsetInBuffer], pa_pucBuffer, ulBytesToCopy);
                pCurrClient->tBufferList[pCurrClient->ulCurrIn].ulCurrSize = ulBytesToCopy;
                pCurrClient->tBufferList[pCurrClient->ulCurrIn].pArg = pa_pArg;
                pCurrClient->ulCurrIn++;
                pCurrClient->ulFillSize++;
                if(pCurrClient->ulCurrIn >= pCurrClient->ulNofBuffers) {
                    pCurrClient->ulCurrIn = 0;
                }
            }
        }
        BMTunlock(&pInst->bLocked);
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTdequeue(T_BMT_CLIENT_HANDLE pa_tHndl, T_BMT_BUFFER_DESC **pa_ptBuffer) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_CLIENT_INST *pInst = (T_BMT_CLIENT_INST *) pa_tHndl;
    if(pInst) {
        BMTlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
        if(pInst->ulFillSize > 0) {
            *pa_ptBuffer = &pInst->tBufferList[pInst->ulCurrOut];
        } else {
            tErr = ERR_BMT_BUFFER_EMPTY;
        }
        BMTunlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTdequeueProcessed(T_BMT_CLIENT_HANDLE pa_tHndl) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_CLIENT_INST *pInst = (T_BMT_CLIENT_INST *) pa_tHndl;
    if(pInst) {
        BMTlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
        if(pInst->ulFillSize > 0) {
            pInst->ulCurrOut++;
            if(pInst->ulCurrOut >= pInst->ulNofBuffers) {
                pInst->ulCurrOut = 0;
            }
            pInst->ulFillSize--;
        } else {
            tErr = ERR_BMT_BUFFER_EMPTY;
        }
        BMTunlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTdequeueProcessedFromInterrupt(T_BMT_CLIENT_HANDLE pa_tHndl) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_CLIENT_INST *pInst = (T_BMT_CLIENT_INST *) pa_tHndl;
    if(pInst) {
        //BMTlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
        if(pInst->ulFillSize > 0) {
            pInst->ulCurrOut++;
            if(pInst->ulCurrOut >= pInst->ulNofBuffers) {
                pInst->ulCurrOut = 0;
            }
            pInst->ulFillSize--;
        } else {
            tErr = ERR_BMT_BUFFER_EMPTY;
        }
        //BMTunlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTflushClient(T_BMT_CLIENT_HANDLE pa_tHndl) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_CLIENT_INST *pInst = (T_BMT_CLIENT_INST *) pa_tHndl;
    if (pInst) {
        BMTlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
        pInst->ulFillSize = 0;
        pInst->ulCurrOut = 0;
        pInst->ulCurrIn = 0;
        BMTunlock(&((T_BMT_MNGR_INST *)pInst->ptMngr)->bLocked);
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}

T_ERROR_CODE BMTflushAll(T_BMT_MNGR_HANDLE pa_tHndl) {
    T_ERROR_CODE tErr = ERR_NONE;
    T_BMT_MNGR_INST *pInst = (T_BMT_MNGR_INST *) pa_tHndl;
    int i=0;
    if(pInst) {
        BMTlock(&(pInst->bLocked));
        for(i=0; i<pInst->ulMaxNofClients; i++) {
            if(pInst->ptClients[i] != 0) {
                T_BMT_CLIENT_INST *pClientInst = (T_BMT_CLIENT_INST *) pInst->ptClients[i];
                pClientInst->ulFillSize = 0;
                pClientInst->ulCurrOut = 0;
                pClientInst->ulCurrIn = 0;
            }
        }
        BMTunlock(&(pInst->bLocked));
    } else {
        tErr = ERR_BMT_INVALID_HANDLE;
    }
    return tErr;
}
