/**  @file eth_udp_streaming_thread.c
*    @ingroup services
*  
*    @brief Simple demo example which shows how distance and amplitude data can be requested from core b, the image data can be 
*           streamed over ehternet, the sensor registers can be written or read and gpios can be used.
*         
*  
*    BLT_DISCLAIMER
*  
*    @author Benedikt Radl
*  
*    @cond svn
*  
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*  
*    @endcond
*/

/**
 * Copyright (c) 2013 Bluetechnix GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE
 **/

#include <stdio.h>
#include <sys/socket.h> 
#include <arpa/inet.h>	
#include "buffer_management.h"
#include "tof_configuration_interface.h"
#include "crc/crc32.h"
#include "crc/crc16.h"
#include "eth_udp_streaming_thread.h"
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#define CMD_GPIO_READ         _IO('b', 0)
#define CMD_GPIO_WRITE          _IO('b', 1)

/**
 *    @brief The thread's run function. it loops and handles communication
 *    @param tSTM100udpBMTClient The buffer client handle address
 *    @param tTCIHandle The register handle address         
**/
void ETHudpStreamingThreadRunFunction(T_BMT_CLIENT_HANDLE tSTM100udpBMTClient, T_TCI_HANDLE tTCIHandle) {
    T_BMT_BUFFER_DESC *ptBuffer = NULL;
    T_ERROR_CODE tErr = ERR_NONE;
    int nSocket = -1;
    int nBytesWritten = 0;
    int nResult = 0;
    int nBytesToWrite = 0;
    int nBytesWrittenTotal = 0;
    int nBytesToWriteTotal = 0;
    int nCurrentPacketSize = 0;
    int nPacketCounter = 0;
    int nFrameCounter = 0;
    char acIPaddr[20];
    int nTimeout = ETH_UDP_STREAMING_TRANSMIT_TIMEOUT;
    unsigned char *pucCurBuf = NULL;
    unsigned long ulCRC = 0;
    struct sockaddr_in tClient;
    bool bFlushBuffer = true;
    unsigned long ulFlags = 0;
    unsigned short usLength = 2;
    unsigned short usTempVal0 = 0;
    unsigned short usTempVal1 = 0;
    unsigned short usTempVal2 = 0;
    unsigned short usCount = 0;
    unsigned long ulFrameCount = 0; 

    memset(&tClient, 0, sizeof(struct sockaddr_in));

    //create a socket
    while (nSocket == -1) {
        nSocket = socket(AF_INET, SOCK_DGRAM, 0); 
        if (nSocket == -1) {
            usleep(10000);
        }
    }
    /*** register access ***/
    //set framereate
    usTempVal0 = 25;
    TCIregisterWrite(tTCIHandle, T_TCI_REGISTERS_FRAMERATE, &usTempVal0, usLength);
    //set integration time
    usTempVal0 = 1000;
    TCIregisterWrite(tTCIHandle, T_TCI_REGISTERS_INTEGRATION_TIME, &usTempVal0, usLength);
    //read default ip address from the sensor registers
    TCIregisterRead(tTCIHandle, T_TCI_REGISTERS_ETH0_UDP_STREAM_IP_0, &usTempVal0, &usLength);
    //printf("TCIregisterRead: addr: 0x%4x, value:0x%04x\n",T_TCI_REGISTERS_ETH0_UDP_STREAM_IP_0, usTempVal0);
    TCIregisterRead(tTCIHandle, T_TCI_REGISTERS_ETH0_UDP_STREAM_IP_1, &usTempVal1, &usLength);
    //printf("TCIregisterRead: addr: 0x%4x, value:0x%04x\n",T_TCI_REGISTERS_ETH0_UDP_STREAM_IP_1, usTempVal1);
    sprintf(acIPaddr, "%d.%d.%d.%d", (usTempVal1 >> 8), (usTempVal1) & 0xff, (usTempVal0 >> 8), (usTempVal0) & 0xff);
    TCIregisterRead(tTCIHandle, T_TCI_REGISTERS_ETH0_UDP_STREAM_PORT, &usTempVal2, &usLength);
    //printf("TCIregisterRead: addr: 0x%4x, value:%u\n",T_TCI_REGISTERS_ETH0_UDP_STREAM_PORT, usTempVal2);
    //set socket address
    tClient.sin_family = AF_INET;
    tClient.sin_port = htons(usTempVal2);
    tClient.sin_addr.s_addr = inet_addr(acIPaddr);
    //printf ("addr: %08x\n", tClient.sin_addr.s_addr);

    while (1) {
        //check Enable UDP streaming bit
        TCIregisterRead(tTCIHandle, T_TCI_REGISTERS_ETH0_CONFIG, &usTempVal0, &usLength);
        while (!(usTempVal0 & 0x2)) {
            usleep(100000);
            bFlushBuffer = true;
            TCIregisterRead(tTCIHandle, T_TCI_REGISTERS_ETH0_CONFIG, &usTempVal0, &usLength);
        }   
    
        //check Ignore CRC for UDP streaming bit  
        if (usTempVal0 & 0x4) {
            ulFlags |= ETH_UDP_STREAMING_HEADER_FLAGS_IGNORE_CRC;
        } else {
            ulFlags &= ~ETH_UDP_STREAMING_HEADER_FLAGS_IGNORE_CRC;
        }

        if (bFlushBuffer) {
            bFlushBuffer = false;
            //remove all items in the queue
            BMTflushClient(tSTM100udpBMTClient);
        }
        //request data from the queue
        tErr = BMTdequeue(tSTM100udpBMTClient, &ptBuffer);

        if (tErr == ERR_NONE) {
            nBytesWrittenTotal = 0;
            nBytesToWriteTotal = ptBuffer->ulCurrSize;
            pucCurBuf = ptBuffer->pucBuf;
            nPacketCounter = 0;
            nFrameCounter = (unsigned short)((pucCurBuf[ETH_UDP_STREAMING_HEADER_SIZE + 16] << 8) | pucCurBuf[ETH_UDP_STREAMING_HEADER_SIZE + 17]);

            nTimeout = ETH_UDP_STREAMING_TRANSMIT_TIMEOUT;
            while (nTimeout && (nBytesWrittenTotal < nBytesToWriteTotal)) {
                if ((nBytesToWriteTotal - nBytesWrittenTotal) >= ETH_UDP_STREAMING_MAX_PACKET_LENGTH) {
                    nCurrentPacketSize = ETH_UDP_STREAMING_MAX_PACKET_LENGTH;
                } else {
                    nCurrentPacketSize = nBytesToWriteTotal - nBytesWrittenTotal;
                }
                nBytesWritten = 0;
                nBytesToWrite = nCurrentPacketSize;
                //build ethernet upd streaming header
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_HEADER_VERSION]     = (unsigned char)(ETH_UDP_STREAMING_HEADER_VERSION >> 8);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_HEADER_VERSION + 1] = (unsigned char)ETH_UDP_STREAMING_HEADER_VERSION;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_FRAME_COUNTER]     = (unsigned char)(nFrameCounter >> 8);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_FRAME_COUNTER + 1] = (unsigned char)nFrameCounter;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_PACKET_COUNTER]     = (unsigned char)(nPacketCounter >> 8);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_PACKET_COUNTER + 1] = (unsigned char)nPacketCounter;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_PACKET_SIZE]     = (unsigned char)(nCurrentPacketSize >> 8);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_PACKET_SIZE + 1] = (unsigned char)nCurrentPacketSize;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_TOTAL_LENGTH]     = (unsigned char)(nBytesToWriteTotal >> 24);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_TOTAL_LENGTH + 1] = (unsigned char)(nBytesToWriteTotal >> 16);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_TOTAL_LENGTH + 2] = (unsigned char)(nBytesToWriteTotal >> 8);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_TOTAL_LENGTH + 3] = (unsigned char)nBytesToWriteTotal;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32]     = 0;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32 + 1] = 0;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32 + 2] = 0;
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32 + 3] = 0;
                
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_FLAGS]     = (unsigned char)(ulFlags >> 24);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_FLAGS + 1] = (unsigned char)(ulFlags >> 16);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_FLAGS + 2] = (unsigned char)(ulFlags >> 8);
                pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_FLAGS + 3] = (unsigned char)ulFlags;
                
                memset(&pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_RESERVED], 0, ETH_UDP_STREAMING_HEADER_RESERVED_LENGTH);
                
                if (!(ulFlags & ETH_UDP_STREAMING_HEADER_FLAGS_IGNORE_CRC)) {
                    //calculate crc32 checksum
                    ulCRC = CRC32ccitt(pucCurBuf, nCurrentPacketSize + ETH_UDP_STREAMING_HEADER_SIZE);
                    pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32]     = (unsigned char)(ulCRC >> 24);
                    pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32 + 1] = (unsigned char)(ulCRC >> 16);
                    pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32 + 2] = (unsigned char)(ulCRC >> 8);
                    pucCurBuf[ETH_UDP_STREAMING_HEADER_INDEX_CRC32 + 3] = (unsigned char)ulCRC;
                }
           
                nTimeout = ETH_UDP_STREAMING_TRANSMIT_TIMEOUT;
				//send the frame
                while (nTimeout && (nBytesWritten < nBytesToWrite)) {					
                    nResult = sendto(nSocket, (char *)pucCurBuf, nBytesToWrite + ETH_UDP_STREAMING_HEADER_SIZE, 0, (struct sockaddr *)&tClient, sizeof(tClient));
                    if (nResult > 0) {
                        nBytesWritten = nResult - ETH_UDP_STREAMING_HEADER_SIZE;
                    } else {
                        printf ("%s\n",strerror(errno));			
                        usleep(1000);
                        nTimeout--;
                    }
                }
                pucCurBuf += nBytesWritten;
                nPacketCounter ++;
                nBytesWrittenTotal += (nBytesWritten);
            }
            //remove the last item in the queue
            tErr = BMTdequeueProcessed(tSTM100udpBMTClient);

            usCount++;
            ulFrameCount++;
            if(usCount == 20) {
                //printf("captured frames: %lu \n",ulFrameCount);
                usCount = 0;		
            }
        } else {
            usleep(1000);
        }

        #ifdef _USE_THREAD_WATCHDOG_
          TWDreset(tTWDhndl);
        #endif

    }
}


/**
 *      @brief Starts the udp streaming over ehternent
**/
int main (int argc, char *argv[])
{
    unsigned short *pusBuf = (unsigned short *) SHARED_BLOCK_IDENTIFIER_ADDRESS;
    unsigned short usIdentifier = pusBuf[0];
    T_BMT_CLIENT_HANDLE tSTM100udpBMTClient = pusBuf[1] + (pusBuf[2] << 16);
    T_TCI_HANDLE tTCIHandle = pusBuf[3] + (pusBuf[4] << 16);
    unsigned short usCRC = pusBuf[(SHARED_BLOCK_LENGTH >> 1) - 1];
    int fd, value;

    /*** I/O control ***/
    fd = open("/dev/gpio-control", O_RDWR);

    // set GPIO 4
    /* ioctl(file pointer, read or write , pin number, value (0 or 1))*/
    ioctl(fd, CMD_GPIO_WRITE, 4, 1); 
    // clear GPIO 3
    ioctl(fd, CMD_GPIO_WRITE, 3, 0);
    // read GPIO 1
    value = ioctl(fd, CMD_GPIO_READ, 1);

    close(fd);

    printf("wait for coreb\n");
    //wait for core b
    while(usIdentifier != SHARED_BLOCK_IDENTIFIER);

    printf("coreb is ready\n");
    //check crc
    if(usCRC == crc16_ccitt((const unsigned short *) BUFFER_CLIENT_HANDLE_ADDRESS, SHARED_BLOCK_LENGTH - 4)) {
        //start udp streaming
        printf("start streaming\n");
        ETHudpStreamingThreadRunFunction(tSTM100udpBMTClient, tTCIHandle);
    } else {
        //crc check failed
        printf("crc check failed\n");
    }

    return 0;
}

