%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% file: exampleETH.m                                                      %
%                                                                         %
% brief: This is an example for getting a distance and amplitude stream.  %
%                                                                         %
% autor: Benedikt Radl                                                    %
%                                                                         %
% Copyright (c) 2014 Bluetechnix GmbH                                     %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

close all;
clear all;

% get API version
[status, version, buildDateTime, supportedDeviceTypes] = BTAgetVersion;
if status ~= 0
    error('BTAgetVersion failed. See "status" for error description');
end

% get a config structure
[status, configStruct] = BTAinitConfig;
if status ~= 0
    error('BTAinitConfig failed. See "status" for error description');
end

% fill the config struct
configStruct.udpDataIpAddr = [224, 0, 0, 1];
configStruct.udpDataIpAddrLen = 4;
configStruct.udpDataPort = 10002;
configStruct.tcpDeviceIpAddr = [192, 168, 0, 10];
configStruct.tcpDeviceIpAddrLen = 4;
configStruct.tcpControlPort = 10001;
configStruct.frameMode = 1;
configStruct.frameQueueMode = 1;
configStruct.frameQueueLength = 5;

% open device
[status, deviceHandle] = BTAopen(configStruct);
if status ~= 0
    error('BTAopen failed. See "status" for error description');
end

% set the interval of the keep alive massages 
status = BTAsetKeepAliveMsgInterval(deviceHandle, 2);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAsetKeepAliveMsgInterval failed. See "status" for error description');
end

% get integration time 
[status, integrationTime] = BTAgetIntegrationTime(deviceHandle);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAgetIntegrationTime failed. See "status" for error description');
end

% you can also read the integration time by using BTAreadRegister
[status, data] = BTAreadRegister(deviceHandle, 5);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAreadRegister failed. See "status" for error description');
end

% get frame rate
[status, frameRate] = BTAgetFrameRate(deviceHandle);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAgetFrameRate failed. See "status" for error description');
end

% set framerate to 5 (for limiting the workload)
status = BTAsetFrameRate(deviceHandle, 5);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAsetFrameRate failed. See "status" for error description');
end

% set framemode to DistAmp
status = BTAsetFrameMode(deviceHandle, 1);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAsetFrameMode failed. See "status" for error description');
end
%wait some time that the changes take effect
pause(2); 

% get one frame
[status, frameHandle, frameCounter, timeStamp] = BTAgetFrame(deviceHandle, 500);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAgetFrame failed. See "status" for error description');
end

% get distance data
[status, distData, integrationTime, modulationFrequency, unit] = BTAgetDistances(frameHandle);
if status ~= 0
    statusFree = BTAfreeFrame(frameHandle);
    statusClose = BTAclose(deviceHandle);
    error('BTAgetDistances failed. See "status" for error description');
end

% get amplitude data
[status, ampData, integrationTime2, modulationFrequency2, unit2] = BTAgetAmplitudes(frameHandle);
if status ~= 0
    statusFree = BTAfreeFrame(frameHandle);
    statusClose = BTAclose(deviceHandle);
    error('BTAgetAmplitudes failed. See "status" for error description');
end

% free frame
[status] = BTAfreeFrame(frameHandle);
if status ~= 0
    statusClose = BTAclose(deviceHandle);
    error('BTAfreeFrame failed. See "status" for error description');
end

%create window for distances
figure('name','Distance data');
set(gcf,'NumberTitle','off')
%set color range from 0 to 3000
clims  = [0 3000];
hDist = imagesc(distData, clims);
colorbar;
cm=flipud(jet);
colormap(cm);

global MK;
MK=0;
set(gcf,'KeyPressFcn','MK=1;');

%create window for amplitudes 
figure('name','Amplitude data');
set(gcf,'NumberTitle','off')
%set color range from 0 to 5000 
clims  = [0 5000];
hAmp=imagesc(ampData, clims);
colorbar;
colormap('GRAY');

global MK2;
MK2=0;
set(gcf,'KeyPressFcn','MK2=1;');

ampWindowIsclosed = 0;
distWindowIsclosed = 0;
%loop 
while ~MK && ~MK2 && ~(ampWindowIsclosed && distWindowIsclosed)
    % get one frame
    [status, frameHandle,  frameCounter, timeStamp] = BTAgetFrame(deviceHandle, 500);
    if status == 0
        %get distances
        [status, distData, integrationTime, modulationFrequency, unit] = BTAgetDistances(frameHandle);
        if status == 0
            if ishandle(hDist)
                %display distances
                set(hDist,'CData',distData);
                refreshdata(hDist);
            else
                distWindowIsclosed = 1;
            end
        end
        %get amplitudes
        [status, ampData, integrationTime2, modulationFrequency2, unit2] = BTAgetAmplitudes(frameHandle);
        if status == 0
            if ishandle(hAmp)
                %display amplitudes
                set(hAmp,'CData',ampData);
                refreshdata(hAmp);
            else
                ampWindowIsclosed = 1;   
            end
        end
        %free frame
        [status] = BTAfreeFrame(frameHandle);
        if status ~= 0
            error('BTAfreeFrame failed. See "status" for error description');
        end
    end
    pause(0.1); 
end

%close device
status = BTAclose(deviceHandle);
if status ~= 0
	error('BTAclose failed. See "status" for error description');
end

close all;
clear all;


