---------------------------------------------------------------------------------------------------
--! @brief     PowerupRst module
--! @details   After external reset signal the counter is started. When HOLD_TIME_G is reached the 
--! rst_o signal is deasserted after 3 c-c pipe and D flip flop. Reset is asynchronously asserted.  
--!    
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Oct 11 2018 created
--! @date Dec 28 2018 last modify
--!
--! @version v0.1
--!
--! @file PowerupRst.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
use work.CslStdRtlPkg.all;

--! @brief     PowerupRst module
--! @details   After external reset signal the counter is started. When HOLD_TIME_G is reached the 
--! rst_o signal is deasserted after 3 c-c pipe and DFF. Reset is asynchronously asserted.
---------------------------------------------------------------------------------------------------
entity PowerupRst is
   generic (
      TPD_G          : time      := 1 ns;
      INV_RST_POL_G    : boolean   := false;   --! external reset reverse polarity generic
      HOLD_TIME_G    : positive  := 1000     --! hold time after external reset deassertion
   );
   port (
      clk_i          : in sl;                --! clock input signal 
      extRst_i       : in sl;                --! external reset signal input
      rst_o          : out sl                --! delayed deasserted reset ouptut signal
   );
end PowerupRst;
---------------------------------------------------------------------------------------------------
architecture rtl of PowerupRst is

   --! internal signal
   signal extRst  : sl;
   
   --! Record containing all register elements
   type RegType is record
      Counter     : unsigned(bitSize(HOLD_TIME_G) - 1 downto 0);
      enCounter   : sl;
      sync        : slv(2 downto 0);
   end record RegType;
 
   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      Counter     => (others => '0'),
      enCounter   => '1',
      sync        => (others => '1'));
   
   --! Output of registers. (The Q output)
   signal r : RegType := REG_INIT_C;
   
   --! Combinatorial input to registers (The D input)
   signal rin : RegType;

---------------------------------------------------------------------------------------------------
begin
   --! generate external reset to internal signal throught inverter
   GEN_POLARITY_T: if INV_RST_POL_G  = true generate
      extRst <= not(extRst_i);
   end generate GEN_POLARITY_T;
   
   --! generate external reset to internal signal
   GEN_POLARITY_F: if INV_RST_POL_G  = false generate
      extRst <= extRst_i;
   end generate GEN_POLARITY_F;
   
   --! Main module logic
   --! Generates rin based on r and any module inputs
   p_Comb : process (r, extRst) is
      variable v : RegType;
   begin
      --! Initialize v with current value of all registers
      v        := r;
      v.sync   := r.sync(1 downto 0) & r.enCounter;
      
      if r.enCounter =  '1' then
         --! counter is enabled => upcounting
         v.counter := r.counter + 1;
      end if;
      
      if r.counter < HOLD_TIME_G then
         --! counter has not yet reacher HOLD_TIME_G => enable counter
         v.enCounter := '1';
      else
         --! counter has reacher HOLD_TIME_G => disable counter
         v.enCounter := '0';
      end if;
                  
      --! Synchronous reset
      --! Override all above assignments and apply init values
      if (extRst = '1') then
         v := REG_INIT_C;
      end if;
      
      --! Assign final state of local variable to rin signal
      --! Registers will assume these values on next rising edge
      rin <= v;
            
   end process p_Comb;
 
   --! Boilerplate sequential process
   --! Assign rin to r on rising edge of clk to create registers
   p_Seq : process (clk_i) is begin
       if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
       end if;
   end process p_Seq;
   
   --! Boilerplate sequential process
   --! Assign rin to r on rising edge of clk to create registers
   p_DFF : process (clk_i,extRst) is begin
      if extRst = '1' then
         rst_o <= '1';
      elsif (rising_edge(clk_i)) then
         rst_o <= r.sync(2);
      end if;
   end process p_DFF;
   
end rtl;
---------------------------------------------------------------------------------------------------