---------------------------------------------------------------------------------------------------
--! @brief     CMU core
--! @details 
--!     The CmuCore contains the main logic of the CMU. The data flow through the device goes as follows. 
--!     12 SfedLink-s receive the messages from SFEDs and monitor their flow and integrity. 
--!     The messages arrive once every 25 us. 
--!     The 12 message streams with the corresponding statuses from SFEDs are connected 
--!     to the message concentrator (SfedMsgConc). The message concentrator creates a single message (ItuMsg) 
--!     containing the relative requested and measured scanning magnet currents of all 12 SFEDs with the 
--!     corresponding status information. The ItuMsg is forwarded every 25 us to the ItuLink and CmuFecLink. 
--!     The ItuLink and CmuFecLink transfer the messages over the fiber link toward the ITU and FEC.
--!
--!     This top module connects all the sub-modules.
--!     
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file CmuCore.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.FiberPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity CmuCore is
   generic(
      TPD_G        : time    := 1 ns;
      SIM_G        : boolean := false;
      BUILD_INFO_G : BuildInfoType
   );
   port (
      clk_i : in sl;
      rst_i : in sl;

      gtRx_i : in  GtArray(27 downto 0);
      gtTx_o : out GtArray(27 downto 0)
   );
end entity ;
---------------------------------------------------------------------------------------------------
architecture rtl of CmuCore is

   --! RegInterface signals
   signal writeMaster : AxiLiteWriteMasterType;
   signal writeSlave  : AxiLiteWriteSlaveType;
   signal readMaster  : AxiLiteReadMasterType;
   signal readSlave   : AxiLiteReadSlaveType;

   --! CMU configuration and control
   signal fecCtrl : FecCtrlType;
   signal clear, st_clear, dds_clear,sfed_clear  : sl;
   signal rstTs   : sl;

   --! CMU statuses
   signal fsmStat       : FsmStatType;
   signal fecStat       : FecStatType;
   signal sfedStatArrayReg : SfedStatArray(N_BL_C*2-1 downto 0);
   signal sfedStatArray : SfedStatArray(N_BL_C*2-1 downto 0);   
   signal sfedErr       : sl;
   signal sfedLink      : sl;

   signal itmStatArray  : ItmStatArray(N_BL_C-1 downto 0);   
   signal itmErr        : sl;
   signal itmLink       : sl;
   
   signal ddsStatArray  : DdsStatArray(N_BL_C-1 downto 0);   
   signal ddsErr        : sl;
   signal ddsLink       : sl;
   
   --! CMU nessages
   signal sfedMsgArray : SfedMsgArray(N_BL_C*2-1 downto 0);
   signal ituMsg       : ItuMsgType;
   signal ituMsg1      : ItuMsg1Type;
   signal ituMsg2      : ItuMsg2Type;

   signal itmMsgArray : ItmMsgArray(N_BL_C-1 downto 0);
   signal ddsMsgArray : DdsMsgArray(N_BL_C-1 downto 0);
   
   signal gtTxDdsSfed  : GtType := GT_INIT_C;  
   
   --! attributes keep for chipscope decoding
   attribute keep           : string;
   attribute keep of gtRx_i : signal is "true";
   attribute keep of gtTx_o : signal is "true";

---------------------------------------------------------------------------------------------------
begin

   ---------------------------------------------------------------------------------------------------
   --! FEC--------------------------------------------------------------------------------------------
   ---------------------------------------------------------------------------------------------------      

    u_CmuFecItmLink : entity work.CmuItmFecLink
      generic map (
         TPD_G       => TPD_G,
         LINK_TIME_G => ite(SIM_G, SIM_FIBER_LINK_TIME_C, FIBER_LINK_TIME_C),
         AXIL_TOUT_G => AXIL_REG_TOUT_C)
      port map (
         clk_i                => clk_i,
         rst_i                => rst_i,
         ituMsg_i             => ituMsg1,       
         gtTx_o               => gtTx_o(1),
         gtRx_i               => gtRx_i(1),
         clear_i              => clear,
         fecStat_o            => fecStat,
         regAxilReadMaster_o  => readMaster,
         regAxilReadSlave_i   => readSlave,
         regAxilWriteMaster_o => writeMaster,
         regAxilWriteSlave_i  => writeSlave,
         rstTs_o              => rstTs);   


--     u_CmuFecItmLink : entity work.ItmItuLink
--      generic map (
--         TPD_G => TPD_G)
--      port map (
--         clk_i    => clk_i,
--         rst_i    => rst_i,
--         ituMsg_i => ituMsg1,
--         gtTx_o   => open); --gtTx_o(21));
   ---------------------------------------------------------------------------------------------------
   --! Register interface-----------------------------------------------------------------------------
   ---------------------------------------------------------------------------------------------------      
   u_RegInterface : entity work.RegInterface
      generic map (
         TPD_G        => TPD_G,
         BUILD_INFO_G => BUILD_INFO_G)
      port map (
         axiClk         => clk_i,
         axiRst         => rst_i,
         axiReadMaster  => readMaster,
         axiReadSlave   => readSlave,
         axiWriteMaster => writeMaster,
         axiWriteSlave  => writeSlave,
         sfedMsg_i      => sfedMsgArray,        
         itmMsg_i      => itmMsgArray,
         ddsMsg_i      => ddsMsgArray,         
         fsmStat_i      => fsmStat,
         fecStat_i      => fecStat,
         sfedStat_i     => sfedStatArrayReg,
         fecCtrl_o      => fecCtrl);

   ---------------------------------------------------------------------------------------------------
   --! CMU FSM----------------------------------------------------------------------------------------
   ---------------------------------------------------------------------------------------------------
   u_CmuFsm : entity work.CmuFsm
      generic map (
         TPD_G       => TPD_G,
         INIT_TIME_G => ite(SIM_G, SIM_INIT_TIME_C, INIT_TIME_C))
      port map (
         clk_i      => clk_i,
         rst_i      => rst_i,
         fecCtrl_i  => fecCtrl,
         fecStat_i  => fecStat,
         sfedErr_i  => sfedErr,
         sfedLink_i => sfedLink,
         clear_o    => clear,
         fsmStat_o  => fsmStat);

   ---------------------------------------------------------------------------------------------------
   --! 12 x SFEDS-------------------------------------------------------------------------------------
   ---------------------------------------------------------------------------------------------------
   SFED_GEN : for i in N_BL_C*2-1 downto 0 generate
      u_SfedLink : entity work.SfedLink
         generic map (
            TPD_G       => TPD_G,
            LINK_TIME_G => ite(SIM_G, SIM_FIBER_LINK_TIME_C, FIBER_LINK_TIME_C),
            WD_TIME_G   => 2600)
          --  WD_TIME_G   => ite(SIM_G, SIM_SFED_WD_TIME_C, SFED_WD_TIME_C))
         port map (
            clk_i      => clk_i,
            rst_i      => rst_i,
            gtRx_i     => gtRx_i(i+2),
            clear_i    => sfed_clear,
            sfedStatReg_o => sfedStatArrayReg(i),
            sfedStat_o => sfedStatArray(i),
            sfedMsg_o  => sfedMsgArray(i)
         );
   end generate SFED_GEN;

   ---------------------------------------------------------------------------------------------------
   --! SFED Message concentrator----------------------------------------------------------------------
   ---------------------------------------------------------------------------------------------------
   u_SfedMsgConc : entity work.SfedMsgConc
      generic map (
         TPD_G        => TPD_G,
         N_BL_G       => N_BL_C,
         ITU_PERIOD_G => ite(SIM_G, SIM_ITU_PERIOD_C, ITU_PERIOD_C))
      port map (
         clk_i      => clk_i,
         rst_i      => rst_i,
         sfedMsg_i  => sfedMsgArray,
         sfedStat_i => sfedStatArray,
         fecCtrl_i  => fecCtrl,
         fsmStat_i  => fsmStat,
         ituMsg_o   => ituMsg,
         stobe_o    => sfed_clear,
         sfedErr_o  => sfedErr,
         sfedLink_o => sfedLink,
         rstTs_i    => rstTs);

   ---------------------------------------------------------------------------------------------------
   --! Send the Itu Message to ITU--------------------------------------------------------------------
   ---------------------------------------------------------------------------------------------------
--   u_ItuLink : entity work.ItuLink
--      generic map (
--         TPD_G => TPD_G)
--      port map (
--         clk_i    => clk_i,
--         rst_i    => rst_i,
--         ituMsg_i => ituMsg,
--         gtTx_o   => open); --gtTx_o(0));

   -- Tie the unused outputs to default
   gtTx_o(N_BL_C*2+1 downto 2) <= (others => GT_INIT_C);

--   ---------------------------------------------------------------------------------------------------
--   --! 6 x ITM -------------------------------------------------------------------------------------
--   ---------------------------------------------------------------------------------------------------
   --ITM_GEN : for i in N_BL_C-1 downto 0 generate
   ITM_GEN : for i in 6-1 downto 0 generate
      u_ItmLink : entity work.ItmLink
         generic map (
            TPD_G       => TPD_G,
            LINK_TIME_G => ite(SIM_G, SIM_FIBER_LINK_TIME_C, FIBER_LINK_TIME_C),
            WD_TIME_G   => 600) 
            --WD_TIME_G   => 1000) --ite(SIM_G, SIM_SFED_WD_TIME_C, SFED_WD_TIME_C))
         port map (
            clk_i      => clk_i,
            rst_i      => rst_i,
            gtRx_i     => gtRx_i(i+14),
            clear_i    => st_clear,
            itmStat_o => itmStatArray(i),
            itmMsg_o  => itmMsgArray(i)
         );
   end generate ITM_GEN;
   
--    ---------------------------------------------------------------------------------------------------
--   --! ITM Message concentrator----------------------------------------------------------------------
--   ---------------------------------------------------------------------------------------------------
   u_ItmMsgConc : entity work.ItmMsgConc
      generic map (
         TPD_G        => TPD_G,
         N_BL_G       => N_BL_C,
         ITU_PERIOD_G => 500) --ite(SIM_G, SIM_ITU_PERIOD_C, ITU_PERIOD_C))
      port map (
         clk_i          => clk_i,
         rst_i          => rst_i,
         itmMsg_i       => itmMsgArray,
         itmStat_i      => itmStatArray,
         fecCtrl_i      => fecCtrl,
         fsmStat_i      => fsmStat,
         stobe_o        => st_clear,         
         ituMsg_o       => ituMsg1,
         itmErr_o       => itmErr,
         itmLink_o      => itmLink,
         rstTs_i        => rstTs);
--  ---------------------------------------------------------------------------------------------------
--   --! Send the Itu-Itm Message to ITU--------------------------------------------------------------------
--   ---------------------------------------------------------------------------------------------------
   u_ItmItuLink : entity work.ItmItuLink
      generic map (
         TPD_G => TPD_G)
      port map (
         clk_i    => clk_i,
         rst_i    => rst_i,
         ituMsg_i => ituMsg1,
         gtTx_o   => gtTx_o(20));

  -- Tie the unused outputs to default
   gtTx_o(N_BL_C*3+1 downto 14) <= (others => GT_INIT_C);


--   ---------------------------------------------------------------------------------------------------
--   --! 6 x DDS -------------------------------------------------------------------------------------
--   ---------------------------------------------------------------------------------------------------
   --ITM_GEN : for i in N_BL_C-1 downto 0 generate
   DDS_GEN : for i in 6-1 downto 0 generate
      u_DdsLink : entity work.DdsLink
         generic map (
            TPD_G       => TPD_G,
            LINK_TIME_G => ite(SIM_G, SIM_FIBER_LINK_TIME_C, FIBER_LINK_TIME_C),
            WD_TIME_G   => 1100) --ite(SIM_G, SIM_SFED_WD_TIME_C, SFED_WD_TIME_C))
         port map (
            clk_i      => clk_i,
            rst_i      => rst_i,
            gtRx_i     => gtRx_i(i+22),
            clear_i    => dds_clear,
            ddsStat_o => ddsStatArray(i),
            ddsMsg_o  => ddsMsgArray(i)
         );
   end generate DDS_GEN;
   
      
--    ---------------------------------------------------------------------------------------------------
--   --! DDS Message concentrator----------------------------------------------------------------------
--   ---------------------------------------------------------------------------------------------------
   u_DdsMsgConc : entity work.DdsMsgConc
      generic map (
         TPD_G        => TPD_G,
         N_BL_G       => N_BL_C,
         ITU_PERIOD_G => 1000) --ite(SIM_G, SIM_ITU_PERIOD_C, ITU_PERIOD_C))
      port map (
         clk_i          => clk_i,
         rst_i          => rst_i,
         ddsMsg_i       => ddsMsgArray,
         ddsStat_i      => ddsStatArray,
         fecCtrl_i      => fecCtrl,
         fsmStat_i      => fsmStat,
         stobe_o        => dds_clear,         
         ituMsg_o       => ituMsg2,
         ddsErr_o       => ddsErr,
         ddsLink_o      => ddsLink,
         rstTs_i        => rstTs);
         
--  ---------------------------------------------------------------------------------------------------
--   --! Send the Itu-DDS Message to ITU--------------------------------------------------------------------
--   ---------------------------------------------------------------------------------------------------
--   u_DdsItuLink : entity work.DdsItuLink
--      generic map (
--         TPD_G => TPD_G)
--      port map (
--         clk_i    => clk_i,
--         rst_i    => rst_i,
--         ituMsg_i => ituMsg2,
--         gtTx_o   => open); --gtTx_o(0));
         
    --  ---------------------------------------------------------------------------------------------------
--   --! Send the Itu-DDS Message to ITU--------------------------------------------------------------------
--   ---------------------------------------------------------------------------------------------------
   u_DdsSfedItuLink : entity work.DdsSfedItuLink
      generic map (
         TPD_G => TPD_G)
      port map (
         clk_i    => clk_i,
         rst_i    => rst_i,
         ddsItuMsg_i => ituMsg2,
         sfedItuMsg_i=> ituMsg,                    
         gtTx_o   => gtTxDdsSfed);
         
         gtTx_o(0) <= gtTxDdsSfed;
         gtTx_o(21) <= gtTxDdsSfed;
         
   
end architecture rtl;
---------------------------------------------------------------------------------------------------