---------------------------------------------------------------------------------------------------
--! @brief  CMU fsm module  
--! @details 
--!   The CmuFsm module implements the main CMU state machine. 
--!   Init state - For startup end self-test
--!   Oper state - For normal operation.
--!   Fail state - When error occurs. Stays until RESET_FEC command
--!
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file CmuFsm.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
--
use work.CslStdRtlPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity CmuFsm is
   generic (
      TPD_G       : time     := 1 ns; --! Simulation delta time offset
      INIT_TIME_G : positive := 1000000000
   );
   port (
      clk_i : in sl; --! clock signal bus
      rst_i : in sl; --! reset signal bus

      fecCtrl_i  : in FecCtrlType; --! fec control signals
      fecStat_i  : in FecStatType;
      sfedErr_i  : in sl; --! Error in any enabled SFED message
      sfedLink_i : in sl;
      --
      clear_o   : out sl; --! Clear status counters
      fsmStat_o : out FsmStatType
   );
end CmuFsm;
---------------------------------------------------------------------------------------------------
architecture rtl of CmuFsm is

   --! internal signals
   signal fecCmdRe : sl;
   signal clrCmdRe : sl;

   --! State type containing all states  
   type StateType is (
         INIT0_S,
         INIT1_S,
         OPER_S,
         FAIL_S
      );

   --! Record containing all register elements
   type RegType is record
      fsmStat   : FsmStatType;
      initCnt   : unsigned(bitSize(INIT_TIME_G)-1 downto 0);
      initClear : sl;
      state     : StateType;
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
         fsmStat   => FSM_STAT_INIT_C,
         initCnt   => (others => '0'),
         initClear => '0',
         state     => INIT0_S
      );

   --! Output of registers
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;

---------------------------------------------------------------------------------------------------
begin


   --! Detect rising edges on commands from FEC
   u0_CslEdgeDet : entity work.CslEdgeDet
      generic map (
         TPD_G          => TPD_G,
         FALLING_EDGE_G => false
      )
      port map (
         clk_i => clk_i,
         rst_i => rst_i,
         sig_i => fecCtrl_i.cmd(0),
         sig_o => fecCmdRe
      );
   u1_CslEdgeDet : entity work.CslEdgeDet
      generic map (
         TPD_G          => TPD_G,
         FALLING_EDGE_G => false
      )
      port map (
         clk_i => clk_i,
         rst_i => rst_i,
         sig_i => fecCtrl_i.clearStat,
         sig_o => clrCmdRe
      );

   --! @brief Finite state machine
   comb : process (r, rst_i, fecCtrl_i, fecCmdRe, clrCmdRe, fecStat_i, sfedErr_i, sfedLink_i) is
      variable v : RegType;
   begin
      v := r;


      if r.state = INIT0_S or r.state = INIT1_S then
         v.initClear := '1';
      else
         v.initClear := '0';
      end if;

      --! State Machine
      case r.state is
         ----------------------------------------------------------------------
         when INIT0_S =>
            v.fsmStat.state := INIT_C;
            v.initCnt       := r.initCnt + 1;

            --! condition for transition to different state
         --   if (r.initCnt > to_unsigned(INIT_TIME_G/2-1,bitSize(INIT_TIME_G)) and
          --        fecStat_i.link = '1' and
          --        sfedLink_i = '1')
            if (r.initCnt > to_unsigned(INIT_TIME_G/2-1,bitSize(INIT_TIME_G)))              
              
            then
               v.initCnt := (others => '0');
               v.state   := INIT1_S;
           --    v.state   := OPER_S;
            end if;
         ----------------------------------------------------------------------
         when INIT1_S =>
            v.fsmStat.state := INIT_C;
            v.initCnt       := r.initCnt + 1;

            --! condition for transition to different state
--            if r.initCnt > to_unsigned(INIT_TIME_G/2-1,bitSize(INIT_TIME_G)) then
--               if (fecStat_i.err = '0' and sfedErr_i='0') then
--                  v.state := OPER_S;
--               else
--                  v.state := FAIL_S;
--               end if;

--            end if;
         if r.initCnt > to_unsigned(INIT_TIME_G/2-1,bitSize(INIT_TIME_G)) then               
                  v.state := OPER_S;              
               end if;

            
         ----------------------------------------------------------------------
         when OPER_S =>
            v.fsmStat.state := OPER_C;

          --  if (fecStat_i.err= '1' or sfedErr_i='1') then
          --     v.state := FAIL_S;
          --  end if;
         ----------------------------------------------------------------------
         when FAIL_S =>
            v.fsmStat.state := FAIL_C;

            --! conditions for transition to different states
            if fecCmdRe = FEC_RESET_C(0) and fecStat_i.err = '0' and sfedErr_i='0' then
               v.state := OPER_S;
            end if;
         ----------------------------------------------------------------------
         when others =>
            --! This state should not be reached
            --! Initialize
            v := REG_INIT_C;
      ----------------------------------------------------------------------      
      end case;

      -- Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      rin <= v;

      --! Output assignment
      clear_o   <= r.initClear or clrCmdRe;
      fsmStat_o <= r.fsmStat;

   end process comb;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------