---------------------------------------------------------------------------------------------------
--! @brief  CMU FEC link module
--! @details    
--!   This module is a Receiver/Transmitter for data throught GT fiber link. In FEC Link HPD and LPD 
--!   data are received and transmitted. Fiber signals and msg stream signals are monitored with 
--!   FiberMon module. LPD data stream is used for burst READ/WRITE with AXI-Lite communication 
--!   protocol. AXI-Lite protocol and handshaking is implemented in ConfigCtrl module.
--!    
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file CmuFecLink.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.FiberPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity CmuItmFecLink is
   generic (
      TPD_G       : time     := 1 ns;
      LINK_TIME_G : positive := 100000;  --! link time for FiberMon watchdog
      AXIL_TOUT_G : positive := 100000); --! AXI-Lite response timeout
   port (
      clk_i : in sl;                                     --! clock signal bus
      rst_i : in sl;                                     --! reset signal bus
                                                         --! TX
      ituMsg_i : in  ItuMsg1Type;                         --! ITU message signals
      gtTx_o   : out GtType := GT_INIT_C;                --! GT transmitter signals
                                                         --! RX
      gtRx_i : in GtType;                                --! GT receiver signals
                                                         --! Fiber Monitor
      clear_i   : in  sl;                                --! clear monitoring counters signal
      fecStat_o : out FecStatType;                       --! FEC Stat signals
                                                         --! AXI-Lite master
      regAxilReadMaster_o  : out AxiLiteReadMasterType;  --! AXI-Lite read master signals
      regAxilReadSlave_i   : in  AxiLiteReadSlaveType;   --! AXI-Lite read slave signals
      regAxilWriteMaster_o : out AxiLiteWriteMasterType; --! AXI-Lite write master signals
      regAxilWriteSlave_i  : in  AxiLiteWriteSlaveType;  --! AXI-Lite write slave signals

      rstTs_o : out sl --! Reset the ItuMsg timestamp  
   );
end CmuItmFecLink;
---------------------------------------------------------------------------------------------------
architecture structure of CmuItmFecLink is

   -- Size of the message in 32-bit words
 --  constant FEC_TX_MSG_SIZE_C : positive := 45;
   constant FEC_TX_MSG_SIZE_C : positive := 63;
   constant FEC_RX_MSG_SIZE_C : positive := 4;
   constant CONF_MSG_SIZE_C   : positive := 24;

   --! Fiber Monitor
   signal link_po    : sl              := '1';
   signal statusBits : slv(2 downto 0) := (others => '0');
   signal fecStat_po : FecStatType;

   --! configuration message
   signal cfgMsg_pi : CfgMsgType;
   signal cfgMsg_po : CfgMsgType;
   signal cfgDrop   : sl;

   --! TX 
   signal hpdTxMsg32Array : Slv32Array(FEC_TX_MSG_SIZE_C-1 downto 0);
   signal hpdTxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdTxAxisSlave  : AxiStreamSlaveType;
   signal hpdTxPktFull    : sl;
   signal hpdTxErr        : slv(1 downto 0);
   signal hpdIdle         : sl;
   signal lpdTxMsg32Array : Slv32Array(CONF_MSG_SIZE_C-1 downto 0);
   signal lpdTxAxisMaster : AxiStreamMasterArray(0 downto 0) := (others => AXI_STREAM_MASTER_INIT_C);
   signal lpdTxAxisSlave  : AxiStreamSlaveArray(0 downto 0);
   signal lpdTxPktFull    : slv(0 downto 0);
   signal lpdTxErr        : Slv2Array(0 downto 0);
   signal lpdIdle         : sl;
   signal cfgTxDrop       : sl;

   --! RX
   signal fecMsgArray     : Slv32Array(FEC_RX_MSG_SIZE_C-1 downto 0);
   signal hpdRxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdRxAxisSlave  : AxiStreamSlaveType;
   signal hpdRxPktEmpty   : sl;

   type Slv32Array2DType is array (0 downto 0) of Slv32Array(CONF_MSG_SIZE_C-1 downto 0);
   signal lpdMsg32Array2D : Slv32Array2DType;
   signal lpdRxAxisMaster : AxiStreamMasterArray(0 downto 0) := (others => AXI_STREAM_MASTER_INIT_C);
   signal lpdRxAxisSlave  : AxiStreamSlaveArray(0 downto 0);
   signal lpdRxPktEmpty   : slv(0 downto 0);
   signal fecDrop         : sl;
   signal cfgRxDrop       : sl;

   --! Output message
   signal fecStrobe : sl;
   signal cfgStrobe : sl;
   signal cfgCrcErr : sl;
   signal cfgLenErr : sl;

   --! Register 
   signal rRstTs : sl;

   --! attributes keep for chipscope decoding => can be deleted after testing
   attribute keep              : string;
   attribute keep of cfgMsg_pi : signal is "true";
   attribute keep of cfgMsg_po : signal is "true";
   attribute keep of cfgCrcErr : signal is "true";
   attribute keep of cfgLenErr : signal is "true";
---------------------------------------------------------------------------------------------------
begin

   --! ITU message encoding to 32 bit vector array for FEC with absolute and relative currents
   --hpdTxMsg32Array <= encHpdTxFec(ituMsg_i);
   hpdTxMsg32Array <= encHpdTxItmItu(ituMsg_i);


   --! Convert ITU message to AXI4-Stream and transmit
   u0_MsgStreamTx : entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_G,
         SIZE_G => FEC_TX_MSG_SIZE_C)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => ituMsg_i.strobe,
         msg32Array_i => hpdTxMsg32Array,
         full_i       => hpdTxPktFull,
         axisMaster_o => hpdTxAxisMaster,
         axisSlave_i  => hpdTxAxisSlave,
         drop_o       => open);

   --! CFG message decoding to 32 bit vector array
   lpdTxMsg32Array <= decCfgFec(cfgMsg_po);

   --! Convert CFG message to AXI4-Stream and transmit
   u1_MsgStreamTx : entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_G,
         SIZE_G => CONF_MSG_SIZE_C)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => cfgMsg_po.strobe,
         msg32Array_i => lpdTxMsg32Array,
         full_i       => lpdTxPktFull(0),
         axisMaster_o => lpdTxAxisMaster(0),
         axisSlave_i  => lpdTxAxisSlave(0),
         drop_o       => cfgTxDrop);

   --! FiberLinkMapTx module to transmit the messages over GT fiber 
   u_FiberLinkMapTx : entity work.FiberLinkMapTx
      generic map (
         TPD_G            => TPD_G,
         HPD_DATA_WIDTH_G => 4,
         HPD_MAX_SIZE_G   => FEC_TX_MSG_SIZE_C,
         LPD_DATA_WIDTH_G => 4,
         LPD_PIPES_G      => 1,
         LPD_MAX_SIZE_G   => CONF_MSG_SIZE_C)
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         hpdTxAxisMaster_i => hpdTxAxisMaster,
         hpdTxAxisSlave_o  => hpdTxAxisSlave,
         hpdTxPktFull_o    => hpdTxPktFull,
         hpdTxErr_o        => hpdTxErr,
         hpdIdle_o         => hpdIdle,
         lpdTxAxisMaster_i => lpdTxAxisMaster,
         lpdTxAxisSlave_o  => lpdTxAxisSlave,
         lpdTxPktFull_o    => lpdTxPktFull,
         lpdTxErr_o        => lpdTxErr,
         lpdIdle_o         => lpdIdle,
         gtDataTx_o        => gtTx_o.data,
         gtCharTx_o        => gtTx_o.char);

   --! driving Clock correction enable signal from Lpd and Hpd Idle signals
   gtTx_o.clkCorrEn <= lpdIdle and hpdIdle;

   --! CFG message encoding from 32 bit vector array
   cfgMsg_pi <= encCfgFec(lpdMsg32Array2D(0), cfgStrobe);

   --! Configuration control
   --! Bridge between Fiber link messages and the AXI-Lite. 
   u_ConfigCtrl : entity work.ConfigCtrl
      generic map (
         TPD_G         => TPD_G,
         DEVICE_ADDR_G => 1, -- Note: Device address in CMU is 1, while in ITU it is 0
         TOUT_G        => AXIL_TOUT_G,
         BURST_SIZE_G  => 16 -- Only max burst size (mostly 1 transaction will be used in CMU)
      )
      port map (
         clk_i         => clk_i,
         rst_i         => rst_i,
         crcErr_i      => cfgCrcErr,
         lenErr_i      => cfgLenErr,
         parErr_i      => '0', -- No parity error since this is not an UART interface
         cfgMsg_i      => cfgMsg_pi,
         writeMaster_o => regAxilWriteMaster_o,
         writeSlave_i  => regAxilWriteSlave_i,
         readMaster_o  => regAxilReadMaster_o,
         readSlave_i   => regAxilReadSlave_i,
         cfgMsg_o      => cfgMsg_po,
         drop_o        => cfgDrop
      );

   --! FiberLinkMapRx module to receive the messages from GT fiber 
   u_FiberLinkMapRx : entity work.FiberLinkMapRx
      generic map (
         TPD_G            => TPD_G,
         HPD_DATA_WIDTH_G => 4,
         HPD_MAX_SIZE_G   => FEC_RX_MSG_SIZE_C,
         LPD_DATA_WIDTH_G => 4,
         LPD_PIPES_G      => 1,
         LPD_MAX_SIZE_G   => CONF_MSG_SIZE_C)
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         gtDataRx_i        => gtRx_i.data,
         gtCharRx_i        => gtRx_i.char,
         link_i            => fecStat_po.link,
         hpdRxAxisMaster_o => hpdRxAxisMaster,
         hpdRxAxisSlave_i  => hpdRxAxisSlave,
         hpdRxPktEmpty_o   => open, -- This is none-critical and is not monitored 
         hpdRxDrop_o       => open, -- This is none-critical and is not monitored 
         lpdRxAxisMaster_o => lpdRxAxisMaster,
         lpdRxAxisSlave_i  => lpdRxAxisSlave,
         lpdRxPktEmpty_o   => lpdRxPktEmpty,
         lpdRxDrop_o       => cfgRxDrop);

   --! HPD MsgStreamRx
   u0_MsgStreamRx : entity work.MsgStreamRx
      generic map (
         TPD_G      => TPD_G,
         SIZE_G     => FEC_RX_MSG_SIZE_C,
         REGISTER_G => false,
         STR_SUP_G  => true)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => hpdRxAxisMaster,
         axisSlave_o  => hpdRxAxisSlave,
         strobe_o     => fecStrobe,
         msg32Array_o => fecMsgArray,
         crcErr_o     => open,  -- This is none-critical and is not monitored fecStb will be suppressed if in error 
         lenErr_o     => open); -- This is none-critical and is not monitored fecStb will be suppressed if in error 

   --! LPD MsgStreamRx
   u1_MsgStreamRx : entity work.MsgStreamRx
      generic map (
         TPD_G     => TPD_G,
         SIZE_G    => CONF_MSG_SIZE_C,
         STR_SUP_G => false)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => lpdRxAxisMaster(0),
         axisSlave_o  => lpdRxAxisSlave(0),
         strobe_o     => cfgStrobe,
         msg32Array_o => lpdMsg32Array2D(0),
         crcErr_o     => cfgCrcErr,
         lenErr_o     => cfgLenErr);

   --! Assign status signals to StatusBits vector
   statusBits(0) <= cfgCrcErr;                         -- RX cfg CRC errors
   statusBits(1) <= cfgLenErr;                         -- RX cfg LEN errors
   statusBits(2) <= cfgRxDrop or cfgTxDrop or cfgDrop; -- CFG packet drop

   --! Monitor the connection status
   u_FiberMon : entity work.FiberMon
      generic map (
         TPD_G             => TPD_G,
         STAT_SIZE_G       => 3,
         LINK_TIME_WIDTH_G => bitSize(LINK_TIME_G)
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         clear_i           => clear_i,
         statusBits_i      => statusBits,
         gtRx_i            => gtRx_i,
         linkTime_i        => toSlv(LINK_TIME_G, bitSize(LINK_TIME_G)),
         ilk_o             => open,
         err_o             => fecStat_po.err,
         sts32array_o(0)   => fecStat_po.gtCdrStableCnt,
         sts32array_o(1)   => fecStat_po.gtDecErrCnt,
         sts32array_o(2)   => fecStat_po.gtDispErrCnt,
         sts32array_o(3)   => fecStat_po.gtByteReAlig,
         sts32array_o(0+4) => fecStat_po.cfgCrcCnt,
         sts32array_o(1+4) => fecStat_po.cfgLenCnt,
         sts32array_o(2+4) => fecStat_po.cfgDropCnt,
         link_o            => fecStat_po.link
      );

   --! Check TIMESYNC message and generate the timestamp reset signal
   --! Since it is a very simple logic a normal sequential process is used instead of the dual process coding style
   seq : process(clk_i)
   begin
      if rising_edge(clk_i) then
         if rst_i = '1' then
            rRstTs <= '0';
         else
            -- Generate Reg strobe
            if (fecStrobe = '1' and
                  fecMsgArray(0) = x"0000_0001" and
                  fecMsgArray(1) = x"0000_0000" and
                  fecMsgArray(2) = x"0000_0000" and
                  fecMsgArray(3) = x"0000_0000"
               ) then
               rRstTs <= '1';
            else
               rRstTs <= '0';
            end if;
         ---------------------------------------------------------------------------------------------
         end if;
      end if;
   end process seq;

   --! Output assignment
   fecStat_o <= fecStat_po;
   rstTs_o   <= rRstTs;

end structure;
---------------------------------------------------------------------------------------------------