---------------------------------------------------------------------------------------------------
--! @brief  Configuration control module  
--! @details 
--!   AXI-Lite communication protocol is generated in this module. Received message record is decoded
--!   and WRITE/READ operations are performed. Burst WRITE/READ is implemented with error monitoring 
--!   and proper AXI-Lite handshaking.
--!   
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Dec 12 2018 created
--! @date Jan 3 2018 last modify
--!
--! @version v1.0
--!
--! @file ConfigCtrl.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity ConfigCtrl is
   generic (
      TPD_G         : time     := 1 ns; --! Simulation delta time offset
      DEVICE_ADDR_G : natural  := 0;    --! Device address
      TOUT_G        : positive := 1000; --! timeout value
      BURST_SIZE_G  : positive := 16    --! maximum burst size
   );
   port (
      clk_i : in sl; --! clock signal bus
      rst_i : in sl; --! reset signal bus

      crcErr_i : in sl; --! CRC error signal bus
      lenErr_i : in sl; --! length error signal bus
      parErr_i : in sl; --! parity error signal bus

      cfgMsg_i : in CfgMsgType; --! input configuration message signals

      writeMaster_o : out AxiLiteWriteMasterType; --! AXI-Lite write master signals
      writeSlave_i  : in  AxiLiteWriteSlaveType;  --! AXI-Lite write slave signals
      readMaster_o  : out AxiLiteReadMasterType;  --! AXI-Lite read master signals
      readSlave_i   : in  AxiLiteReadSlaveType;   --! AXI-Lite read slave signals

      cfgMsg_o : out CfgMsgType; --! output configuration message signals
      drop_o   : out sl          --! drop message signal output
   );
end ConfigCtrl;
---------------------------------------------------------------------------------------------------
architecture rtl of ConfigCtrl is

   --! State type containing all states  
   type StateType is (
         IDLE_S,
         READ_S,
         READ_WAIT_S,
         WRITE_S,
         WRITE_WAIT_S,
         MSG_SEND_S
      );

   --! Record containing all register elements
   type RegType is record
      writeMaster : AxiLiteWriteMasterType;
      readMaster  : AxiLiteReadMasterType;
      cfgMsg      : CfgMsgType;
      burstCnt    : natural;
      toutCnt     : natural;
      strobe      : sl;
      statusOr    : sl;
      drop        : sl;
      state       : StateType;
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
         writeMaster => AXI_LITE_WRITE_MASTER_INIT_C,
         readMaster  => AXI_LITE_READ_MASTER_INIT_C,
         cfgMsg      => CFG_MSG_INIT_C,
         burstCnt    => 0,
         toutCnt     => 0,
         strobe      => '0',
         statusOr    => '0',
         drop        => '0',
         state       => IDLE_S
      );

   --! Output of registers
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;
---------------------------------------------------------------------------------------------------
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, cfgMsg_i, writeSlave_i, readSlave_i, crcErr_i, lenErr_i) is
      variable v : RegType;
   begin
      v          := r;
      v.strobe   := cfgMsg_i.strobe;
      v.statusOr := r.cfgMsg.parErr or
         r.cfgMsg.crcErr or
         r.cfgMsg.lenErr or
         r.cfgMsg.adrErr or
         r.cfgMsg.devErr or
         r.cfgMsg.touErr or
         r.cfgMsg.cmdErr or
         r.cfgMsg.acsErr;

      if cfgMsg_i.strobe = '1' and (r.state /= IDLE_S) then
         --! drop is asserted if strobe is detected while FSM is not in IDLE state
         v.drop := '1';
      else
         v.drop := '0';
      end if;

      --! State Machine
      case r.state is
         ----------------------------------------------------------------------
         when IDLE_S =>
            --! register input message and errors
            v.writeMaster := AXI_LITE_WRITE_MASTER_INIT_C;
            v.readMaster  := AXI_LITE_READ_MASTER_INIT_C;
            if (cfgMsg_i.strobe = '1') then
               v.cfgMsg        := cfgMsg_i;
               v.cfgMsg.crcErr := crcErr_i;
               v.cfgMsg.lenErr := lenErr_i;
            end if;

            v.cfgMsg.strobe := '0';
            v.burstCnt      := 0;
            v.toutCnt       := 0;

            if (v.cfgMsg.cmd /= x"01") and (v.cfgMsg.cmd /= x"02") then
               --! unknown command
               v.cfgMsg.cmdErr := '1';
            end if;

            if v.cfgMsg.dev /= slv(to_unsigned(DEVICE_ADDR_G, 32)) then
               --! wrong device ID
               v.cfgMsg.devErr := '1';
            end if;

            if (v.cfgMsg.len > slv(to_unsigned(BURST_SIZE_G,32))) or
               (v.cfgMsg.len = x"00000000") then
               --! length of brust READ/WRITE in message in bigger that BURST_SIZE_G
               v.cfgMsg.lenErr := '1';
            end if;

            --! Next state
            if (r.cfgMsg.cmd = x"01" and r.strobe = '1') then
               --! received command READ registers
               v.state := READ_S;
            end if;

            --! Next state
            if (r.cfgMsg.cmd = x"02" and r.strobe = '1') then
               --! received command WRITE registers
               v.state := WRITE_S;
            end if;

            --! Next state
            if (r.strobe='1') and v.statusOr = '1' then
               --! received strobe and message with errors
               v.state := MSG_SEND_S;
            end if;
         ----------------------------------------------------------------------
         when READ_S =>
            --! AXI-LIte READ protocl handshaking
            v.readMaster.araddr  := slv(unsigned(r.cfgMsg.addr) + to_unsigned(r.burstCnt*4,32));
            v.readMaster.arprot  := (others => '0');
            v.readMaster.arvalid := '1';
            v.readMaster.rready  := '1';
            v.toutCnt            := 0;
            v.state              := READ_WAIT_S;
         ----------------------------------------------------------------------
         when READ_WAIT_S =>
            --! timeout for protocol handshaking
            v.toutCnt := r.toutCnt + 1;

            --! AXI-LIte READ protocl handshaking
            if readSlave_i.arready = '1' then
               v.readMaster.arvalid := '0';
            end if;

            if readSlave_i.rvalid = '1' then
               --! read signals are valid
               v.readMaster.rready       := '0';
               v.cfgMsg.data(r.burstCnt) := readSlave_i.rdata;

               if readSlave_i.rresp = "10" then
                  --! read only error
                  v.cfgMsg.acsErr := '1';
               end if;

               if readSlave_i.rresp = "11" then
                  --! address is not represented in Axi Register Space
                  v.cfgMsg.adrErr := '1';
               end if;
            end if;

            --! next state
            if r.toutCnt >= TOUT_G-1 then
               --! timeout without response
               v.cfgMsg.touErr := '1';
            else
               --! response before timeout => OK
               if r.readMaster.arvalid = '0' and r.readMaster.rready = '0' then
                  v.burstCnt := r.burstCnt + 1;
                  if r.burstCnt = to_integer(unsigned(r.cfgMsg.len))-1 then
                     --! all registers in burst READ protocol were acquired
                     v.state := MSG_SEND_S;
                  else
                     --! next register in burst READ is to be acquired
                     v.state := READ_S;
                  end if;
               end if;
            end if;

            if v.statusOr = '1' then
               --! error was receiver, send message
               v.state := MSG_SEND_S;
            end if;
         ----------------------------------------------------------------------
         when WRITE_S =>
            --! AXI-LIte WRITE protocl handshaking
            v.writeMaster.awaddr  := slv(unsigned(r.cfgMsg.addr) + to_unsigned(r.burstCnt*4,32));
            v.writeMaster.awprot  := (others => '0');
            v.writeMaster.wstrb   := (others => '1');
            v.writeMaster.wdata   := r.cfgMsg.data(r.burstCnt);
            v.writeMaster.awvalid := '1';
            v.writeMaster.wvalid  := '1';
            v.writeMaster.bready  := '1';
            v.toutCnt             := 0;
            v.state               := WRITE_WAIT_S;
         ----------------------------------------------------------------------
         when WRITE_WAIT_S =>
            --! timeout for protocol handshaking
            v.toutCnt := r.toutCnt + 1;

            if writeSlave_i.awready = '1' then
               v.writeMaster.awvalid := '0';
            end if;

            if writeSlave_i.wready = '1' then
               v.writeMaster.wvalid := '0';
            end if;

            if writeSlave_i.bvalid = '1' then
               --! write signals are valid
               v.writeMaster.bready := '0';
               if writeSlave_i.bresp = "10" then
                  --! read only error
                  v.cfgMsg.acsErr := '1';
               end if;

               if writeSlave_i.bresp = "11" then
                  --! address is not represented in Axi Register Space
                  v.cfgMsg.adrErr := '1';
               end if;
            end if;

            --! next state
            if r.toutCnt >= TOUT_G-1 then
               --! timeout without response
               v.cfgMsg.touErr := '1';
            else
               --! response before timeout => OK
               if r.writeMaster.awvalid = '0' and r.writeMaster.wvalid = '0' and
                  r.writeMaster.bready = '0' then
                  v.burstCnt := r.burstCnt + 1;
                  if r.burstCnt = to_integer(unsigned(r.cfgMsg.len))-1 then
                     --! all registers in burst WRITE protocol were rewritten
                     v.state := MSG_SEND_S;
                  else
                     --! next register in burst WRITE is to be rewritten
                     v.state := WRITE_S;
                  end if;
               end if;
            end if;

            --! 
            if v.statusOr = '1' then
               --! error was receiver, send message
               v.burstCnt := r.burstCnt; --! actual written/read registers
               v.state    := MSG_SEND_S;
            end if;
         ----------------------------------------------------------------------
         when MSG_SEND_S =>
            --! burst READ/WRITE is complete => sending ITU response
            v.cfgMsg.strobe := '1';
            v.cfgMsg.rsp    := '1';
            v.cfgMsg.len    := slv(to_unsigned(r.burstCnt,32));
            v.state         := IDLE_S;
         ----------------------------------------------------------------------
         when others =>
            --! This state should not be reached
            --! Initialize
            v := REG_INIT_C;
      ----------------------------------------------------------------------      
      end case;

      -- Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      rin <= v;

      --! Output assignment
      writeMaster_o <= r.writeMaster;
      readMaster_o  <= r.readMaster;
      drop_o        <= r.drop;
      cfgMsg_o      <= r.cfgMsg;
   end process comb;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------