---------------------------------------------------------------------------------------------------
--! @brief  ITU link module
--! @details    
--!   The ITU link sends an ITU message to the ITU with a constant frequency of 40 kHz (every 25 us). 
--!   The ITU message ituMsg_i, received from SfedMsgConc, is converted to AXI-Stream within the MsgStreamTx module 
--!   and connected to the FiberLinkTx. 
--!   The FiberLinkTx converts the message to the Fiber message format sends it over the HPD pipe (data(15:8)).
--!    
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Oct 07 2019 last modify
--!
--! @version v0.1
--!
--! @file ItuLink.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.FiberPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity DdsItuLink is
   generic (
      TPD_G       : time     := 1 ns
   );
   port (
      clk_i : in sl;                                     --! clock signal bus
      rst_i : in sl;                                     --! reset signal bus
                                                         --! TX
      ituMsg_i : in  ItuMsg2Type;                        --! ITU message signals
      gtTx_o   : out GtType := GT_INIT_C                 --! GT transmitter signals
   );
end DdsItuLink;
---------------------------------------------------------------------------------------------------
architecture structure of DdsItuLink is
   
   -- Size of the message in 32-bit words
   constant ITU_MSG_SIZE_C : positive := 2;

   --! Fiber Monitor
   signal link_po    : sl              := '1';
   signal statusBits : slv(2 downto 0) := (others => '0');
   signal fecStat_po : FecStatType;

   --! TX 
   signal hpdTxMsg32Array : Slv32Array(14 downto 0);
   signal hpdTxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdTxAxisSlave  : AxiStreamSlaveType;
   signal hpdTxPktFull    : sl;
   signal hpdIdle         : sl;

   --! RX
   signal hpdMsg32array_o : Slv32Array(4-1 downto 0);
   signal hpdRxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdRxAxisSlave  : AxiStreamSlaveType;
   signal hpdRxPktEmpty_o : sl;
   type lpdMsg32Array2D is array (0 downto 0) of Slv32Array(24-1 downto 0);
   signal lpdMsg32Array2D_o : lpdMsg32Array2D;
   signal lpdRxAxisMaster   : AxiStreamMasterArray(0 downto 0) := (others => AXI_STREAM_MASTER_INIT_C);
   signal lpdRxAxisSlave    : AxiStreamSlaveArray(0 downto 0);
   signal lpdRxPktEmpty_o   : slv(0 downto 0);
   signal fecDrop           : sl;
   signal cfgRxDrop         : sl;

   --! Output message
   signal fecStrobe : sl;
   signal cfgStrobe : sl;
   signal fecCrcErr : sl;
   signal fecLenErr : sl;
   signal cfgCrcErr : sl;
   signal cfgLenErr : sl;

---------------------------------------------------------------------------------------------------
begin

   --! ITU message encoding to 32 bit vector array for ITU with relative currents only
   hpdTxMsg32Array <= encHpdTxDdsItu(ituMsg_i);

   --! Convert ITU message to AXI4-Stream and transmit
   u0_MsgStreamTx : entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_G,
         SIZE_G =>  15) --ITU_MSG_SIZE_C)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => ituMsg_i.strobe,
         msg32Array_i => hpdTxMsg32Array,
         full_i       => hpdTxPktFull,
         axisMaster_o => hpdTxAxisMaster,
         axisSlave_i  => hpdTxAxisSlave,
         drop_o       => open);

   --! FiberLinkTx module to transmit the messages over GT fiber 
   u_FiberLinkTx : entity work.FiberLinkTx
      generic map (
         TPD_G        => TPD_G,
         IDLE_G       => '0' & x"00",
         DATA_WIDTH_G => 4,
         MAX_SIZE_G   => 15) --ITU_MSG_SIZE_C)
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         axisMaster_i(0)   => hpdTxAxisMaster,
         axisSlave_o(0)    => hpdTxAxisSlave,
         txPktFull_o(0)    => hpdTxPktFull,
         txErr_o           => open,
         idle_o            => hpdIdle,
         dataTx_o          => gtTx_o.data(15 downto 8),
         charTx_o          => gtTx_o.char(1));
   
   --! driving Clock correction enable signal from Lpd and Hpd Idle signals
   gtTx_o.clkCorrEn         <= hpdIdle;
   -- LPD always comma
   gtTx_o.data(7 downto 0)  <= x"BC";
   gtTx_o.char(0)           <= '1';
---------------------------------------------------------------------------------------------------
end structure;
---------------------------------------------------------------------------------------------------