---------------------------------------------------------------------------------------------------
--! @brief  ITM Message concentrator
--! @details    
--!      The message concentrator receives messages from the 6 ITMs (itmMsg_i), processes the messages 
--!      and the internal statuses, and combines the relevant information into the (ituMsg_o).
--!      
--!      Note: The ITU message strobe and the ITM message data are not synced. 
--!      Registers the data and produces 1 c-c delay.
--!    
--!--!
--!--!
--! @version v1.0
--!
--! @file ItmMsgConc.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity DdsMsgConc is
   generic (
      TPD_G  : time    := 1 ns;
      N_BL_G : positive:= 6;
      ITU_PERIOD_G : positive:= 1000
   );
   port(
      clk_i       	: in  sl;
      rst_i      	: in  sl;

      -- ITM messages and statuses
      ddsMsg_i   	: in DdsMsgArray(N_BL_G-1 downto 0);
      ddsStat_i  	: in DdsStatArray(N_BL_G-1 downto 0);
      fecCtrl_i   	: in FecCtrlType;      
      fsmStat_i   	: in FsmStatType;
      rstTs_i     	: in sl;
      --
      stobe_o       : out sl;
      ituMsg_o    	: out ItuMsg2Type;
      ddsErr_o   	: out sl;
      ddsLink_o  	: out sl
   );
end DdsMsgConc;
---------------------------------------------------------------------------------------------------
architecture rtl of DdsMsgConc is

   --! Record containing all register elements 
   type RegType is record
      ituMsg2Buff    : ItuMsg2Type;
      counter       : unsigned(63 downto 0);
      ddsLinks      : slv(N_BL_G-1 downto 0);
      en            : sl;
   end record RegType;

   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      ituMsg2Buff    => ITU_MSG2_INIT_C,
      counter       => (others => '0'),
      ddsLinks     => (others => '0'),
      en            => '0');

   --! Internal signal
   signal strobe : sl;
   
   --! Output of registers          
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;
---------------------------------------------------------------------------------------------------
begin

   u_PulseTrGen: entity work.PulseTrGen
      generic map (
         TPD_G   => TPD_G,
         WIDTH_G => 16)
      port map (
         clk_i    => clk_i,
         rst_i    => rst_i,
         en_i     => r.en,
         period_i => toSlv(ITU_PERIOD_G-1,16),
         tick_o   => strobe);

     --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   combi : process (r, rst_i, strobe, ddsMsg_i, ddsStat_i, fecCtrl_i, fsmStat_i, rstTs_i) is
      variable v : RegType;
     
   begin
      -- Register the current value
      v  := r;
      
      -- Data path
      --------------------------------------------------------------------------------------------      
		
		-- Status Flags		
		-- Errors
		    v.ituMsg2Buff.statFlagsDds0(2)		:= uOr(ddsStat_i(0).crcCnt)or uOr(ddsStat_i(0).lenCnt)or uOr(ddsStat_i(0).dropCnt) ;
			v.ituMsg2Buff.statFlagsDds0(1)		:= uOr(ddsStat_i(0).gtDecErrCnt) or uOr(ddsStat_i(0).gtDispErrCnt) or uOr(ddsStat_i(0).gtByteReAlig); 
	       	v.ituMsg2Buff.statFlagsDds0(0)		:= uOr(ddsStat_i(0).toutCnt) ;
	       	    	     
    	    v.ituMsg2Buff.statFlagsDds1(2)		:= uOr(ddsStat_i(1).crcCnt)or uOr(ddsStat_i(1).lenCnt)or uOr(ddsStat_i(1).dropCnt) ;
			v.ituMsg2Buff.statFlagsDds1(1)		:= uOr(ddsStat_i(1).gtDecErrCnt) or uOr(ddsStat_i(1).gtDispErrCnt) or uOr(ddsStat_i(1).gtByteReAlig);
			v.ituMsg2Buff.statFlagsDds1(0)		:= uOr(ddsStat_i(1).toutCnt) ;
			 
			v.ituMsg2Buff.statFlagsDds2(2)		:= uOr(ddsStat_i(2).crcCnt)or uOr(ddsStat_i(2).lenCnt)or uOr(ddsStat_i(2).dropCnt) ;
			v.ituMsg2Buff.statFlagsDds2(1)		:= uOr(ddsStat_i(2).gtDecErrCnt) or uOr(ddsStat_i(2).gtDispErrCnt) or uOr(ddsStat_i(2).gtByteReAlig);
			v.ituMsg2Buff.statFlagsDds2(0)		:= uOr(ddsStat_i(2).toutCnt) ; 
			
			v.ituMsg2Buff.statFlagsDds3(2)		:= uOr(ddsStat_i(3).crcCnt)or uOr(ddsStat_i(3).lenCnt)or uOr(ddsStat_i(3).dropCnt) ;
			v.ituMsg2Buff.statFlagsDds3(1)		:= uOr(ddsStat_i(3).gtDecErrCnt) or uOr(ddsStat_i(3).gtDispErrCnt) or uOr(ddsStat_i(3).gtByteReAlig);
			v.ituMsg2Buff.statFlagsDds3(0)		:= uOr(ddsStat_i(3).toutCnt) ; 
			
			v.ituMsg2Buff.statFlagsDds4(2)		:= uOr(ddsStat_i(4).crcCnt)or uOr(ddsStat_i(4).lenCnt)or uOr(ddsStat_i(4).dropCnt) ;
			v.ituMsg2Buff.statFlagsDds4(1)		:= uOr(ddsStat_i(4).gtDecErrCnt) or uOr(ddsStat_i(4).gtDispErrCnt) or uOr(ddsStat_i(4).gtByteReAlig);
			v.ituMsg2Buff.statFlagsDds4(0)		:= uOr(ddsStat_i(4).toutCnt) ; 
			
			v.ituMsg2Buff.statFlagsDds5(2)		:= uOr(ddsStat_i(5).crcCnt)or uOr(ddsStat_i(5).lenCnt)or uOr(ddsStat_i(5).dropCnt) ;
			v.ituMsg2Buff.statFlagsDds5(1)		:= uOr(ddsStat_i(5).gtDecErrCnt) or uOr(ddsStat_i(5).gtDispErrCnt) or uOr(ddsStat_i(5).gtByteReAlig);
			v.ituMsg2Buff.statFlagsDds5(0)		:= uOr(ddsStat_i(5).toutCnt) ; 
					
		--  Data 
			v.ituMsg2Buff.dataDds0      	    :=  ddsMsg_i(0);
			v.ituMsg2Buff.dataDds1      	    :=  ddsMsg_i(1);
			v.ituMsg2Buff.dataDds2      	    :=  ddsMsg_i(2);
			v.ituMsg2Buff.dataDds3      	    :=  ddsMsg_i(3);
			v.ituMsg2Buff.dataDds4      	    :=  ddsMsg_i(4);
			v.ituMsg2Buff.dataDds5      	    :=  ddsMsg_i(5);			                   			                 
               

      -- Enable timestamp and message strobe
      --------------------------------------------------------------------------------------------
      if (fsmStat_i.state = OPER_C or fsmStat_i.state = FAIL_C) then
         v.en := '1';
      else
         v.en := '0';      
      end if;
      
      -- Timestamp counter
      --------------------------------------------------------------------------------------------
      if (r.en = '1') then
         v.counter := r.counter + 1;
     end if;
      
      -- Reset/clear the counter if disabled or requested by FEC in TIMESYNC message
      if (r.en = '0' or rstTs_i = '1') then      
         v.counter := (others => '0');  
      end if;

      --
      v.ituMsg2Buff.timeStamp := slv(r.counter);
      
     -- Strobe
      --------------------------------------------------------------------------------------------
      v.ituMsg2Buff.strobe := strobe;
      
      -- Reset
      --------------------------------------------------------------------------------------------
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      -- Register the variable for next clock cycle
      rin <= v;
      
      -- Output 
      ituMsg_o <= r.ituMsg2Buff;
      
      -- Logic OR of all the enabled SFED statuses
      ddsErr_o <=  '0'; --uor(r.ituMsg1Buff.statFlags);
      ddsLink_o <= '0';  --uand(r.sfedLinks);
      stobe_o <= strobe;
   end process combi;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk_i_i to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------