---------------------------------------------------------------------------------------------------
--! @brief  ITM Message concentrator
--! @details    
--!      The message concentrator receives messages from the 6 ITMs (itmMsg_i), processes the messages 
--!      and the internal statuses, and combines the relevant information into the (ituMsg_o).
--!      
--!      Note: The ITU message strobe and the ITM message data are not synced. 
--!      Registers the data and produces 1 c-c delay.
--!    
--!--!
--!--!
--! @version v1.0
--!
--! @file ItmMsgConc.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity ItmMsgConc is
   generic (
      TPD_G  : time    := 1 ns;
      N_BL_G : positive:= 6;
      ITU_PERIOD_G : positive:= 500
   );
   port(
      clk_i       	: in  sl;
      rst_i      	: in  sl;

      -- ITM messages and statuses
      itmMsg_i   	: in ItmMsgArray(N_BL_G-1 downto 0);
      itmStat_i  	: in ItmStatArray(N_BL_G-1 downto 0);
      fecCtrl_i   	: in FecCtrlType;      
      fsmStat_i   	: in FsmStatType;
      rstTs_i     	: in sl;
      --
      stobe_o       : out sl;
      ituMsg_o    	: out ItuMsg1Type;
      itmErr_o   	: out sl;
      itmLink_o  	: out sl
   );
end ItmMsgConc;
---------------------------------------------------------------------------------------------------
architecture rtl of ItmMsgConc is

   --! Record containing all register elements 
   type RegType is record
      ituMsg1Buff    : ItuMsg1Type;
      counter       : unsigned(63 downto 0);
      itmLinks      : slv(N_BL_G-1 downto 0);
      en            : sl;
   end record RegType;

   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      ituMsg1Buff    => ITU_MSG1_INIT_C,
      counter       => (others => '0'),
      itmLinks     => (others => '0'),
      en            => '0');

   --! Internal signal
   signal strobe : sl;
   
   --! Output of registers          
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;
---------------------------------------------------------------------------------------------------
begin

   u_PulseTrGen: entity work.PulseTrGen
      generic map (
         TPD_G   => TPD_G,
         WIDTH_G => 16)
      port map (
         clk_i    => clk_i,
         rst_i    => rst_i,
         en_i     => r.en,
         period_i => toSlv(ITU_PERIOD_G-1,16),
         tick_o   => strobe);

     --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   combi : process (r, rst_i, strobe, itmMsg_i, itmStat_i, fecCtrl_i, fsmStat_i, rstTs_i) is
      variable v : RegType;
     
   begin
      -- Register the current value
      v  := r;
      
      -- Data path
      --------------------------------------------------------------------------------------------      
		
		-- Status Flags		
		-- Errors
		    v.ituMsg1Buff.statFlagsItm0(2)		:= uOr(itmStat_i(0).crcCnt)or uOr(itmStat_i(0).lenCnt)or uOr(itmStat_i(0).dropCnt) ;
			v.ituMsg1Buff.statFlagsItm0(1)		:= uOr(itmStat_i(0).gtDecErrCnt) or uOr(itmStat_i(0).gtDispErrCnt) or uOr(itmStat_i(0).gtByteReAlig); 
	       	v.ituMsg1Buff.statFlagsItm0(0)		:= uOr(itmStat_i(0).toutCnt) ;
	       	    	     
    	    v.ituMsg1Buff.statFlagsItm1(2)		:= uOr(itmStat_i(1).crcCnt)or uOr(itmStat_i(1).lenCnt)or uOr(itmStat_i(1).dropCnt) ;
			v.ituMsg1Buff.statFlagsItm1(1)		:= uOr(itmStat_i(1).gtDecErrCnt) or uOr(itmStat_i(1).gtDispErrCnt) or uOr(itmStat_i(1).gtByteReAlig);
			v.ituMsg1Buff.statFlagsItm1(0)		:= uOr(itmStat_i(1).toutCnt) ;
			 
			v.ituMsg1Buff.statFlagsItm2(2)		:= uOr(itmStat_i(2).crcCnt)or uOr(itmStat_i(2).lenCnt)or uOr(itmStat_i(2).dropCnt) ;
			v.ituMsg1Buff.statFlagsItm2(1)		:= uOr(itmStat_i(2).gtDecErrCnt) or uOr(itmStat_i(2).gtDispErrCnt) or uOr(itmStat_i(2).gtByteReAlig);
			v.ituMsg1Buff.statFlagsItm2(0)		:= uOr(itmStat_i(2).toutCnt) ; 
			
			v.ituMsg1Buff.statFlagsItm3(2)		:= uOr(itmStat_i(3).crcCnt)or uOr(itmStat_i(3).lenCnt)or uOr(itmStat_i(3).dropCnt) ;
			v.ituMsg1Buff.statFlagsItm3(1)		:= uOr(itmStat_i(3).gtDecErrCnt) or uOr(itmStat_i(3).gtDispErrCnt) or uOr(itmStat_i(3).gtByteReAlig);
			v.ituMsg1Buff.statFlagsItm3(0)		:= uOr(itmStat_i(3).toutCnt) ; 
			
			v.ituMsg1Buff.statFlagsItm4(2)		:= uOr(itmStat_i(4).crcCnt)or uOr(itmStat_i(4).lenCnt)or uOr(itmStat_i(4).dropCnt) ;
			v.ituMsg1Buff.statFlagsItm4(1)		:= uOr(itmStat_i(4).gtDecErrCnt) or uOr(itmStat_i(4).gtDispErrCnt) or uOr(itmStat_i(4).gtByteReAlig);
			v.ituMsg1Buff.statFlagsItm4(0)		:= uOr(itmStat_i(4).toutCnt) ; 
			
			v.ituMsg1Buff.statFlagsItm5(2)		:= uOr(itmStat_i(5).crcCnt)or uOr(itmStat_i(5).lenCnt)or uOr(itmStat_i(5).dropCnt) ;
			v.ituMsg1Buff.statFlagsItm5(1)		:= uOr(itmStat_i(5).gtDecErrCnt) or uOr(itmStat_i(5).gtDispErrCnt) or uOr(itmStat_i(5).gtByteReAlig);
			v.ituMsg1Buff.statFlagsItm5(0)		:= uOr(itmStat_i(5).toutCnt) ; 
					
		--  Data 
			v.ituMsg1Buff.dataItm0      	    :=  itmMsg_i(0);
			v.ituMsg1Buff.dataItm1      	    :=  itmMsg_i(1);
			v.ituMsg1Buff.dataItm2      	    :=  itmMsg_i(2);
			v.ituMsg1Buff.dataItm3      	    :=  itmMsg_i(3);
			v.ituMsg1Buff.dataItm4      	    :=  itmMsg_i(4);
			v.ituMsg1Buff.dataItm5      	    :=  itmMsg_i(5);			                   			                 
               

      -- Enable timestamp and message strobe
      --------------------------------------------------------------------------------------------
      if (fsmStat_i.state = OPER_C or fsmStat_i.state = FAIL_C) then
         v.en := '1';
      else
         v.en := '0';      
      end if;
      
      -- Timestamp counter
      --------------------------------------------------------------------------------------------
      if (r.en = '1') then
         v.counter := r.counter + 1;
     end if;
      
      -- Reset/clear the counter if disabled or requested by FEC in TIMESYNC message
      if (r.en = '0' or rstTs_i = '1') then      
         v.counter := (others => '0');  
      end if;

      --
      v.ituMsg1Buff.timeStamp := slv(r.counter);
      
     -- Strobe
      --------------------------------------------------------------------------------------------
      v.ituMsg1Buff.strobe := strobe;
      
      -- Reset
      --------------------------------------------------------------------------------------------
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      -- Register the variable for next clock cycle
      rin <= v;
      
      -- Output 
      ituMsg_o <= r.ituMsg1Buff;
      
      -- Logic OR of all the enabled SFED statuses
      itmErr_o <=  '0'; --uor(r.ituMsg1Buff.statFlags);
      itmLink_o <= '0';  --uand(r.sfedLinks);
      stobe_o <= strobe;
   end process combi;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk_i_i to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------