---------------------------------------------------------------------------------------------------
--! @brief  Register space module
--! @details    
--!   The module is Axi-Lite slave. In this module Status registers are implemented. 
--!   read only.
--!
--!   SVN location of register map is on link:
--!       https://internal.cosylab.com/svn/acc/projects/MedAustron/CMU/trunk/documentation/architecture/CmuRegisters.xlsx
--! 
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file RegInterface.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity RegInterface is
   generic (
      TPD_G            : time            := 1 ns;
      AXI_ERROR_RESP_G : slv(1 downto 0) := AXI_RESP_SLVERR_C;
      BUILD_INFO_G     : BuildInfoType
   );
   port(
      axiClk : in sl;
      axiRst : in sl;

      axiReadMaster  : in  AxiLiteReadMasterType;
      axiReadSlave   : out AxiLiteReadSlaveType;
      axiWriteMaster : in  AxiLiteWriteMasterType;
      axiWriteSlave  : out AxiLiteWriteSlaveType;

      -- SFED messages
      sfedMsg_i : in SfedMsgArray(N_BL_C*2-1 downto 0);    
      -- ITM messages
      itmMsg_i : in ItmMsgArray(N_BL_C-1 downto 0);
       -- DDS messages
      ddsMsg_i : in DdsMsgArray(N_BL_C-1 downto 0);
      
      -- Statuses
      fsmStat_i  : in FsmStatType;
      fecStat_i  : in FecStatType;
      sfedStat_i : in SfedStatArray(N_BL_C*2-1 downto 0);
      -- Controls
      fecCtrl_o : out FecCtrlType
   );
end RegInterface;
---------------------------------------------------------------------------------------------------
architecture rtl of RegInterface is

   --! internal signals
   signal s_RdAddr : natural := 0;
   signal s_WrAddr : natural := 0;

   --! Record containing all register elements 
   type RegType is record
      buildInfo : BuildInfoRetType;
      sfedMsg   : sfedMsgArray(N_BL_C*2-1 downto 0);
      fsmStat   : FsmStatType;
      fecStat   : FecStatType;
      sfedStat  : SfedStatArray(N_BL_C*2-1 downto 0);
      -----------------------------------------
      fecCtrl : FecCtrlType;
      -----------------------------------------      
      axiReadSlave  : AxiLiteReadSlaveType;
      axiWriteSlave : AxiLiteWriteSlaveType;
      
      itmMsg   : itmMsgArray(N_BL_C-1 downto 0);
      ddsMsg   : ddsMsgArray(N_BL_C-1 downto 0);
      
   end record RegType;

   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
         buildInfo => BUILD_INFO_DEFAULT_C,
         sfedMsg   => (others => SFED_MSG_INIT_C),
         --test
         itmMsg   => (others => ITM_MSG_INIT_C),
         ddsMsg   => (others => DDS_MSG_INIT_C),
         --test
         fsmStat   => FSM_STAT_INIT_C,
         fecStat   => FEC_STAT_INIT_C,
         sfedStat  => (others => SFED_STAT_INIT_C),
         --------------------------------------------------
         fecCtrl => FEC_CTRL_INIT_C,
         --------------------------------------------------         
         axiReadSlave  => AXI_LITE_READ_SLAVE_INIT_C,
         axiWriteSlave => AXI_LITE_WRITE_SLAVE_INIT_C);
         
   --! Output of registers          
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;
---------------------------------------------------------------------------------------------------
begin

   --! Convert address to integer (lower two bits of address are always '0')
   s_RdAddr <= toInt(axiReadMaster.araddr(16 downto 0));
   s_WrAddr <= toInt(axiWriteMaster.awaddr(16 downto 0));

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   --test
   --comb : process (axiRst, axiReadMaster, axiWriteMaster, r, s_RdAddr, s_WrAddr, fecStat_i, sfedStat_i, fsmStat_i, sfedMsg_i) is
   comb : process (axiRst, axiReadMaster, axiWriteMaster, r, s_RdAddr, s_WrAddr, fecStat_i, sfedStat_i, fsmStat_i, sfedMsg_i,itmMsg_i,ddsMsg_i) is
   --test
      variable v            : RegType;
      variable axiStatus    : AxiLiteStatusType;
      variable axiWriteResp : slv(1 downto 0);
      variable axiReadResp  : slv(1 downto 0);
   begin
      -- Register the current value
      v           := r;
      v.buildInfo := toBuildInfo(BUILD_INFO_G);
      v.sfedMsg   := sfedMsg_i;
      v.fsmStat   := fsmStat_i;
      v.fecStat   := fecStat_i;
      v.sfedStat  := sfedStat_i;
      v.itmMsg   := itmMsg_i;
      v.ddsMsg   := ddsMsg_i;

      ----------------------------------------------------------------------------------------------
      -- Axi-Lite interface
      ----------------------------------------------------------------------------------------------
      axiSlaveWaitTxn(axiWriteMaster, axiReadMaster, v.axiWriteSlave, v.axiReadSlave, axiStatus);

      ----------------------------------------------------------------------------------------------
      -- Write registers
      ----------------------------------------------------------------------------------------------
      if (axiStatus.writeEnable = '1') then
         axiWriteResp := ite(axiWriteMaster.awaddr(1 downto 0) = "00", AXI_RESP_OK_C, AXI_ERROR_RESP_G);
         case (s_WrAddr) is
            when 16#10000# =>
               v.fecCtrl.cmd := axiWriteMaster.wdata(0 downto 0);
            when 16#10004# =>
               v.fecCtrl.clearStat := axiWriteMaster.wdata(0);
            when 16#10008# =>
               v.fecCtrl.sfedEnable := axiWriteMaster.wdata(5 downto 0);
            when others =>
               axiWriteResp := AXI_RESP_SLVERR_C;
         end case;
         axiSlaveWriteResponse(v.axiWriteSlave, axiWriteResp);
      end if;

      ----------------------------------------------------------------------------------------------
      -- Read registers
      ----------------------------------------------------------------------------------------------
      if (axiStatus.readEnable = '1') then
         axiReadResp          := ite(axiReadMaster.araddr(1 downto 0) = "00", AXI_RESP_OK_C, AXI_ERROR_RESP_G);
         v.axiReadSlave.rdata := (others => '0');
         case (s_RdAddr) is
            when 16#00000# =>
               v.axiReadSlave.rdata := r.buildInfo.fwVersion;
            when 16#00004# to 16#00014# =>
               for i in 0 to 4 loop
                  if (s_RdAddr = i*4 + 16#00004#) then
                     v.axiReadSlave.rdata := r.buildInfo.gitHash(31 + i*32 downto i*32);
                  end if;
               end loop;
            when 16#00018# to 16#00114# =>
               for i in 0 to 63 loop
                  if (s_RdAddr = i*4 + 16#00018#) then
                     v.axiReadSlave.rdata := r.buildInfo.buildString(i);
                  end if;
               end loop;
            ------------------------------------------------------------------
            when 16#00200# =>
               v.axiReadSlave.rdata(31 downto 2) := (others => '0');
               v.axiReadSlave.rdata(1 downto 0)  := r.fsmStat.state;
            ------------------------------------------------------------------
            when 16#00300# =>
               v.axiReadSlave.rdata(31 downto 2) := (others => '0');
               v.axiReadSlave.rdata(1)           := r.fecStat.link;
               v.axiReadSlave.rdata(0)           := r.fecStat.err;
            when 16#00304# =>
               v.axiReadSlave.rdata := r.fecStat.cfgCrcCnt;
            when 16#00308# =>
               v.axiReadSlave.rdata := r.fecStat.cfgLenCnt;
            when 16#0030C# =>
               v.axiReadSlave.rdata := r.fecStat.cfgDropCnt;
            when 16#00310# =>
               v.axiReadSlave.rdata := r.fecStat.gtDecErrCnt;
            when 16#00314# =>
               v.axiReadSlave.rdata := r.fecStat.gtDispErrCnt;
            when 16#00318# =>
               v.axiReadSlave.rdata := r.fecStat.gtCdrStableCnt;
            when 16#0031C# =>
               v.axiReadSlave.rdata := r.fecStat.gtByteReAlig;
            ------------------------------------------------------------------
            when 16#00400# to 16#00FFF# =>
               for i in 0 to 11 loop
                  if (s_RdAddr = 16#0000# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 2) := (others => '0');
                     v.axiReadSlave.rdata(1)           := r.sfedStat(i).link;
                     v.axiReadSlave.rdata(0)           := r.sfedStat(i).err;
                  end if;
                  if (s_RdAddr = 16#0004# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).crcCnt;
                  end if;
                  if (s_RdAddr = 16#0008# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).lenCnt;
                  end if;
                  if (s_RdAddr = 16#000C# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).dropCnt;
                  end if;
                  if (s_RdAddr = 16#0010# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).toutCnt;
                  end if;
                  if (s_RdAddr = 16#0014# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).tmstpStallCnt;
                  end if;
                  if (s_RdAddr = 16#0018# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).ddsLinkCnt;
                  end if;
                  if (s_RdAddr = 16#001C# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).ddsParityCnt;
                  end if;
                  if (s_RdAddr = 16#0020# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).gtDecErrCnt;
                  end if;
                  if (s_RdAddr = 16#0024# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).gtDispErrCnt;
                  end if;
                  if (s_RdAddr = 16#0028# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).gtCdrStableCnt;
                  end if;
                  if (s_RdAddr = 16#002C# + 16#00400# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedStat(i).gtByteReAlig;
                  end if;
               end loop;
            ------------------------------------------------------------------
            when 16#01000# to 16#01C00# =>
               for i in 0 to 11 loop
                  if (s_RdAddr = 16#0000# + 16#01000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 6) := (others => '0');
                     v.axiReadSlave.rdata(5 downto 0)  := r.sfedMsg(i).statFlags;
                  end if;
                  if (s_RdAddr = 16#0004# + 16#01000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedMsg(i).setpAbsCurr;
                  end if;
                  if (s_RdAddr = 16#0008# + 16#01000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedMsg(i).measAbsCurr;
                  end if;
                  if (s_RdAddr = 16#000C# + 16#01000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 16) := (others => '0');
                     v.axiReadSlave.rdata(15 downto 0)  := r.sfedMsg(i).setpRelCurr;
                  end if;
                  if (s_RdAddr = 16#0010# + 16#01000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 16) := (others => '0');
                     v.axiReadSlave.rdata(15 downto 0)  := r.sfedMsg(i).measRelCurr;
                  end if;
                  if (s_RdAddr = 16#0014# + 16#01000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedMsg(i).timeStamp(31 downto 0);
                  end if;
                  if (s_RdAddr = 16#0018# + 16#01000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.sfedMsg(i).timeStamp(63 downto 32);
                  end if;
               end loop;
            ------------------------------------------------------------------
           
            when 16#02000# to 16#02600# =>
               for i in 0 to 5 loop
                  if (s_RdAddr = 16#0000# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 16) := (others => '0');
                     v.axiReadSlave.rdata(15 downto 0)  := r.itmMsg(i).statFlags;
                  end if;
                  if (s_RdAddr = 16#0004# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 16) := (others => '0');
                     v.axiReadSlave.rdata (15 downto 0) := r.itmMsg(i).timeStamp;
                  end if;
                  if (s_RdAddr = 16#0008# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.itmMsg(i).countsLow;
                  end if;
                  if (s_RdAddr = 16#000C# + 16#02000# + 16#00100#*i) then                     
                     v.axiReadSlave.rdata  := r.itmMsg(i).countsHigh;
                  end if;
                  if (s_RdAddr = 16#0010# + 16#02000# + 16#00100#*i) then                     
                     v.axiReadSlave.rdata  := r.itmMsg(i).tempIn;
                  end if;
                  if (s_RdAddr = 16#0014# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.itmMsg(i).tempOut;
                  end if;
                  if (s_RdAddr = 16#0018# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.itmMsg(i).pressureIn;                                        
                  end if;
                  if (s_RdAddr = 16#001C# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.itmMsg(i).pressureOut;                                        
                  end if;
                  if (s_RdAddr = 16#0020# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.itmMsg(i).fluxIn;                                        
                  end if;
                  if (s_RdAddr = 16#0024# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.itmMsg(i).fluxOut;                                        
                  end if;
                  if (s_RdAddr = 16#0028# + 16#02000# + 16#00100#*i) then
                     v.axiReadSlave.rdata := r.itmMsg(i).itmCrc;                                        
                  end if;
               end loop;
          ------------------------------------------------------------------     

            when 16#03000# to 16#03600# =>
               for i in 0 to 5 loop
                  if (s_RdAddr = 16#0000# + 16#03000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 4) := (others => '0');
                     v.axiReadSlave.rdata(3 downto 0)  := r.ddsMsg(i).ddsInstId;
                  end if;
                  if (s_RdAddr = 16#0004# + 16#03000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 8) := (others => '0');
                     v.axiReadSlave.rdata (7 downto 0) := r.ddsMsg(i).msgId;
                  end if;
                  if (s_RdAddr = 16#0008# + 16#03000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 1) := (others => '0');
                     v.axiReadSlave.rdata(0) := r.ddsMsg(i).isDelActive;
                  end if;
                  if (s_RdAddr = 16#000C# + 16#03000# + 16#00100#*i) then                     
                     v.axiReadSlave.rdata(31 downto 1) := (others => '0');
                     v.axiReadSlave.rdata(0)  := r.ddsMsg(i).isPosValid;
                  end if;
                  if (s_RdAddr = 16#0010# + 16#03000# + 16#00100#*i) then                     
                     v.axiReadSlave.rdata(31 downto 8) := (others => '0');
                     v.axiReadSlave.rdata(7 downto 0) := r.ddsMsg(i).currLayerId;
                  end if;
                  if (s_RdAddr = 16#0014# + 16#03000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 14) := (others => '0');
                     v.axiReadSlave.rdata(13 downto 0) := r.ddsMsg(i).currSpotId;
                  end if;
                  if (s_RdAddr = 16#0018# + 16#03000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 13) := (others => '0');
                     v.axiReadSlave.rdata(12 downto 0) := r.ddsMsg(i).posX;                                        
                  end if;
                  if (s_RdAddr = 16#001C# + 16#03000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 13) := (others => '0');
                     v.axiReadSlave.rdata(12 downto 0) := r.ddsMsg(i).posY;                                        
                  end if;
                  if (s_RdAddr = 16#0020# + 16#03000# + 16#00100#*i) then
                     v.axiReadSlave.rdata(31 downto 2) := (others => '0');
                     v.axiReadSlave.rdata (1 downto 0) := r.ddsMsg(i).pad;                                        
                  end if;                                       
             end loop;
               ------------------------------------------------------------------
            when 16#10000# =>
               v.axiReadSlave.rdata(31 downto 1) := (others => '0');
               v.axiReadSlave.rdata(0 downto 0)  := r.fecCtrl.cmd;
            when 16#10004# =>
               v.axiReadSlave.rdata(31 downto 1) := (others => '0');
               v.axiReadSlave.rdata(0)           := r.fecCtrl.clearStat;
            when 16#10008# =>
               v.axiReadSlave.rdata(31 downto 6) := (others => '0');
               v.axiReadSlave.rdata(5 downto 0)  := r.fecCtrl.sfedEnable;
               

               
            when others =>
               axiReadResp := AXI_RESP_DECERR_C;
         end case;
         axiSlaveReadResponse(v.axiReadSlave, axiReadResp);
      end if;

      -- Reset
      if (axiRst = '1') then
         v := REG_INIT_C;
      end if;

      -- Register the variable for next clock cycle
      rin <= v;

      -- Outputs
      axiReadSlave  <= r.axiReadSlave;
      axiWriteSlave <= r.axiWriteSlave;
      fecCtrl_o     <= r.fecCtrl;
   end process comb;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (axiClk) is
   begin
      if (rising_edge(axiClk)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------