---------------------------------------------------------------------------------------------------
--! @brief  SFED link module
--! @details    
--!   The CMU receives a message from the SFED with a constant frequency of 40 kHz (every 25 us). 
--!   The message, received from fiber, is processed by a FiberLinkRx module connected to the HPD pipe (data(15:8)).
--!   The MsgStreamRx makes a parallel message array valid at strobe.
--!   Fiber signals and msg stream signals are monitored with FiberMon module. Watchdog is used for 
--!   monitoring frequency. 
--!    
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file SfedLink.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.FiberPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity SfedLink is
   generic (
      TPD_G       : time     := 1 ns;
      LINK_TIME_G : positive := 100000; --! link time generic for FiberMon
      WD_TIME_G   : positive := 2500);  --! watchdog timer generic     
   port (
      clk_i  : in sl;     --! clock signal bus
      rst_i  : in sl;     --! reset signal bus
      gtRx_i : in GtType; --! GT Rx signals
                          --! SfedLink operation signals
      clear_i : in sl;    --! clear counters signal

      sfedStatReg_o : out SfedStatType; --! SFED stat signals
      sfedStat_o : out SfedStatType; --! SFED stat signals
      sfedMsg_o  : out SfedMsgType   --! SFED message 
   );
end SfedLink;
---------------------------------------------------------------------------------------------------
architecture structure of SfedLink is

   --! Size of the message in 32-bit words
   constant SFED_MSG_SIZE_C : positive := 8;

   --! Fiber Monitor
   signal link_po    : sl              := '1';
   signal statusBits : slv(6 downto 0) := (others => '0');
   signal sfedStatReg : SfedStatType;
   signal sfedStat   : SfedStatType;

   --! configuration message
   signal sfedMsg : SfedMsgType;

   --! RX Axi4-stream
   signal hpdRxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdRxAxisSlave  : AxiStreamSlaveType;

   --! Output message
   signal strobe     : sl;
   signal msg32array : Slv32Array(7 downto 0);
   signal crcErr     : sl;
   signal lenErr     : sl;

   signal tmstpStall : sl;
   signal tout       : sl;
   signal wdRst      : sl;
   signal sfedDrop   : sl;

   --! SfedLink record
   type RegType is record
      tmstpStall   : sl;
      timeStampReg : slv(63 downto 0);
   end record RegType;

   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
         tmstpStall   => '0',
         timeStampReg => (others => '0'));

   --! Output of registers          
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;
   ---------------------------------------------------------------------------------------------------  
   --! attributes keep for chipscope decoding
   attribute keep           : string;
   attribute keep of crcErr : signal is "true";
   attribute keep of lenErr : signal is "true";
   attribute keep of tout   : signal is "true";
   attribute keep of r      : signal is "true";
   attribute keep of gtRx_i : signal is "true";
   ---------------------------------------------------------------------------------------------------  

---------------------------------------------------------------------------------------------------  
begin

   --! Watchdog reset
 --    wdRst <= strobe or clear_i or tout;       
   wdRst <= strobe;

   --! Watchdog module instantiation
   u0_Watchdog : entity work.Watchdog
      generic map (
         TPD_G   => TPD_G,
         WIDTH_G => bitSize(WD_TIME_G)
      )
      port map (
         clk_i   => clk_i,
         rst_i   => rst_i,
         en_i    => '1',
         wdRst_i => wdRst,
         time_i  => toSlv(WD_TIME_G, bitSize(WD_TIME_G)),
         tout_o  => tout
      );

   --! Fiber Link Rx module instantiation
   u0_FiberLinkRx : entity work.FiberLinkRx
      generic map (
         TPD_G        => TPD_G,
         DATA_WIDTH_G => 4,
         PIPES_G      => 1,
         IDLE_G       => '0' & x"00",
         SOF_CHAR_G   => x"1C",
         EOF_CHAR_G   => x"FE",
         MAX_SIZE_G   => SFED_MSG_SIZE_C
      )
      port map (
         clk_i           => clk_i,
         rst_i           => rst_i,
         dataRx_i        => gtRx_i.data(15 downto 8),
         charRx_i        => gtRx_i.char(1),
         link_i          => sfedStat.link,
         axisMaster_o(0) => hpdRxAxisMaster,
         axisSlave_i(0)  => hpdRxAxisSlave,
         pktEmpty_o      => open,
         drop_o          => sfedDrop
      );

   --! Message Stream Rx module instantiation
   u_MsgStreamRx : entity work.MsgStreamRx
      generic map (
         TPD_G      => TPD_G,
         SIZE_G     => SFED_MSG_SIZE_C,
         REGISTER_G => true,
         STR_SUP_G  => false
      )
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => hpdRxAxisMaster,
         axisSlave_o  => hpdRxAxisSlave,
         strobe_o     => strobe,
         msg32Array_o => msg32array,
         crcErr_o     => crcErr,
         lenErr_o     => lenErr);

   --! sfed message encoding from 32 bit vector array
   sfedMsg <= encSfedMsg(msg32array, strobe);

   --! Assign status signals to StatusBits vector
   statusBits(0) <= crcErr;                   -- RX sfed CRC errors
   statusBits(1) <= lenErr;                   -- RX sfed LEN errors
   statusBits(2) <= sfedDrop;                 -- RX sfed drop errors
   statusBits(3) <= r.tmstpStall;             -- timestamp stall
   statusBits(4) <= tout;                     -- timeout counter
   statusBits(5) <= not sfedMsg.statFlags(0); -- DDS Link ok
   statusBits(6) <= sfedMsg.statFlags(1);     -- DDS oarity error

   --! Fiber Monitor module instantiation
   u_FiberMon : entity work.FiberMon
      generic map (
         TPD_G             => TPD_G,
         STAT_SIZE_G       => 7,
         LINK_TIME_WIDTH_G => bitSize(LINK_TIME_G)
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         clear_i           => clear_i,
         statusBits_i      => statusBits,
         gtRx_i            => gtRx_i,
         linkTime_i        => toSlv(LINK_TIME_G, bitSize(LINK_TIME_G)),
         ilk_o             => open,
         err_o             => sfedStat.err,
         sts32arrayReg_o(0)   => sfedStatReg.gtCdrStableCnt,
         sts32arrayReg_o(1)   => sfedStatReg.gtDecErrCnt,
         sts32arrayReg_o(2)   => sfedStatReg.gtDispErrCnt,
         sts32arrayReg_o(3)   => sfedStatReg.gtByteReAlig,
         sts32arrayReg_o(0+4) => sfedStatReg.crcCnt,
         sts32arrayReg_o(1+4) => sfedStatReg.lenCnt,
         sts32arrayReg_o(2+4) => sfedStatReg.dropCnt,
         sts32arrayReg_o(3+4) => sfedStatReg.tmstpStallCnt,
         sts32arrayReg_o(4+4) => sfedStatReg.toutCnt,
         sts32arrayReg_o(5+4) => sfedStatReg.ddsLinkCnt,
         sts32arrayReg_o(6+4) => sfedStatReg.ddsParityCnt,
         sts32array_o(0)   => sfedStat.gtCdrStableCnt,
         sts32array_o(1)   => sfedStat.gtDecErrCnt,
         sts32array_o(2)   => sfedStat.gtDispErrCnt,
         sts32array_o(3)   => sfedStat.gtByteReAlig,
         sts32array_o(0+4) => sfedStat.crcCnt,
         sts32array_o(1+4) => sfedStat.lenCnt,
         sts32array_o(2+4) => sfedStat.dropCnt,
         sts32array_o(3+4) => sfedStat.tmstpStallCnt,
         sts32array_o(4+4) => sfedStat.toutCnt,
         sts32array_o(5+4) => sfedStat.ddsLinkCnt,
         sts32array_o(6+4) => sfedStat.ddsParityCnt,
         link_o            => sfedStat.link
      );

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (rst_i, clk_i, r, strobe, sfedMsg) is
      variable v : RegType;
   begin
      -- Latch the current value
      v            := r;
      v.tmstpStall := '0';

--      -- timestamp register
--      if (strobe = '1') then
--         v.timeStampReg := sfedMsg.timeStamp;
--         if sfedMsg.timeStamp = r.timeStampReg then
--            --! new timestamp is the same as the old timestamp => Stall
--            v.tmstpStall := '1';
--         end if;
--      end if;

      -- Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      -- Register the variable for next clock cycle
      rin <= v;
   end process comb;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

   --! Output assignment
   sfedStatReg_o <= sfedStatReg;
   sfedStat_o <= sfedStat;
   sfedMsg_o  <= sfedMsg;

end structure;
---------------------------------------------------------------------------------------------------