---------------------------------------------------------------------------------------------------
--! @brief ItuCore top module test bench 
--! @details
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokaljt@cosylab.com)
--!
--! @date Jul 01 2019 created
--! @date Jul 01 2019 last modify
--!
--! @version v0.1
--!
--! @file CmuCoreTb.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.FiberPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity CmuCoreTb is
end entity CmuCoreTb;
---------------------------------------------------------------------------------------------------
architecture Bhv of CmuCoreTb is

   -- Testbench DUT generics as constants
   constant TPD_C              : time := 1 ns;
   constant CLK_PERIOD_C       : time := 10 ns;
   constant CLK_DELAY_C        : time := 0 ns;
   
   -- Internal logic
   signal stall               : sl:='0';  
   signal tsCounter           : unsigned(63 downto 0);
   signal strCounter          : unsigned(bitsize(SIM_ITU_PERIOD_C)-1 downto 0);
   signal clearCmu            : sl:='0';     

   -- Testbench DUT ports as signals
   signal clk_i         : sl := '0';
   signal rst_i         : sl := '1';

   signal gtRx_i       : GtArray(13 downto 0):=(others => GT_INIT_C);
   signal gtTx_o       : GtArray(13 downto 0);
   signal gtTx_po      : GtType;
   
   --! TX FEC
   signal fecStrobe    : sl := '0';
   signal fecDrop      : sl := '0';
   signal fecConfigMsg : Slv32Array(23 downto 0) := (others => x"00000000");
   signal fecConfigRsp : Slv32Array(23 downto 0) := (others => x"00000000");   
   --
   signal fecTxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal fecTxAxisSlave  : AxiStreamSlaveType;

   --! TX SFED
   type Slv32Array2DType is array(N_BL_C*2-1 downto 0) of Slv32Array(7 downto 0);      
   signal strobe         : slv(N_BL_C*2-1 downto 0) := (others => '0');   
   signal sfedMsg32Array : Slv32Array2DType:= (others=>(others => x"00000000"));
   --
   signal sfedAxisMaster : AxiStreamMasterArray(N_BL_C*2-1 downto 0) := (others => AXI_STREAM_MASTER_INIT_C);
   signal sfedAxisSlave  : AxiStreamSlaveArray(N_BL_C*2-1 downto 0);
   --
   signal hpdIdle         : slv(N_BL_C*2-1 downto 0) := (others => '0'); 
   signal sfedPeriod      : slv(bitSize(SIM_ITU_PERIOD_C)-1 downto 0) := (others => '0');
   
   --! Inject
   signal injCrc        : sl := '0';
   signal injLen        : sl := '0';
   signal injId         : sl := '0';
   signal cntID         : slv(7 downto 0) := (others => '1');
      
   -- ITU Message verificator
   signal ituStat  : SfedStatType;            
   signal ituMsg   : ItuMsgType;
   signal fecStat  : SfedStatType; 
   signal fecMsg   : ItuMsgType;   

---------------------------------------------------------------------------------------------------
begin

   -----------------------------------------------------------
   -- Clocks and Reset
   -----------------------------------------------------------
   process is
   begin
      wait for CLK_DELAY_C;
      while (true) loop
         clk_i <= not clk_i;
         wait for CLK_PERIOD_C/2.0;
      end loop;
   end process;

   -----------------------------------------------------------
   -- Entity Under Test
   -----------------------------------------------------------
   u0_CmuCore : entity work.CmuCore
      generic map (
         TPD_G        => TPD_C,
         SIM_G        => true,
         BUILD_INFO_G => BUILD_INFO_DEFAULT_SLV_C
      )
      port map (
         clk_i      => clk_i,
         rst_i      => rst_i,
         gtRx_i     => gtRx_i,
         gtTx_o     => gtTx_o
      );

   -----------------------------------------------------------
   -- 12x SFED link simulation
   -----------------------------------------------------------
   SFED_GEN : for i in N_BL_C*2-1 downto 0 generate  
      u0_MsgStreamTx : entity work.MsgStreamTx
         generic map (
            TPD_G  => TPD_C,
            SIZE_G => 8)
         port map (
            clk_i        => clk_i,
            rst_i        => rst_i,
            strobe_i     => strobe(i),
            msg32Array_i => sfedMsg32Array(i),
            full_i       => '0',
            axisMaster_o => sfedAxisMaster(i),
            axisSlave_i  => sfedAxisSlave(i),
            drop_o       => open);

      --! FiberLinkTx module to transmit the messages over GT fiber 
      u_FiberLinkMapTx : entity work.FiberLinkTx
         generic map (
            TPD_G        => TPD_C,
            DATA_WIDTH_G => 4,
            MAX_SIZE_G   => 8)
         port map (
            clk_i             => clk_i,
            rst_i             => rst_i,
            axisMaster_i(0)   => sfedAxisMaster(i),
            axisSlave_o(0)    => sfedAxisSlave(i),
            txPktFull_o       => open,
            txErr_o           => open,
            idle_o            => hpdIdle(i),
            dataTx_o          => gtRx_i(2+i).data(15 downto 8),
            charTx_o          => gtRx_i(2+i).char(1));
      
      --! driving Clock correction enable signal from Lpd and Hpd Idle signals
      gtRx_i(2+i).clkCorrEn         <= hpdIdle(i);
     --gtRx_i(2+i).cdrStable  <= '0';
   --------------------------------------------------------------
   end generate SFED_GEN;

   -----------------------------------------------------------
   -- ITU link simulation (CMU link module from the ITU)
   -----------------------------------------------------------
   u_ItuLink: entity work.SimItuLink
      generic map (
         TPD_G       => TPD_C,
         LINK_TIME_G => SIM_FIBER_LINK_TIME_C,
         WD_TIME_G   => SIM_SFED_WD_TIME_C)
      port map (
         clk_i      => clk_i,
         rst_i      => rst_i,
         gtRx_i     => gtTx_o(0),
         clear_i    => clearCmu,
         cmuStat_o  => ituStat,
         ituMsg_o   => ituMsg);
         
   -----------------------------------------------------------
   -- FEC link HPD simulation (CMU link module from the ITU)
   -----------------------------------------------------------
   u_FecHpdLink: entity work.SimFecLink
      generic map (
         TPD_G       => TPD_C,
         LINK_TIME_G => SIM_FIBER_LINK_TIME_C,
         WD_TIME_G   => SIM_SFED_WD_TIME_C)
      port map (
         clk_i      => clk_i,
         rst_i      => rst_i,
         cfgStrobe_i   => fecStrobe,
         cfgRequest_i  => fecConfigMsg,
         cfgResponse_o => fecConfigRsp,
         cfgDrop_o     => fecDrop,       
         --------------------------------
         gtTx_o     => gtTx_po,     
         gtRx_i     => gtTx_o(1),
         clear_i    => clearCmu,
         ituStat_o  => fecStat,
         ituMsg_o   => fecMsg);         
   
   -----------------------------
   -- Inject error into the conf. message 
   ----------------------------- 
   u0_InjectErrFiber : entity work.InjectErrFiber
      generic map (
         TPD_G      => TPD_C
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         en_i              => '1',
         injId_i           => '0',
         id_i              => x"00",
         injCrc_i          => '0',
         injLen_i          => '0',
         gtTxData_i        => gtTx_po.data(15 downto 8),
         gtTxChar_i        => gtTx_po.char(1),
         gtClkCorrEn_i     => gtTx_po.clkCorrEn,
         gtCorruptTxData_o => gtRx_i(1).data(15 downto 8),
         gtCorruptTxChar_o => gtRx_i(1).char(1),
         gtClkCorrEn_o     => gtRx_i(1).clkCorrEn
      );
   u1_InjectErrFiber : entity work.InjectErrFiber
      generic map (
         TPD_G      => TPD_C
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         en_i              => '1',
         injId_i           => injId,
         id_i              => cntID,
         injCrc_i          => injCrc,
         injLen_i          => injLen,
         gtTxData_i        => gtTx_po.data(7 downto 0),
         gtTxChar_i        => gtTx_po.char(0),
         gtClkCorrEn_i     => gtTx_po.clkCorrEn,
         gtCorruptTxData_o => gtRx_i(1).data(7 downto 0),
         gtCorruptTxChar_o => gtRx_i(1).char(0),
         gtClkCorrEn_o     => gtRx_i(1).clkCorrEn
      );   
   
   -----------------------------------------------------------
   -- Simulation procedure
   -----------------------------------------------------------
   sfedPeriod <= toSlv(SIM_ITU_PERIOD_C, bitsize(SIM_ITU_PERIOD_C));
   -- Proces for generatig input array message
   process(clk_i,rst_i)
   begin
      if rising_edge(clk_i) then
         if rst_i = '1' then
            sfedMsg32Array <= (others => (others => x"00000000"));
            tsCounter    <= (others => '0');
            strCounter   <= (others => '0');
         else
            if stall='0' then
               tsCounter <= tsCounter + 1;
            end if;
            strCounter <= strCounter + 1;
            
            if (strCounter = SIM_ITU_PERIOD_C-1) then
               ---------------------------------------------------------------------------------------------
               strCounter <=(others => '0');
               strobe <= (others =>'1');
               ---------------------------------------------------------------------------------------------
               for j in 0 to N_BL_C*2-1 loop
                  sfedMsg32Array(j)(0) <= x"0000003d";
                  for i in 1 to 4 loop
                     sfedMsg32Array(j)(i) <= slv(unsigned(sfedMsg32Array(j)(i)) + to_unsigned((j+i),32));
                  end loop;
                  sfedMsg32Array(j)(5) <= (others => '0');               
                  sfedMsg32Array(j)(6) <= slv(tsCounter(31 downto 0));
                  sfedMsg32Array(j)(7) <= slv(tsCounter(63 downto 32));
               end loop;            
               ---------------------------------------------------------------------------------------------
            else 
               strobe <= (others =>'0');            
            end if;
         end if;
      end if;
   end process;

   p_FEC_cfg_Process : process
   begin
   
      wait for 200 * CLK_PERIOD_C;
      ----------------------------------------
      wait for 500 * CLK_PERIOD_C;
      rst_i <= '0';
      wait for 1000 * CLK_PERIOD_C; 
      ----------------------------------------
      -- Stall the timestamp
      ----------------------------------------      
      stall <= '1';      
      wait for 1000 * CLK_PERIOD_C; 
      stall <= '0';  
      ----------------------------------------
      -- Clear the error counters
      ----------------------------------------      
      wait until clk_i = '1';
      clearCmu         <= '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"01000002";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010004";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      fecConfigMsg(8)  <= x"00000001";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"02000002";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010004";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      fecConfigMsg(8)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      clearCmu         <= '0';
      ----------------------------------------
      -- Reset the FSM
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------      
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"03000002";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010000";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      fecConfigMsg(8)  <= x"00000001";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"04000002";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010000";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      fecConfigMsg(8)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Inject the fiber error into SFED 12
      -- The error should be ignored by the FSM and the Concentrator since it is disabled.      
      ----------------------------------------      
      gtRx_i(13).dispErr <= '1';      
      wait for 1 * CLK_PERIOD_C; 
      gtRx_i(13).dispErr <= '0';
      ----------------------------------------
      wait for 200 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Clear the error counters
      ----------------------------------------      
      wait until clk_i = '1';
      clearCmu         <= '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"05000002";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010004";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      fecConfigMsg(8)  <= x"00000001";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"06000002";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010004";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      fecConfigMsg(8)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      clearCmu         <= '0';
      ----------------------------------------
      -- Enable all SFEDS
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------      
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"07000002";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010008";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      fecConfigMsg(8)  <= x"0000003f";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      -- Read all SFED enables
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"08000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00010008";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Read whole message 12
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"09000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00001B00";
      fecConfigMsg(3)  <= x"00000007";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';      
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Read a wrong address (adrErr)
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"0A000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"0001000C";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------      
      ----------------------------------------
      -- Read a wrong device (devErr)
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"0B000001";
      fecConfigMsg(1)  <= x"00000000";
      fecConfigMsg(2)  <= x"00001004";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Wrong command (cmdErr)
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"0C000000";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00001004";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------      
      ----------------------------------------
      -- Read whole message 8
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"0C000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00001700";
      fecConfigMsg(3)  <= x"00000007";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';      
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Inject the CRC into the CFG message
      ----------------------------------------
      wait until clk_i = '1';
      injCrc  <= '1';         
      wait until clk_i = '1';
      injCrc  <= '0';         
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------      
      ----------------------------------------
      -- Read STATE register -- This read should produce a crc error in the response
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"0D000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00000200";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';  
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Read whole message 11
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"0E000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00001A00";
      fecConfigMsg(3)  <= x"00000007";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';      
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Inject the Len into the CFG message
      ----------------------------------------
      wait until clk_i = '1';
      injLen  <= '1';         
      wait until clk_i = '1';
      injLen  <= '0';         
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------      
      ----------------------------------------
      -- Read STATE register -- This read should produce a len error in the response
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"0F000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00000200";
      fecConfigMsg(3)  <= x"00000001";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';  
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      -- Read whole message 12
      ----------------------------------------
      wait until clk_i = '1';
      fecStrobe        <= '1';
      fecConfigMsg(0)  <= x"10000001";
      fecConfigMsg(1)  <= x"00000001";
      fecConfigMsg(2)  <= x"00001B00";
      fecConfigMsg(3)  <= x"00000007";
      fecConfigMsg(4)  <= x"00000000";
      fecConfigMsg(5)  <= x"00000000";
      fecConfigMsg(6)  <= x"00000000";
      fecConfigMsg(7)  <= x"00000000";
      wait until clk_i = '1';
      fecStrobe        <= '0';      
      ----------------------------------------
      wait for 100 * CLK_PERIOD_C;
      ----------------------------------------
      ----------------------------------------
      wait for 1000 * CLK_PERIOD_C;
      ----------------------------------------
      assert false report "Simulation completed..." severity failure;
   end process p_FEC_cfg_Process;



end architecture Bhv;
---------------------------------------------------------------------------------------------------