---------------------------------------------------------------------------------------------------
--! @brief  GTY core wrapper module
--! @details
--!   Wrapper for GTY IP core is implemented in this module. Clock correction is also implemented
--!   in this module. Clock correction is inserted into data only if data is IDLE and on configured 
--!   period.
--!
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Oct 07 2019 last modify
--!
--! @version v0.1
--!
--! @file GtyCoreWrapper.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use work.CslStdRtlPkg.all;
use work.FiberPkg.all;
---------------------------------------------------------------------------------------------------
entity GtyCoreWrapper is
   generic (
      TPD_G             : time                     := 1 ns;
      CLK_CORR_2_G      : slv(8 downto 0)          := '1' & x"FD";   --! clock correction symbol 2
      CLK_CORR_1_G      : slv(8 downto 0)          := '1' & x"BC";   --! clock correction symbol 1
      CLK_CORR_PER_G    : positive                 := 2500;          --! clock correction period
      N_CHAN_G          : positive                 := 28; --22;            --! clock correction period
      HOLD_TIME_G       : positive                 := 1000           --! hold time for PowerupRst module
   );
   port (
      gtRst_i           : in sl;                --! GT reset bus
      --
      gtRefClk_i        : in slv(3 downto 0);   --! GT reference clocks
      gtFreeClk_i       : in sl;                --! GT free running clock
      --
      gtUsrClk_o        : out sl;               --! GT user clock
      gtUsrRst_o        : out sl;               --! GT user clock reset
      --
      gtCommAligEn_i    : in sl := '1';         --! GT comma alignement enable signal
      gtTxPol_i         : in sl := '0';         --! GT Tx polarity signal
      gtRxPol_i         : in sl := '0';         --! GT Rx polarity signal
      gtTxP_o           : out slv(N_CHAN_G-1 downto 0);  --! GT Serial Transmit Positive
      gtTxN_o           : out slv(N_CHAN_G-1 downto 0);  --! GT Serial Transmit Negative
      gtRxP_i           : in  slv(N_CHAN_G-1 downto 0);  --! GT Serial Receive Positive
      gtRxN_i           : in  slv(N_CHAN_G-1 downto 0);  --! GT Serial Receive Negative
      gtClkCorrPer_i    : in  slv(15 downto 0);                         --! GT clock correction period
      gtRx_o            : out GtArray(N_CHAN_G-1 downto 0)  := (others => GT_INIT_C); --! GT Rx 
      gtTx_i            : in  GtArray(N_CHAN_G-1 downto 0)  := (others => GT_INIT_C);  --! GT Tx 
      loopback_i        : in slv(N_CHAN_G*3-1 downto 0):= (others => '0')
      
   );
end GtyCoreWrapper;
---------------------------------------------------------------------------------------------------
architecture structure of GtyCoreWrapper is
   -- Differential swing constants from Arista User Guide
   
   --! 780mV
   constant TXDIFFCTRL_C : slv(4 downto 0)   := "10000";
   constant TXPRECURSOR_C : slv(4 downto 0)  := "00000";    
   constant TXPOSTCURSOR_C : slv(4 downto 0) := "00000";   
    
   -- User clock 
   signal txUsrClk      : sl; 
   signal txUsrRst      : sl;
   signal txUsrActive   : sl;
   
   -- Arrays
   signal txUsrClkArr      : slv(N_CHAN_G-1 downto 0);
   signal gtCommAligEnArr  : slv(N_CHAN_G-1 downto 0);   
   signal gtRxPolArr       : slv(N_CHAN_G-1 downto 0);
   signal gtTxPolArr       : slv(N_CHAN_G-1 downto 0);
   
   ----------------------------------------------------------   
   signal dispErr       : slv(16*N_CHAN_G-1 downto 0);
   signal outOfTable    : slv(8*N_CHAN_G-1 downto 0);
   signal rxkChar       : slv(16*N_CHAN_G-1 downto 0);
   signal txkChar       : slv(8*N_CHAN_G-1 downto 0):= (others => '0');
   signal txUserData    : slv(16*N_CHAN_G-1 downto 0);
   signal rxUserData    : slv(16*N_CHAN_G-1 downto 0);
   signal txDiffCtrl   : slv(N_CHAN_G*5-1 downto 0) := (others => '1');
   signal txPostCursor : slv(N_CHAN_G*5-1 downto 0) := (others => '0');
   signal txPreCursor  : slv(N_CHAN_G*5-1 downto 0) := (others => '0');
   
   signal gtRstDone     : sl;
   signal gtCdrStable   : sl;    
   signal gtPowerGood   : slv(N_CHAN_G-1 downto 0);
   signal pmaRxRstDone  : slv(N_CHAN_G-1 downto 0);
   signal pmaTxRstDone  : slv(N_CHAN_G-1 downto 0);   
  
   signal ByteAlig      : slv(N_CHAN_G-1 downto 0);
   signal ByteReAlig    : slv(N_CHAN_G-1 downto 0);
   signal buffStat      : slv(N_CHAN_G*3-1 downto 0);
   signal rxClkCorCnt   : slv(N_CHAN_G*2-1 downto 0);
   signal gtTxCor_po    : GtArray(N_CHAN_G*2-1 downto 0)  := (others => GT_INIT_C);
   
  constant  CLK_CORR1_PER_G : positive  := 500;    
   
   --! keep signals for debug in chipscope
   attribute keep : string;
   attribute keep of gtTxCor_po   : signal is "true";   
   attribute keep of buffStat     : signal is "true";
   attribute keep of rxClkCorCnt  : signal is "true";
   attribute keep of pmaRxRstDone : signal is "true";
   attribute keep of pmaTxRstDone : signal is "true";
   attribute keep of gtPowerGood  : signal is "true";
---------------------------------------------------------------------------------------------------
begin
   
   -----------------
   -- GTY TX mapping signals for trasmitting data
   -----------------   
   TX_LANES_GEN : for i in N_CHAN_G-1 downto 0 generate
      txUserData((i*16)+15 downto (i*16))   <= gtTxCor_po(i).data;
      txKchar((i*8)+7 downto (i*8))         <= x"0" & "00" & gtTxCor_po(i).char;
      txDiffCtrl((i*5)+4 downto i*5)   <= TXDIFFCTRL_C;
      txPostCursor((i*5)+4 downto i*5) <= TXPRECURSOR_C; 
      txPreCursor((i*5)+4 downto i*5)  <= TXPOSTCURSOR_C;
   end generate TX_LANES_GEN;
   
   -----------------
   -- GTY RX mapping signals for receiveing data
   ----------------- 
   RX_LANES_GEN : for i in N_CHAN_G-1 downto 0 generate
      gtRx_o(i).data          <= rxUserData((16*i)+15 downto 16*i);
      gtRx_o(i).char          <= rxKchar((16*i)+1 downto 16*i);
      gtRx_o(i).rstDone       <= gtRstDone;
      gtRx_o(i).dispErr       <= uOr(dispErr((16*i)+1 downto 16*i));
      gtRx_o(i).decErr        <= uOr(outOfTable((8*i)+1 downto 8*i));
      gtRx_o(i).byteAlig      <= ByteAlig(i);
      gtRx_o(i).byteReAlig    <= ByteReAlig(i);
      gtRx_o(i).cdrStable     <= gtCdrStable;
      
      --Arrays
      txUsrClkArr(i)       <= txUsrClk; 
      gtCommAligEnArr(i)   <= gtCommAligEn_i;
      gtRxPolArr(i)        <= gtRxPol_i;
      gtTxPolArr(i)        <= gtTxPol_i;
      
   end generate RX_LANES_GEN;   

   --! Clock Correction for all the modules
   TX_CORR_LANES_GEN : for i in 14-1 downto 0 generate
      u0_ClkCorrTx : entity work.ClkCorrTx
         generic map (
            CLK_CORR_2_G    => CLK_CORR_2_G,
            CLK_CORR_1_G    => CLK_CORR_1_G,
            CLK_CORR_PER_G  => CLK_CORR_PER_G
         )
         port map (
            clk_i          => txUsrClk,
            rst_i          => txUsrRst,
            en_i           => '1',
            idle_i         => gtTx_i(i).clkCorrEn,
            clkCorrPer_i   => gtClkCorrPer_i,
            gtTx_i         => gtTx_i(i),
            gtTx_o         => gtTxCor_po(i)
         );
   end generate TX_CORR_LANES_GEN;
   
      --! Clock Correction for all the modules
   TX_CORR1_LANES_GEN : for i in N_CHAN_G-1 downto 14 generate
      u0_ClkCorrTx : entity work.ClkCorrTx
         generic map (
            CLK_CORR_2_G    => CLK_CORR_2_G,
            CLK_CORR_1_G    => CLK_CORR_1_G,
            CLK_CORR_PER_G  => CLK_CORR1_PER_G
         )
         port map (
            clk_i          => txUsrClk,
            rst_i          => txUsrRst,
            en_i           => '1',
            idle_i         => gtTx_i(i).clkCorrEn,
            clkCorrPer_i   => gtClkCorrPer_i,
            gtTx_i         => gtTx_i(i),
            gtTx_o         => gtTxCor_po(i)
         );
   end generate TX_CORR1_LANES_GEN;
      
   --! GTY IP core instantiation
   u0_GtyCore : entity work.GtyCore
      port map (
         -- Tx clock
         gtwiz_userclk_tx_reset_in(0)    => '0',
         gtwiz_userclk_tx_srcclk_out     => open,
         gtwiz_userclk_tx_usrclk_out(0)  => txUsrClk,
         gtwiz_userclk_tx_usrclk2_out    => open,
         gtwiz_userclk_tx_active_out(0)  => txUsrActive,
         -- Rx clock
         gtwiz_userclk_rx_active_in(0)   => txUsrActive,
         rxusrclk_in                     => txUsrClkArr,
         rxusrclk2_in                    => txUsrClkArr,
         -- Reset
         gtwiz_reset_clk_freerun_in(0)         => gtFreeClk_i,
         gtwiz_reset_all_in(0)                 => gtRst_i,
         gtwiz_reset_tx_pll_and_datapath_in(0) => '0',
         gtwiz_reset_tx_datapath_in(0)         => '0',
         gtwiz_reset_rx_pll_and_datapath_in(0) => '0',
         gtwiz_reset_rx_datapath_in(0)         => '0',
         gtwiz_reset_rx_cdr_stable_out(0)      => gtCdrStable,
         gtwiz_reset_tx_done_out               => open,
         gtwiz_reset_rx_done_out(0)            => gtRstDone,
         -- Data 
         gtwiz_userdata_tx_in      => txUserData,
         gtwiz_userdata_rx_out     => rxUserData,
         -- Referenc clocks
         gtrefclk01_in(0)          => gtRefClk_i(0),
         gtrefclk01_in(1)          => gtRefClk_i(0),   
         gtrefclk01_in(2)          => gtRefClk_i(1),
         gtrefclk01_in(3)          => gtRefClk_i(1),   
         gtrefclk01_in(4)          => gtRefClk_i(2), 
         gtrefclk01_in(5)          => gtRefClk_i(2),   
         gtrefclk01_in(6)          => gtRefClk_i(3),
              
         qpll1outclk_out           => open,
         qpll1outrefclk_out        => open,
         
         -- Rx
         gtyrxn_in                 => gtRxP_i,
         gtyrxp_in                 => gtRxN_i,
         loopback_in               => loopback_i,
         rx8b10ben_in              => (others => '1'),
         rxbufreset_in             => (others => '0'),
         rxcommadeten_in           => (others => '1'),
         rxmcommaalignen_in        => gtCommAligEnArr,
         rxpcommaalignen_in        => gtCommAligEnArr,
         rxpolarity_in             => gtRxPolArr,

         -- Tx
         tx8b10ben_in              => (others => '1'),
         txctrl0_in                => (others => '0'),
         txctrl1_in                => (others => '0'),
         txctrl2_in                => txKchar,
         txdiffctrl_in             => txDiffCtrl,  
         txpolarity_in             => gtTxPolArr,
         txpostcursor_in           => txPostCursor,
         txprecursor_in            => txPreCursor,          
         gtpowergood_out           => gtPowerGood,         
         gtytxn_out                => gtTxP_o,
         gtytxp_out                => gtTxN_o, 
         
         -- Status
         rxbufstatus_out           => buffStat,
         rxbyteisaligned_out       => ByteAlig,
         rxbyterealign_out         => ByteReAlig,
         rxclkcorcnt_out           => rxClkCorCnt,
         rxcommadet_out            => open,         
         
         rxctrl0_out               => rxkChar,
         rxctrl1_out               => dispErr,
         rxctrl2_out               => open,
         rxctrl3_out               => outOfTable,          
         -- Status
         rxoutclk_out              => open,
         rxpmaresetdone_out        => pmaRxRstDone,
         rxprgdivresetdone_out     => open,
         txpmaresetdone_out        => pmaTxRstDone,
         txprgdivresetdone_out     => open
   );
   
      --! Synchronise the reset to the tx user clock.
   u0_PowerupRst : entity work.PowerupRst
      generic map (
         TPD_G         => TPD_G,
         INV_RST_POL_G => false,
         HOLD_TIME_G   => HOLD_TIME_G
      )
      port map (
         clk_i    => txUsrClk,
         extRst_i => gtRst_i,
         rst_o    => txUsrRst
      );

   -- Assign the tx user clock to the output
   gtUsrClk_o <= txUsrClk; 
   gtUsrRst_o <= txUsrRst;
   
end structure;
---------------------------------------------------------------------------------------------------