
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

package CslAxiPkg is

   -------------------------------------------------------------------------------------------------
   -- AXI bus response codes
   -------------------------------------------------------------------------------------------------
   constant AXI_RESP_OK_C : std_logic_vector(1 downto 0) := "00";  -- Access ok

   constant AXI_RESP_EXOKAY_C : std_logic_vector(1 downto 0) := "01";  -- Exclusive access ok
   -- Note: There are no "exclusive access" in AXI-Lite.  This is just a placeholder constant.

   constant AXI_RESP_SLVERR_C : std_logic_vector(1 downto 0) := "10";  -- Slave Error
   -- Note: A SLVERR response is returned to the master if the AXI peripheral interface receives any 
   --       of the following unsupported accesses:
   --
   --          1) Any accesses with AWSIZE information other than 32-bit receives a SLVERR response.
   --          2) Any accesses with AWLEN information other than zero receives a SLVERR response.
   --          3) Any access that is unaligned, for example, where AWADDRP[1:0] is not equal to 2'b00, 
   --             returns a SLVERR response where a read access returns all zeros and a write access 
   --             does not modify the address location.
   --          4) Any write access that attempts to make use of the WSTRB lines, 
   --             for example where any bits of WSTRB[3:0] are 0, returns a SLVERR response 
   --             and does not modify the address location.   

   constant AXI_RESP_DECERR_C : std_logic_vector(1 downto 0) := "11";  -- Decode Error
   -- Note: Any transaction that does not decode to a legal master interface destination, 
   --       or programmers view register, receives a DECERR response. For an AHB master, 
   --       the AXI DECERR is mapped back to an AHB ERROR.

--------------------------------------------------------------------------------
-- Axi Lite definitions
--------------------------------------------------------------------------------

   --------------------------------------------------------
   -- AXI bus, read master signal record
   --------------------------------------------------------

   -- Base Record
   type AxiLiteReadMasterType is record
      -- Read Address channel
      araddr  : std_logic_vector(31 downto 0);
      arprot  : std_logic_vector(2 downto 0);
      arvalid : std_logic;
      -- Read data channel
      rready  : std_logic;
   end record;

   -- Initialization constants
   constant AXI_LITE_READ_MASTER_INIT_C : AxiLiteReadMasterType := (
      araddr  => (others => '0'),
      arprot  => (others => '0'),
      arvalid => '0',
      rready  => '1'
      );

   -- Array
   type AxiLiteReadMasterArray is array (natural range<>) of AxiLiteReadMasterType;
   
   --------------------------------------------------------
   -- AXI bus, read slave signal record
   --------------------------------------------------------

   -- Base Record
   type AxiLiteReadSlaveType is record
      -- Read Address channel
      arready : std_logic;
      -- Read data channel
      rdata   : std_logic_vector(31 downto 0);
      rresp   : std_logic_vector(1 downto 0);
      rvalid  : std_logic;
   end record;

   -- Initialization constants
   constant AXI_LITE_READ_SLAVE_INIT_C : AxiLiteReadSlaveType := (
      arready => '0',
      rdata   => (others => '0'),
      rresp   => (others => '0'),
      rvalid  => '0'
      );

   -- Array
   type AxiLiteReadSlaveArray is array (natural range<>) of AxiLiteReadSlaveType;

   --------------------------------------------------------
   -- AXI bus, write master signal record
   --------------------------------------------------------

   -- Base Record
   type AxiLiteWriteMasterType is record
      -- Write address channel
      awaddr  : std_logic_vector(31 downto 0);
      awprot  : std_logic_vector(2 downto 0);
      awvalid : std_logic;
      -- Write data channel
      wdata   : std_logic_vector(31 downto 0);
      wstrb   : std_logic_vector(3 downto 0);
      wvalid  : std_logic;
      -- Write ack channel
      bready  : std_logic;
   end record;

   -- Initialization constants
   constant AXI_LITE_WRITE_MASTER_INIT_C : AxiLiteWriteMasterType := (
      awaddr  => (others => '0'),
      awprot  => (others => '0'),
      awvalid => '0',
      wdata   => (others => '0'),
      wstrb   => (others => '1'),
      wvalid  => '0',
      bready  => '1'
      );

   -- Array
   type AxiLiteWriteMasterArray is array (natural range<>) of AxiLiteWriteMasterType;
   --------------------------------------------------------
   -- AXI bus, write slave signal record
   --------------------------------------------------------

   -- Base Record
   type AxiLiteWriteSlaveType is record
      -- Write address channel
      awready : std_logic;
      -- Write data channel
      wready  : std_logic;
      -- Write ack channel
      bresp   : std_logic_vector(1 downto 0);
      bvalid  : std_logic;

   end record;

   -- Initialization constants
   constant AXI_LITE_WRITE_SLAVE_INIT_C : AxiLiteWriteSlaveType := (
      awready => '0',
      wready  => '0',
      bresp   => (others => '0'),
      bvalid  => '0'
      );

   -- Array
   type AxiLiteWriteSlaveArray is array (natural range<>) of AxiLiteWriteSlaveType;

   ----------------------------------------------------------------------------------
   -- Constants for endpoint abstractions (migrated from legacy AxiLiteMasterPkg.vhd)
   ----------------------------------------------------------------------------------
   type AxiLiteReqType is record
      request   : sl;
      rnw     : sl;
      address : slv(31 downto 0);
      wrData  : slv(31 downto 0);
   end record AxiLiteReqType;

   constant AXI_LITE_REQ_INIT_C : AxiLiteReqType := (
      request   => '0',
      rnw     => '1',
      address => (others => '0'),
      wrData  => (others => '0'));
   type AxiLiteAckType is record
      done  : sl;
      resp   : slv(1 downto 0);
      rdData : slv(31 downto 0);
   end record AxiLiteAckType;

   constant AXI_LITE_ACK_INIT_C : AxiLiteAckType := (
      done  => '0',
      resp   => (others => '0'),
      rdData => (others => '0'));     
   
--------------------------------------------------------------------------------
-- Axi4 defintions
--------------------------------------------------------------------------------
   -------------------------------------
   -- AXI bus, read master signal record
   -------------------------------------
   type AxiReadMasterType is record
      -- Read Address channel
      arvalid  : sl;                    -- Address valid
      araddr   : slv(63 downto 0);      -- Address
      arid     : slv(31 downto 0);      -- Address ID
      arlen    : slv(7 downto 0);       -- Transfer count
      arsize   : slv(2 downto 0);       -- Bytes per transfer
      arburst  : slv(1 downto 0);       -- Burst Type
      arlock   : slv(1 downto 0);       -- Lock control
      arprot   : slv(2 downto 0);       -- Protection control
      arcache  : slv(3 downto 0);       -- Cache control
      arqos    : slv(3 downto 0);       -- QoS value                                        
      arregion : slv(3 downto 0);       -- Region identifier                                       
      -- Read data channel
      rready   : sl;                    -- Master is ready for data
   end record;

   type AxiReadMasterArray is array (natural range<>) of AxiReadMasterType;

   constant AXI_READ_MASTER_INIT_C : AxiReadMasterType := (
      arvalid  => '0',
      araddr   => (others => '0'),
      arid     => (others => '0'),
      arlen    => (others => '0'),
      arsize   => (others => '0'),
      arburst  => (others => '0'),
      arlock   => (others => '0'),
      arprot   => (others => '0'),
      arcache  => (others => '0'),
      arqos    => (others => '0'),
      arregion => (others => '0'),
      rready   => '0');

   ------------------------------------
   -- AXI bus, read slave signal record
   ------------------------------------
   type AxiReadSlaveType is record
      -- Read Address channel
      arready : sl;                     -- Slave is ready for address
      -- Read data channel
      rdata   : slv(1023 downto 0);     -- Read data from slave
      rlast   : sl;                     -- Read data last strobe
      rvalid  : sl;                     -- Read data is valid
      rid     : slv(31 downto 0);       -- Read ID tag
      rresp   : slv(1 downto 0);        -- Read data result
   end record;

   type AxiReadSlaveArray is array (natural range<>) of AxiReadSlaveType;

   constant AXI_READ_SLAVE_INIT_C : AxiReadSlaveType := (
      arready => '0',
      rdata   => (others => '0'),
      rlast   => '0',
      rvalid  => '0',
      rid     => (others => '0'),
      rresp   => (others => '0'));

   --------------------------------------
   -- AXI bus, write master signal record
   --------------------------------------
   type AxiWriteMasterType is record
      -- Write address channel
      awvalid  : sl;                    -- Address valid
      awaddr   : slv(63 downto 0);      -- Address
      awid     : slv(31 downto 0);      -- Address ID
      awlen    : slv(7 downto 0);       -- Transfer count (burst length)
      awsize   : slv(2 downto 0);       -- Bytes per transfer
      awburst  : slv(1 downto 0);       -- Burst Type
      awlock   : slv(1 downto 0);       -- Lock control
      awprot   : slv(2 downto 0);       -- Protection control
      awcache  : slv(3 downto 0);       -- Cache control
      awqos    : slv(3 downto 0);       -- QoS value                                        
      awregion : slv(3 downto 0);       -- Region identifier                                       
      -- Write data channel
      wdata    : slv(1023 downto 0);    -- Write data
      wlast    : sl;                    -- Write data is last
      wvalid   : sl;                    -- Write data is valid
      wid      : slv(31 downto 0);      -- Write ID tag
      wstrb    : slv(127 downto 0);     -- Write enable strobes, 1 per byte
      -- Write ack channel
      bready   : sl;                    -- Write master is ready for status
   end record;

   type AxiWriteMasterArray is array (natural range<>) of AxiWriteMasterType;

   constant AXI_WRITE_MASTER_INIT_C : AxiWriteMasterType := (
      awvalid  => '0',
      awaddr   => (others => '0'),
      awid     => (others => '0'),
      awlen    => (others => '0'),
      awsize   => (others => '0'),
      awburst  => (others => '0'),
      awlock   => (others => '0'),
      awprot   => (others => '0'),
      awcache  => (others => '0'),
      awqos    => (others => '0'),
      awregion => (others => '0'),
      wdata    => (others => '0'),
      wlast    => '0',
      wvalid   => '0',
      wid      => (others => '0'),
      wstrb    => (others => '1'),
      bready   => '0');

   -------------------------------------
   -- AXI bus, write slave signal record
   -------------------------------------
   type AxiWriteSlaveType is record
      -- Write address channel
      awready : sl;                     -- Write slave is ready for address
      -- Write data channel
      wready  : sl;                     -- Write slave is ready for data
      -- Write ack channel
      bresp   : slv(1 downto 0);        -- Write access status
      bvalid  : sl;                     -- Write status valid
      bid     : slv(31 downto 0);       -- Channel ID
   end record;
   
   type AxiWriteSlaveArray is array (natural range<>) of AxiWriteSlaveType;
   
   constant AXI_WRITE_SLAVE_INIT_C : AxiWriteSlaveType := (
      awready => '0',
      wready  => '0',
      bresp   => (others => '0'),
      bvalid  => '0',
      bid     => (others => '0'));


--------------------------------------------------------------------------------
-- Axi Stream defintions
--------------------------------------------------------------------------------
   constant AXI_STREAM_MAX_TDATA_WIDTH_C : positive := 512;  -- Units of bits
   constant AXI_STREAM_MAX_TKEEP_WIDTH_C : positive := (AXI_STREAM_MAX_TDATA_WIDTH_C/8);  -- Units of bytes

   type AxiStreamMasterType is record
      tValid : sl;
      tData  : slv(AXI_STREAM_MAX_TDATA_WIDTH_C-1 downto 0);
      tStrb  : slv(AXI_STREAM_MAX_TKEEP_WIDTH_C-1 downto 0);
      tKeep  : slv(AXI_STREAM_MAX_TKEEP_WIDTH_C-1 downto 0);
      tLast  : sl;
      tDest  : slv(7 downto 0);
      tId    : slv(7 downto 0);
      tUser  : slv(AXI_STREAM_MAX_TDATA_WIDTH_C-1 downto 0);
   end record AxiStreamMasterType;

   constant AXI_STREAM_MASTER_INIT_C : AxiStreamMasterType := (
      tValid => '0',
      tData  => (others => '0'),
      tStrb  => (others => '1'),
      tKeep  => (others => '1'),
      tLast  => '0',
      tDest  => (others => '0'),
      tId    => (others => '0'),
      tUser  => (others => '0'));
   
   type AxiStreamMasterArray is array (natural range<>) of AxiStreamMasterType;

   type AxiStreamSlaveType is record
      tReady : std_logic;
   end record AxiStreamSlaveType;
   
   constant AXI_STREAM_SLAVE_INIT_C : AxiStreamSlaveType := (tReady => '0');

   constant AXI_STREAM_SLAVE_FORCE_C : AxiStreamSlaveType := (tReady => '1');

   type AxiStreamSlaveArray is array (natural range<>) of AxiStreamSlaveType;

    ------------------------
    -- AXI bus configuration
    ------------------------
    type AxiConfigType is record
       ADDR_WIDTH_C : positive range 12 to 64;
       DATA_BYTES_C : positive range 1 to 128;
       ID_BITS_C    : positive range 1 to 32;
       LEN_BITS_C   : natural range 0 to 8;
    end record AxiConfigType;
     
    type AxiLiteStatusType is record
       writeEnable : sl;
       readEnable  : sl;
    end record AxiLiteStatusType;
 
    constant AXI_LITE_STATUS_INIT_C : AxiLiteStatusType := (
       writeEnable => '0',
  readEnable => '0');
    --------------------------------------------------------
    -- AXI bus, read/write endpoint record, RTH 1/27/2016
    --------------------------------------------------------
    type AxiLiteEndpointType is record
       axiReadMaster  : AxiLiteReadMasterType;
       axiReadSlave   : AxiLiteReadSlaveType;
       axiWriteMaster : AxiLiteWriteMasterType;
       axiWriteSlave  : AxiLiteWriteSlaveType;
       axiStatus      : AxiLiteStatusType;
    end record AxiLiteEndpointType;
 
    constant AXI_LITE_ENDPOINT_INIT_C : AxiLiteEndpointType := (
       axiReadMaster  => AXI_LITE_READ_MASTER_INIT_C,
       axiReadSlave   => AXI_LITE_READ_SLAVE_INIT_C,
       axiWriteMaster => AXI_LITE_WRITE_MASTER_INIT_C,
       axiWriteSlave  => AXI_LITE_WRITE_SLAVE_INIT_C,
       axiStatus      => AXI_LITE_STATUS_INIT_C);
     

 
    function axiConfig (
       constant ADDR_WIDTH_C : in positive range 12 to 64 := 32;
       constant DATA_BYTES_C : in positive range 1 to 128 := 4;
       constant ID_BITS_C    : in positive range 1 to 32  := 12;
       constant LEN_BITS_C   : in natural range 0 to 8   := 4)
       return AxiConfigType;
 
    constant AXI_CONFIG_INIT_C : AxiConfigType := axiConfig(
       ADDR_WIDTH_C => 32,
       DATA_BYTES_C => 4,
       ID_BITS_C    => 12,
       LEN_BITS_C => 4);

    function getAxiLen (
       axiConfig  : AxiConfigType;
       burstBytes : integer range 1 to 4096 := 4096)
    return slv;
    
      -------------------------------------------------------------------------------------------------
    -- Slave AXI Processing procedures
    -------------------------------------------------------------------------------------------------
 
    procedure axiSlaveWaitWriteTxn (
       signal axiWriteMaster  : in    AxiLiteWriteMasterType;
       variable axiWriteSlave : inout AxiLiteWriteSlaveType;
       variable writeEnable   : inout sl);
 
    procedure axiSlaveWaitReadTxn (
       signal axiReadMaster  : in    AxiLiteReadMasterType;
       variable axiReadSlave : inout AxiLiteReadSlaveType;
       variable readEnable   : inout sl);
 
    procedure axiSlaveWaitTxn (
       signal axiWriteMaster  : in    AxiLiteWriteMasterType;
       signal axiReadMaster   : in    AxiLiteReadMasterType;
       variable axiWriteSlave : inout AxiLiteWriteSlaveType;
       variable axiReadSlave  : inout AxiLiteReadSlaveType;
       variable axiStatus     : inout AxiLiteStatusType);
 
    procedure axiSlaveWriteResponse (
       variable axiWriteSlave : inout AxiLiteWriteSlaveType;
       axiResp                : in    slv(1 downto 0) := AXI_RESP_OK_C);
 
    procedure axiSlaveReadResponse (
       variable axiReadSlave : inout AxiLiteReadSlaveType;
       axiResp : in slv(1 downto 0) := AXI_RESP_OK_C);
    
end CslAxiPkg;

package body CslAxiPkg is

    function axiConfig (
       constant ADDR_WIDTH_C : in positive range 12 to 64 := 32;
       constant DATA_BYTES_C : in positive range 1 to 128 := 4;
       constant ID_BITS_C    : in positive range 1 to 32  := 12;
       constant LEN_BITS_C   : in natural range 0 to 8   := 4)
       return AxiConfigType is
       variable ret : AxiConfigType;
    begin
       ret := (
          ADDR_WIDTH_C => ADDR_WIDTH_C,
          DATA_BYTES_C => DATA_BYTES_C,
          ID_BITS_C    => ID_BITS_C,
          LEN_BITS_C   => LEN_BITS_C);
       return ret;
  end function axiConfig;
  
    function getAxiLen (
     axiConfig  : AxiConfigType;
     burstBytes : integer range 1 to 4096 := 4096)
     return slv is
  begin
     -- burstBytes / data bytes width is number of txns required.
     -- Subtract by 1 for A*LEN value for even divides.
     -- Convert to SLV and truncate to size of A*LEN port for this AXI bus
     -- This limits number of txns approraiately based on size of len port
     -- Then resize to 8 bits because our records define A*LEN as 8 bits always.
     return resize(toSlv(wordCount(burstBytes,axiConfig.DATA_BYTES_C)-1, axiConfig.LEN_BITS_C), 8);
  end function getAxiLen;

   procedure axiSlaveWaitWriteTxn (
      signal axiWriteMaster  : in    AxiLiteWriteMasterType;
      variable axiWriteSlave : inout AxiLiteWriteSlaveType;
      variable writeEnable   : inout sl) is
   begin
      ----------------------------------------------------------------------------------------------
      -- AXI Write Logic
      ----------------------------------------------------------------------------------------------
      writeEnable := '0';

      axiWriteSlave.awready := '0';
      axiWriteSlave.wready  := '0';

      -- Incomming Write txn and last txn has concluded
      if (axiWriteMaster.awvalid = '1' and axiWriteMaster.wvalid = '1' and axiWriteSlave.bvalid = '0') then
         writeEnable := '1';
      end if;

      -- Reset resp valid
      if (axiWriteMaster.bready = '1') then
         axiWriteSlave.bvalid := '0';
      end if;
   end procedure;

   procedure axiSlaveWaitReadTxn (
      signal axiReadMaster  : in    AxiLiteReadMasterType;
      variable axiReadSlave : inout AxiLiteReadSlaveType;
      variable readEnable   : inout sl) is
   begin
      ----------------------------------------------------------------------------------------------
      -- AXI Read Logic
      ----------------------------------------------------------------------------------------------
      readEnable := '0';

      axiReadSlave.arready := '0';

      -- Incomming read txn and last txn has concluded
      if (axiReadMaster.arvalid = '1' and axiReadSlave.rvalid = '0') then
         readEnable := '1';
      end if;

      -- Reset rvalid upon rready
      if (axiReadMaster.rready = '1') then
         axiReadSlave.rvalid := '0';
         axiReadSlave.rdata  := (others => '0');
      end if;
   end procedure axiSlaveWaitReadTxn;

   procedure axiSlaveWaitTxn (
      signal axiWriteMaster  : in    AxiLiteWriteMasterType;
      signal axiReadMaster   : in    AxiLiteReadMasterType;
      variable axiWriteSlave : inout AxiLiteWriteSlaveType;
      variable axiReadSlave  : inout AxiLiteReadSlaveType;
      variable axiStatus     : inout AxiLiteStatusType) is
   begin
      axiSlaveWaitWriteTxn(axiWriteMaster, axiWriteSlave, axiStatus.writeEnable);
      axiSlaveWaitReadTxn(axiReadMaster, axiReadSlave, axiStatus.readEnable);
   end procedure;

   procedure axiSlaveWriteResponse (
      variable axiWriteSlave : inout AxiLiteWriteSlaveType;
      axiResp                : in    slv(1 downto 0) := AXI_RESP_OK_C) is
   begin
      axiWriteSlave.awready := '1';
      axiWriteSlave.wready  := '1';
      axiWriteSlave.bvalid  := '1';
      axiWriteSlave.bresp   := axiResp;
   end procedure;

   procedure axiSlaveReadResponse (
      variable axiReadSlave : inout AxiLiteReadSlaveType;
      axiResp               : in    slv(1 downto 0) := AXI_RESP_OK_C) is
   begin
      axiReadSlave.arready := '1';      -- not sure this is necessary
      axiReadSlave.rvalid  := '1';
      axiReadSlave.rresp   := axiResp;
   end procedure;

end package body CslAxiPkg;


