---------------------------------------------------------------------------------------------------
--! @brief Cosylab Axi Stream Asynchronous Asymmetric module
--! @details 
--!    The module is capable of resizing Axi Stream channels. The module is also used for clock
--!    domain crossing of Axi Stream protocol and as FIFO. With generic THRESHOLD_G output signal
--!    pause_o is configured. Signal pause_o is '1' when fifo fullness (word count) exceeds THRESHOLD_G.
--!    This does not stop fifo process and only serves as feedback to the user.
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Feb 26 2019 created
--! @date Mar 28 2018 last modify
--!
--! @version 1.0
--!
--! @file CslAxisFifo.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.CslDmaTypePkg.all;
---------------------------------------------------------------------------------------------------
entity CslAxisFifo is
   generic (
      -- General Configurations
      TPD_G          : time      := 1 ns; --! time propagation delay
      PIPELINE_G     : natural   := 1;    --! output data delay in clock cycles
      BRAM_G         : boolean   := true; --! boolean for FIFO resources false => distributed LUT
      AWIDTH_G       : positive  := 8;    --! fifo address width
      THRESHOLD_G    : positive  := 128;  --! threshold for pause signal
      TUSER_NORMAL_G : boolean   := true; --! tuser MODE. Normal => all user bits, Not normal => last user bits
      TID_WIDTH_G    : positive  := 1;    --! width of TID vector
      TDEST_WIDTH_G  : positive  := 1;    --! width of TDEST vector
      TUSER_WIDTH_G  : positive  := 1;    --! width of TUSER vector per byte
      BYTES_SLV_G    : positive  := 8;    --! nubmer of bytes in slave port
      BYTES_MST_G    : positive  := 4);   --! number of bytes in master port
   port (
      -- Slave Port
      sAxisClk_i     : in sl;                    --! axis slave clock signal bus
      sAxisRst_i     : in sl;                    --! axis slave reset signal bus
      sAxisMaster_i  : in  AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;  --! axis slave input signals
      sAxisSlave_o   : out AxiStreamSlaveType;   --! axis slave output signals

      -- Master Port
      mAxisClk_i     : in sl;                    --! axis master clock signal bus
      mAxisRst_i     : in sl;                    --! axis master reset signal bus
      mAxisMaster_o  : out AxiStreamMasterType;  --! axis master output signals
      mAxisSlave_i   : in  AxiStreamSlaveType := AXI_STREAM_SLAVE_INIT_C;   --! axis master input signals
      
      wrCnt_o        : out slv(31 downto 0);     --! word count in fifo
      pause_o        : out sl                    --! axis pause signal => fifo fullness exceeds THRESHOLD_G value
   );
end CslAxisFifo;
---------------------------------------------------------------------------------------------------    
architecture rtl of CslAxisFifo is

   --! assign wider port to constant
   constant BYTE_MAX_C        : positive  := ite(BYTES_SLV_G > BYTES_MST_G, BYTES_SLV_G, BYTES_MST_G);  
   --! data width of fifo
   constant DWIDTH_C          : positive  := 8*BYTE_MAX_C + BYTE_MAX_C + BYTE_MAX_C + 1 + TID_WIDTH_G 
                                             + TDEST_WIDTH_G + 
                                             ite(TUSER_NORMAL_G, TUSER_WIDTH_G*BYTE_MAX_C, TUSER_WIDTH_G);
   
   --! internal registers Slave/Write side of FIFO
   signal   fifoWinc          : sl;
   signal   fifoWfull         : sl;
   signal   fifoWdata         : slv(DWIDTH_C-1 downto 0);
   signal   fifoWriteMaster   : AxiStreamMasterType         := AXI_STREAM_MASTER_INIT_C;
   signal   fifoWriteSlave    : AxiStreamSlaveType          := AXI_STREAM_SLAVE_INIT_C;  
   signal   fifoWriteGetTkeep : positive;
   
   --! internal registers Master/Read side of FIFO
   signal   fifoRinc          : sl;
   signal   fifoRempty        : sl;
   signal   fifoRdata         : slv(DWIDTH_C-1 downto 0);
   signal   fifoRcnt          : slv(AWIDTH_G downto 0);
   signal   fifoReadMaster    : AxiStreamMasterType         := AXI_STREAM_MASTER_INIT_C;
   signal   fifoReadSlave     : AxiStreamSlaveType          := AXI_STREAM_SLAVE_INIT_C;
   signal   fifoReadGetTkeep  : positive;
   
   --! tReady/tValid adapter for pipeline addjustment
   signal   rData             : slv(DWIDTH_C-1 downto 0);
---------------------------------------------------------------------------------------------------
begin
   
   -- Instantiate the CslAxisResize module
   u0_CslAxisResize : entity work.CslAxisResize
      generic map (
         TPD_G          => TPD_G,              
         TID_WIDTH_G    => TID_WIDTH_G,    --! width of TID vector
         TDEST_WIDTH_G  => TDEST_WIDTH_G,  --! width of TDEST vector
         TUSER_WIDTH_G  => TUSER_WIDTH_G,  --! width of TUSER vector per byte
         BYTES_SLV_G    => BYTES_SLV_G,    --! nubmer of bytes in slave port
         BYTES_MST_G    => BYTE_MAX_C)     --! number of bytes in master port
      port map (
         clk_i          => sAxisClk_i,        --! clock signal bus
         rst_i          => sAxisRst_i,        --! reset signal bus
         sAxisMaster_i  => sAxisMaster_i,     --! slave axis input signals
         sAxisSlave_o   => sAxisSlave_o,      --! slave axis output signals
         mAxisMaster_o  => fifoWriteMaster,   --! master axis output signals
         mAxisSlave_i   => fifoWriteSlave);   --! master axis input signals
   
   --! Parsing Axi Stream to FIFO write data      
   fifoWriteSlave.tReady   <= not(fifoWfull);      
           
   fifoWinc                <= not(fifoWfull) and fifoWriteMaster.tValid;
   
   fifoWdata(DWIDTH_C-1)                              <= fifoWriteMaster.tLast;
   
   fifoWdata(BYTE_MAX_C - 1 downto 0)                 <= fifoWriteMaster.tStrb(BYTE_MAX_C - 1 downto 0);
   
   fifoWdata(2*BYTE_MAX_C - 1 downto BYTE_MAX_C)      <= fifoWriteMaster.tKeep(BYTE_MAX_C - 1 downto 0);
   
   fifoWdata(TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto 2*BYTE_MAX_C)               
                                                      <= fifoWriteMaster.tDest(TDEST_WIDTH_G - 1 downto 0);
                                                      
   fifoWdata(TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto TDEST_WIDTH_G+2*BYTE_MAX_C)    
                                                      <= fifoWriteMaster.tId(TID_WIDTH_G - 1 downto 0);
                                                      
   fifoWdata(BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C)    
                                                      <= fifoWriteMaster.tData(BYTE_MAX_C*8 - 1 downto 0);
   GEN_WR_TUSER: if TUSER_NORMAL_G = true  generate
      fifoWdata(BYTE_MAX_C*TUSER_WIDTH_G+BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C)    
                                                      <= fifoWriteMaster.tUser(BYTE_MAX_C*TUSER_WIDTH_G - 1 downto 0);
   end generate GEN_WR_TUSER;
   
   GEN_WR_TUSER_LAST: if TUSER_NORMAL_G = false  generate   
      --! @brief Combinational process
      --! @details Main module logic. Generates rin based on r and any module inputs
      parseWr : process (fifoWriteMaster) is
         variable retVar    : natural;
         variable i         : natural;
      begin
         retVar    := 0;
         for i in 0 to BYTE_MAX_C - 1 loop
            if (fifoWriteMaster.tKeep(i) = '1') then
               retVar := (i);
            end if;
         end loop;
         fifoWriteGetTkeep <= retVar + 1;
      end process parseWr; 

      fifoWdata(TUSER_WIDTH_G+BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C)    
                                                      <= fifoWriteMaster.tUser(fifoWriteGetTkeep*TUSER_WIDTH_G - 1 downto (fifoWriteGetTkeep-1)*TUSER_WIDTH_G);
   end generate GEN_WR_TUSER_LAST;
  
  -- Instantiate the CslAsyncFIFO module
   u0_CslAsyncFIFO : entity work.CslAsyncFIFO
      generic map (
         TPD_G          => TPD_G,
         AWIDTH_G       => AWIDTH_G,   --! address width
         DWIDTH_G       => DWIDTH_C,   --! data width
         PIPELINE_G     => PIPELINE_G, --! output data delay in clock cycles
         WRITE_FIRST_G  => true,       --! write first/read first
         BRAM_G         => BRAM_G)     --! if false distributed LUT are used
      port map (
         wclk_i         => sAxisClk_i, --! write local clock bus          
         wrst_i         => sAxisRst_i, --! write reset signal bus          
         winc_i         => fifoWinc,   --! write pointer increment        
         wdata_i        => fifoWdata,  --! write data signal vector bus         
         wfull_o        => fifoWfull,  --! FIFO is full => WR pointer wraped one more time than WR pointer         
         rclk_i         => mAxisClk_i, --! read local clock bus          
         rrst_i         => mAxisRst_i, --! read reset signal bus           
         rinc_i         => fifoRinc,   --! read pointer increment        
         rdata_o        => fifoRdata,  --! read data signal vector bus         
         rempty_o       => fifoRempty, --! FIFO is empty => RD pointer caches WR pointer          
         flush_i        => '0',        --! empty FIFO and reset all counters   
         rcnt_o         => fifoRcnt,   --! fifo fullness counter         
         unf_o          => open,       --! underflow flag signal    
         ovf_o          => open        --! overflow flag signal   
      );
      
   pause_o                           <= '1' when unsigned(fifoRcnt) > to_unsigned(THRESHOLD_G,AWIDTH_G+1) else '0';
   wrCnt_o(AWIDTH_G downto 0)        <= fifoRcnt;
   wrCnt_o(31 downto AWIDTH_G + 1)   <= (others => '0');

   -- Instantiate the CslStreamFifoReadAdapter module
   u0_CslStreamFifoReadAdapter : entity work.CslStreamFifoReadAdapter
      generic map (
         TPD_G      => TPD_G,
         DWIDTH_G   => DWIDTH_C,
         PIPELINE_G => PIPELINE_G)
      port map (
         clk_i    => mAxisClk_i,
         rst_i    => mAxisRst_i,
         rinc_o   => fifoRinc,
         rdata_i  => fifoRdata,
         rempty_i => fifoRempty,
         ready_i  => fifoReadSlave.tReady,
         valid_o  => fifoReadMaster.tValid,
         data_o   => rData
      );

   --! Parsing Axi Stream from FIFO read data 
   fifoReadMaster.tLast    <= rData(DWIDTH_C-1);
   
   fifoReadMaster.tStrb(fifoReadMaster.tStrb'high downto BYTE_MAX_C)    
      <= (others => '0');
   fifoReadMaster.tStrb(BYTE_MAX_C - 1 downto 0)    
      <= rData(BYTE_MAX_C - 1 downto 0);
   
   fifoReadMaster.tKeep(fifoReadMaster.tKeep'high downto BYTE_MAX_C)    
      <= (others => '0');
   fifoReadMaster.tKeep(BYTE_MAX_C - 1 downto 0)      
      <= rData(2*BYTE_MAX_C - 1 downto BYTE_MAX_C);
   
   fifoReadMaster.tDest(TDEST_WIDTH_G - 1 downto 0)   
      <= rData(TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto 2*BYTE_MAX_C);
   
   fifoReadMaster.tId(TID_WIDTH_G - 1 downto 0)       
      <= rData(TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto TDEST_WIDTH_G+2*BYTE_MAX_C); 
                                                      
   fifoReadMaster.tData(BYTE_MAX_C*8 - 1 downto 0)    
      <= rData(BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C); 
     
   GEN_RD_TUSER: if TUSER_NORMAL_G = true  generate
      fifoReadMaster.tUser(BYTE_MAX_C*TUSER_WIDTH_G - 1 downto 0)    
         <= rData(BYTE_MAX_C*TUSER_WIDTH_G+BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C); 
   end generate GEN_RD_TUSER;
   
   GEN_RD_TUSER_LAST: if TUSER_NORMAL_G = false  generate
      parseRd : process (rData, fifoReadMaster) is
         variable retVar    : natural;
         variable i         : natural;
      begin
         retVar := 0;
         for i in 0 to BYTE_MAX_C - 1 loop
            if (fifoReadMaster.tKeep(i) = '1') then
               retVar := (i);
            end if;
         end loop;

         for i in 0 to BYTE_MAX_C - 1 loop
            if (retVar = i) then
               fifoReadMaster.tUser((i)*TUSER_WIDTH_G + TUSER_WIDTH_G - 1 downto i*TUSER_WIDTH_G)    
                  <= rData(TUSER_WIDTH_G+BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C - 1 downto BYTE_MAX_C*8+TID_WIDTH_G+TDEST_WIDTH_G+2*BYTE_MAX_C); 
            else
               fifoReadMaster.tUser((i)*TUSER_WIDTH_G + TUSER_WIDTH_G - 1 downto i*TUSER_WIDTH_G) 
                  <= (others => '0');
            end if;
         end loop;

         fifoReadMaster.tUser(fifoReadMaster.tUser'high downto (BYTE_MAX_C)*TUSER_WIDTH_G) <= (others => '0');
      end process parseRd;
   end generate GEN_RD_TUSER_LAST;
   
   -- Instantiate the CslAxisResize module
   u1_CslAxisResize : entity work.CslAxisResize
      generic map (
         TPD_G          => TPD_G,
         TID_WIDTH_G    => TID_WIDTH_G,     --! width of TID vector
         TDEST_WIDTH_G  => TDEST_WIDTH_G,   --! width of TDEST vector
         TUSER_WIDTH_G  => TUSER_WIDTH_G,   --! width of TUSER vector per byte
         BYTES_SLV_G    => BYTE_MAX_C,      --! nubmer of bytes in slave port
         BYTES_MST_G    => BYTES_MST_G)     --! number of bytes in master port
      port map (
         clk_i          => mAxisClk_i,      --! clock signal bus
         rst_i          => mAxisRst_i,      --! reset signal bus
         sAxisMaster_i  => fifoReadMaster,  --! slave axis input signals
         sAxisSlave_o   => fifoReadSlave,   --! slave axis output signals
         mAxisMaster_o  => mAxisMaster_o,   --! master axis output signals
         mAxisSlave_i   => mAxisSlave_i);   --! master axis input signals
   
end rtl;
---------------------------------------------------------------------------------------------------