---------------------------------------------------------------------------------------------------
--! @brief  CslAxiToAxilBlock testbench
--! @details  Testbench employs BRAM with axiLite interface. The tesbench performs three axi4 writes 
--!           to RAM with lengths 256, 16 and 1 transaction. Data is read back after every transaction and is compared to 
--!           written data. If data matches, transaction was successfull. Both axi4 and axiLite interface are connected to 
--!           Xilinx axi protocol checker to ensure interfaces are according to the axi specifications.
--!
--! @author Jost Reberc(jost.reebrc@cosylab.com)
--!
--! @date March 18 2019 created
--!
--! @file CslAxiToAxilBlockTb.vhd
---------------------------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.numeric_std.all; 
use ieee.std_logic_unsigned.all; 
use work.CslStdRtlPkg.all; 
use work.CslAxiPkg.all; 
use work.CslTbPkg.all; 
use work.CslPrintfPkg.all; 
---------------------------------------------------------------------------------------------------
entity CslAxiToAxilBlockTb is
end CslAxiToAxilBlockTb; 
---------------------------------------------------------------------------------------------------
architecture behavior of CslAxiToAxilBlockTb is 
   
   --Inputs
   signal clk_i            : sl                    := '0'; --! internal clock signal bus
   signal rstn_i           : sl                    := '1'; --! internal reset signal bus
   signal axiReadMaster_i  : AxiReadMasterType     := AXI_READ_MASTER_INIT_C; 
   signal axilReadSlave_i  : AxiLiteReadSlaveType  := AXI_LITE_READ_SLAVE_INIT_C; 
   signal axiWriteMaster_i : AxiWriteMasterType    := AXI_WRITE_MASTER_INIT_C; 
   signal axilWriteSlave_i : AxiLiteWriteSlaveType := AXI_LITE_WRITE_SLAVE_INIT_C; 
   
   --Outputs
   signal axiReadSlave_o    : AxiReadSlaveType := AXI_READ_SLAVE_INIT_C; 
   signal axilReadMaster_o  : AxiLiteReadMasterType; 
   signal axiWriteSlave_o   : AxiWriteSlaveType := AXI_WRITE_SLAVE_INIT_C; 
   signal axilWriteMaster_o : AxiLiteWriteMasterType; 
   
   -- other signals
   signal rdData : Slv32Array(0 to 255); 
   signal wrData : Slv32Array(0 to 255); 
   
   -- Clock period definitions 
   constant T_C            : time             := 10.0 ns; --! Clock period constant
   constant TPD_C          : time             := 1.0 ns;  --! Propagation delay
   constant NUM_OF_READS_C : integer          := 10; 
   constant ONE_VECT_C     : slv(31 downto 0) := (others => '1'); 
   constant ZERO_VECT_C    : slv(31 downto 0) := (others => '0'); 
---------------------------------------------------------------------------------------------------
begin
   --! Instantiate the Unit Under Test (UUT)
   uut : entity work.CslAxiToAxilBlock 
      port map(
         clk_i          => clk_i,
         rstn_i         => rstn_i,                               -- in std_logic;
         s_axi_awaddr   => axiWriteMaster_i.awaddr(31 DOWNTO 0), --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_awlen    => axiWriteMaster_i.awlen(7 DOWNTO 0),   --: IN STD_LOGIC_VECTOR(7 DOWNTO 0);
         s_axi_awsize   => axiWriteMaster_i.awsize(2 DOWNTO 0),  --: IN STD_LOGIC_VECTOR(2 DOWNTO 0);
         s_axi_awburst  => axiWriteMaster_i.awburst(1 DOWNTO 0), --: IN STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_awlock   => axiWriteMaster_i.awlock(0 DOWNTO 0),  --: IN STD_LOGIC_VECTOR(0 DOWNTO 0);
         s_axi_awcache  => axiWriteMaster_i.awcache(3 DOWNTO 0), --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_awprot   => axiWriteMaster_i.awprot(2 DOWNTO 0),  --: IN STD_LOGIC_VECTOR(2 DOWNTO 0);
         s_axi_awregion => axiWriteMaster_i.awregion(3 DOWNTO 0),--: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_awqos    => axiWriteMaster_i.awqos(3 DOWNTO 0),   --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_awvalid  => axiWriteMaster_i.awvalid,             --: IN STD_LOGIC;
         s_axi_awready  => axiWriteSlave_o.awready,              --: OUT STD_LOGIC;
         s_axi_wdata    => axiWriteMaster_i.wdata(31 DOWNTO 0),  -- : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_wstrb    => axiWriteMaster_i.wstrb(3 DOWNTO 0),   --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_wlast    => axiWriteMaster_i.wlast,               --: IN STD_LOGIC;
         s_axi_wvalid   => axiWriteMaster_i.wvalid,              --: IN STD_LOGIC;
         s_axi_wready   => axiWriteSlave_o.wready,               --: OUT STD_LOGIC;
         s_axi_bresp    => axiWriteSlave_o.bresp(1 DOWNTO 0),    --: OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_bvalid   => axiWriteSlave_o.bvalid,               --: OUT STD_LOGIC;
         s_axi_bready   => axiWriteMaster_i.bready,              --: IN STD_LOGIC;
         s_axi_araddr   => axiReadMaster_i.araddr(31 DOWNTO 0),  --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_arlen    => axiReadMaster_i.arlen(7 DOWNTO 0),    --: IN STD_LOGIC_VECTOR(7 DOWNTO 0);
         s_axi_arsize   => axiReadMaster_i.arsize(2 DOWNTO 0),   --: IN STD_LOGIC_VECTOR(2 DOWNTO 0);
         s_axi_arburst  => axiReadMaster_i.arburst(1 DOWNTO 0),  --: IN STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_arlock   => axiReadMaster_i.arlock(0 DOWNTO 0),   --: IN STD_LOGIC_VECTOR(0 DOWNTO 0);
         s_axi_arcache  => axiReadMaster_i.arcache(3 DOWNTO 0),  --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_arprot   => axiReadMaster_i.arprot(2 DOWNTO 0),   --: IN STD_LOGIC_VECTOR(2 DOWNTO 0);
         s_axi_arregion => axiReadMaster_i.arregion(3 DOWNTO 0), --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_arqos    => axiReadMaster_i.arqos(3 DOWNTO 0),    --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_arvalid  => axiReadMaster_i.arvalid,              --: IN STD_LOGIC;
         s_axi_arready  => axiReadSlave_o.arready,               --: OUT STD_LOGIC;
         s_axi_rdata    => axiReadSlave_o.rdata(31 DOWNTO 0),    --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_rresp    => axiReadSlave_o.rresp(1 DOWNTO 0),     --: OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_rlast    => axiReadSlave_o.rlast,                 --: OUT STD_LOGIC;
         s_axi_rvalid   => axiReadSlave_o.rvalid,                --: OUT STD_LOGIC;
         s_axi_rready   => axiReadMaster_i.rready,               --: IN STD_LOGIC;
         
         --Axi Lite
         m_axi_awaddr  => axilWriteMaster_o.awaddr(31 DOWNTO 0), --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         m_axi_awprot  => axilWriteMaster_o.awprot(2 DOWNTO 0),  --: OUT STD_LOGIC_VECTOR(2 DOWNTO 0);
         m_axi_awvalid => axilWriteMaster_o.awvalid,             --: OUT STD_LOGIC;
         m_axi_awready => axilWriteSlave_i.awready,              --: IN STD_LOGIC;
         m_axi_wdata   => axilWriteMaster_o.wdata(31 DOWNTO 0),  --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         m_axi_wstrb   => axilWriteMaster_o.wstrb(3 DOWNTO 0),   --: OUT STD_LOGIC_VECTOR(3 DOWNTO 0);
         m_axi_wvalid  => axilWriteMaster_o.wvalid,              -- : OUT STD_LOGIC;
         m_axi_wready  => axilWriteSlave_i.wready,               --: IN STD_LOGIC;
         m_axi_bresp   => axilWriteSlave_i.bresp(1 DOWNTO 0),    --: IN STD_LOGIC_VECTOR(1 DOWNTO 0);
         m_axi_bvalid  => axilWriteSlave_i.bvalid,               --: IN STD_LOGIC;
         m_axi_bready  => axilWriteMaster_o.bready,              --: OUT STD_LOGIC;
         m_axi_araddr  => axilReadMaster_o.araddr(31 DOWNTO 0),  --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         m_axi_arprot  => axilReadMaster_o.arprot(2 DOWNTO 0),   --: OUT STD_LOGIC_VECTOR(2 DOWNTO 0);
         m_axi_arvalid => axilReadMaster_o.arvalid,              --: OUT STD_LOGIC;
         m_axi_arready => axilReadSlave_i.arready,               --: IN STD_LOGIC;
         m_axi_rdata   => axilReadSlave_i.rdata(31 DOWNTO 0),    --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         m_axi_rresp   => axilReadSlave_i.rresp(1 DOWNTO 0),     --: IN STD_LOGIC_VECTOR(1 DOWNTO 0);
         m_axi_rvalid  => axilReadSlave_i.rvalid,                --: IN STD_LOGIC;
         m_axi_rready  => axilReadMaster_o.rready                --: OUT STD_LOGIC
      ); 
   
   U_axiSlave : entity work.axiLiteRam
      port map (
         s_aclk        => clk_i,
         s_aresetn     => rstn_i,
         s_axi_awaddr  => axilWriteMaster_o.awaddr(31 downto 0), --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_awvalid => axilWriteMaster_o.awvalid,             --: IN STD_LOGIC;
         s_axi_awready => axilWriteSlave_i.awready,              --: OUT STD_LOGIC;
         s_axi_wdata   => axilWriteMaster_o.wdata(31 downto 0),  --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_wstrb   => axilWriteMaster_o.wstrb(3 downto 0),   --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_wvalid  => axilWriteMaster_o.wvalid,              --: IN STD_LOGIC;
         s_axi_wready  => axilWriteSlave_i.wready,               --: OUT STD_LOGIC;
         s_axi_bresp   => axilWriteSlave_i.bresp(1 downto 0),    --: OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_bvalid  => axilWriteSlave_i.bvalid,               -- : OUT STD_LOGIC;
         s_axi_bready  => axilWriteMaster_o.bready,              --: IN STD_LOGIC;
         s_axi_araddr  => axilReadMaster_o.araddr,               --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_arvalid => axilReadMaster_o.arvalid,              --: IN STD_LOGIC;
         s_axi_arready => axilReadSlave_i.arready,               --: OUT STD_LOGIC;
         s_axi_rdata   => axilReadSlave_i.rdata,                 --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_rresp   => axilReadSlave_i.rresp,                 -- : OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_rvalid  => axilReadSlave_i.rvalid,                --: OUT STD_LOGIC;
         s_axi_rready  => axilReadMaster_o.rready                --: IN STD_LOGIC
      ); 
   
   U_axiLiteCheck : entity work.axiLiteChecker
      port map (
         pc_status      => open,
         pc_asserted    => open,
         aclk           => clk_i,
         aresetn        => rstn_i,
         pc_axi_awaddr  => axilWriteMaster_o.awaddr,
         pc_axi_awprot  => axilWriteMaster_o.awprot,
         pc_axi_awvalid => axilWriteMaster_o.awvalid,
         pc_axi_awready => axilWriteSlave_i.awready,
         pc_axi_wdata   => axilWriteMaster_o.wdata,
         pc_axi_wstrb   => axilWriteMaster_o.wstrb,
         pc_axi_wvalid  => axilWriteMaster_o.wvalid,
         pc_axi_wready  => axilWriteSlave_i.wready,
         pc_axi_bresp   => axilWriteSlave_i.bresp,
         pc_axi_bvalid  => axilWriteSlave_i.bvalid,
         pc_axi_bready  => axilWriteMaster_o.bready,
         pc_axi_araddr  => axilReadMaster_o.araddr,
         pc_axi_arprot  => ZERO_VECT_C(2 downto 0),
         pc_axi_arvalid => axilReadMaster_o.arvalid,
         pc_axi_arready => axilReadSlave_i.arready,
         pc_axi_rdata   => axilReadSlave_i.rdata,
         pc_axi_rresp   => axilReadSlave_i.rresp,
         pc_axi_rvalid  => axilReadSlave_i.rvalid,
         pc_axi_rready  => axilReadMaster_o.rready
      ); 
   
   U_axiChecker : entity work.axiChecker
      port map (
         pc_status       => open,
         pc_asserted     => open,
         aclk            => clk_i,
         aresetn         => rstn_i,
         pc_axi_awid     => axiWriteMaster_i.awid(3 downto 0),
         pc_axi_awaddr   => axiWriteMaster_i.awaddr(31 downto 0),
         pc_axi_awlen    => axiWriteMaster_i.awlen(7 downto 0),
         pc_axi_awsize   => axiWriteMaster_i.awsize(2 downto 0),
         pc_axi_awburst  => axiWriteMaster_i.awburst(1 downto 0),
         pc_axi_awlock   => axiWriteMaster_i.awlock(0 downto 0),
         pc_axi_awcache  => axiWriteMaster_i.awcache(3 downto 0),
         pc_axi_awprot   => axiWriteMaster_i.awprot(2 downto 0),
         pc_axi_awqos    => axiWriteMaster_i.awqos(3 downto 0),
         pc_axi_awregion => axiWriteMaster_i.awregion(3 downto 0),
         pc_axi_awvalid  => axiWriteMaster_i.awvalid,
         pc_axi_awready  => axiWriteSlave_o.awready,
         pc_axi_wlast    => axiWriteMaster_i.wlast,
         pc_axi_wdata    => axiWriteMaster_i.wdata(31 downto 0),
         pc_axi_wstrb    => axiWriteMaster_i.wstrb(3 downto 0),
         pc_axi_wvalid   => axiWriteMaster_i.wvalid,
         pc_axi_wready   => axiWriteSlave_o.wready,
         pc_axi_bid      => axiWriteSlave_o.bid(3 downto 0),
         pc_axi_bresp    => axiWriteSlave_o.bresp(1 downto 0),
         pc_axi_bvalid   => axiWriteSlave_o.bvalid,
         pc_axi_bready   => axiWriteMaster_i.bready,
         pc_axi_arid     => axiReadMaster_i.arid(3 downto 0),
         pc_axi_araddr   => axiReadMaster_i.araddr(31 downto 0),
         pc_axi_arlen    => axiReadMaster_i.arlen(7 downto 0),
         pc_axi_arsize   => axiReadMaster_i.arsize(2 downto 0),
         pc_axi_arburst  => axiReadMaster_i.arburst(1 downto 0),
         pc_axi_arlock   => axiReadMaster_i.arlock(0 downto 0),
         pc_axi_arcache  => axiReadMaster_i.arcache(3 downto 0),
         pc_axi_arprot   => axiReadMaster_i.arprot(2 downto 0),
         pc_axi_arqos    => axiReadMaster_i.arqos(3 downto 0),
         pc_axi_arregion => axiReadMaster_i.arregion(3 downto 0),
         pc_axi_arvalid  => axiReadMaster_i.arvalid,
         pc_axi_arready  => axiReadSlave_o.arready,
         pc_axi_rid      => axiReadSlave_o.rid(3 downto 0),
         pc_axi_rlast    => axiReadSlave_o.rlast,
         pc_axi_rdata    => axiReadSlave_o.rdata(31 downto 0),
         pc_axi_rresp    => axiReadSlave_o.rresp,
         pc_axi_rvalid   => axiReadSlave_o.rvalid,
         pc_axi_rready   => axiReadMaster_i.rready
      ); 
   
   --! @brief Clock procces 
   --! @details Clock signal generator
   --! @param[in]  T_C
   --! @param[out] clk_i
   p_SyncClkGen : process
   begin
      clk_i <= '1'; 
      wait for T_C/2; 
      clk_i <= '0'; 
      wait for T_C/2; 
   end process; 
   
   --! @brief 
   --! @details 
   --! @param[in] 
   --! @param[out]
   p_Sim : process
      variable ramVect_v : std_logic_vector(31 downto 0); 
   begin
      ramVect_v := (others => '0'); 
      for I in 0 to 255 loop --create some write data
         wrData(I) <= ramVect_v; 
         ramVect_v := ramVect_v+3; 
         wait for T_C * 1; 
      end loop; 
      
      -- input signals
      rstn_i <= '1'; 
      wait for T_C * 10; 
      
      -- reset
      rstn_i <= '0'; 
      wait for T_C * 10; 
      rstn_i <= '1'; 
      wait for T_C * 10; 
      
      -- simulate--
      -- 256 writes
      tbAxiWrite32(clk_i, TPD_C, axiWriteMaster_i, axiWriteSlave_o, x"0000_0000", (others => (others => '0')), x"ff"); --overwrite whole space
      tbAxiWrite32(clk_i, TPD_C, axiWriteMaster_i, axiWriteSlave_o, x"0000_0000", wrData, x"ff"); 
      tbAxiRead32(clk_i, TPD_C, axiReadMaster_i, axiReadSlave_o, x"0000_0000", x"ff",rdData); 
      if(wrData(0 to 255)/=rdData(0 to 255))then --Check correctnes
         printf(":: Data does not match!  \@"); 
         assert false severity failure; 
      end if; 
      rdData <= (others => (others => '0')); 
      
      -- 16 writes
      tbAxiWrite32(clk_i, TPD_C, axiWriteMaster_i, axiWriteSlave_o, x"0000_0000", (others => (others => '0')), x"ff"); --overwrite whole space
      tbAxiWrite32(clk_i, TPD_C, axiWriteMaster_i, axiWriteSlave_o, x"0000_0000", wrData, x"0f"); 
      tbAxiRead32(clk_i, TPD_C, axiReadMaster_i, axiReadSlave_o, x"0000_0000", x"0f",rdData); 
      if(wrData(0 to 15)/=rdData(0 to 15))then --Check correctnes
         printf(":: Data does not match!  \@"); 
         assert false severity failure; 
      end if; 
      rdData <= (others => (others => '0')); 
      
      -- 1 write
      tbAxiWrite32(clk_i, TPD_C, axiWriteMaster_i, axiWriteSlave_o, x"0000_0000", (others => (others => '0')), x"ff"); --overwrite whole space
      tbAxiWrite32(clk_i, TPD_C, axiWriteMaster_i, axiWriteSlave_o, x"0000_0000", wrData, x"00"); 
      tbAxiRead32(clk_i, TPD_C, axiReadMaster_i, axiReadSlave_o, x"0000_0000", x"00",rdData); 
      if(wrData(0)/=rdData(0))then --Check correctnes
         printf(":: Data does not match!  \@"); 
         assert false severity failure; 
      end if; 
      rdData <= (others => (others => '0')); 
      
      wait for T_C * 50; 
      
      -- Stop simulation
      assert false report "SIMULATION COMPLEATED" severity failure; 
      
   end process; 
end; 
---------------------------------------------------------------------------------------------------
