---------------------------------------------------------------------------------------------------
--! @brief  CslAxiToAxilRead.vhd testbench
--! @details  Testbench employs BRAM with axiLite interface which is initialized with values from .coe file (coefs.coe). Copy of that
--!           file (coefsTb.coe) is parsed by this tesbench and stored into ramData array. The tesbench performs three reads 
--!           from RAM, with lengths 256, 16 and 1 transaction. CslAxiToAxilRead converts axi4 interface to axi 
--!           Lite interface, reads RAM content and returns read data (rdData_o), which is then compared to data
--!           from file. If data matches, transaction was successfull. Both axi4 and axiLite interface are 
--!           connected to Xilinx axi protocol checker to ensure interfaces are according to the axi specifications.
--!
--! @author Jost Reberc(jost.reebrc@cosylab.com)
--!
--! @date March 15 2019 created
--!
--! @version 
--!
--!
--! @file CslAxiToAxilReadTb.vhd
---------------------------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.numeric_std.all; 
use ieee.std_logic_unsigned.all; 
use work.CslStdRtlPkg.all; 
use work.CslAxiPkg.all; 
use work.CslTbPkg.all; 
use work.CslPrintfPkg.all; 
use ieee.std_logic_textio.all; 
use std.textio.all; 
---------------------------------------------------------------------------------------------------
entity CslAxiToAxilReadTb is
end CslAxiToAxilReadTb; 
---------------------------------------------------------------------------------------------------
architecture behavior of CslAxiToAxilReadTb is 
   
   --Inputs
   signal clk_i           : sl                   := '0'; --! internal clock signal bus
   signal rst_i           : sl                   := '0'; --! internal reset signal bus
   signal axiReadMaster_i : AxiReadMasterType    := AXI_READ_MASTER_INIT_C; 
   signal axilReadSlave_i : AxiLiteReadSlaveType := AXI_LITE_READ_SLAVE_INIT_C; 
   
   --Outputs
   signal axiReadSlave_o   : AxiReadSlaveType := AXI_READ_SLAVE_INIT_C; 
   signal axilReadMaster_o : AxiLiteReadMasterType; 
   
   -- other signals
   file fileVectors : text; 
   signal rdData_o  : Slv32Array(0 to 255); 
   signal ramData   : Slv32Array(0 to 255); 
   
   -- Clock period definitions 
   constant T_C            : time             := 10.0 ns; --! Clock period constant
   constant TPD_C          : time             := 1.0 ns;  --! Propagation delay
   constant NUM_OF_READS_C : integer          := 10; 
   constant ONE_VECT_C     : slv(31 downto 0) := (others => '1'); 
   constant ZERO_VECT_C    : slv(31 downto 0) := (others => '0'); 
   
---------------------------------------------------------------------------------------------------
begin
   --! Instantiate the Unit Under Test (UUT)
   uut : entity work.CslAxiToAxilRead
      generic map(
         --! Propagation delay
         TPD_G => TPD_C
      )
      port map( 
         clk_i            => clk_i,
         rst_i            => rst_i, --: in  sl;
         axiReadMaster_i  => axiReadMaster_i,
         axiReadSlave_o   => axiReadSlave_o,
         axilReadMaster_o => axilReadMaster_o, --: out AxiLiteReadMasterType;
         axilReadSlave_i  => axilReadSlave_i   --: in  AxiLiteReadSlaveType
      ); 
   
   U_axiSlave : entity work.axiLiteRam
      port map (
         s_aclk        => clk_i,
         s_aresetn     => not rst_i,
         s_axi_awaddr  => ZERO_VECT_C,              --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_awvalid => ZERO_VECT_C(0),           --: IN STD_LOGIC;
         s_axi_awready => open,                     --: OUT STD_LOGIC;
         s_axi_wdata   => ZERO_VECT_C,              --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_wstrb   => ZERO_VECT_C(3 downto 0),  --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_wvalid  => ZERO_VECT_C(0),           --: IN STD_LOGIC;
         s_axi_wready  => open,                     --: OUT STD_LOGIC;
         s_axi_bresp   => open,                     --: OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_bvalid  => open,                     -- : OUT STD_LOGIC;
         s_axi_bready  => ZERO_VECT_C(0),           --: IN STD_LOGIC;
         s_axi_araddr  => axilReadMaster_o.araddr,  --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_arvalid => axilReadMaster_o.arvalid, --: IN STD_LOGIC;
         s_axi_arready => axilReadSlave_i.arready,  --: OUT STD_LOGIC;
         s_axi_rdata   => axilReadSlave_i.rdata,    --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_rresp   => axilReadSlave_i.rresp,    -- : OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_rvalid  => axilReadSlave_i.rvalid,   --: OUT STD_LOGIC;
         s_axi_rready  => axilReadMaster_o.rready   --: IN STD_LOGIC
      ); 
   
   U_axiLiteReadCheck : entity work.axiLiteReadChecker
      port map (
         pc_status      => open,
         pc_asserted    => open,
         aclk           => clk_i,
         aresetn        => not rst_i,
         pc_axi_araddr  => axilReadMaster_o.araddr,
         pc_axi_arprot  => ZERO_VECT_C(2 downto 0),
         pc_axi_arvalid => axilReadMaster_o.arvalid,
         pc_axi_arready => axilReadSlave_i.arready,
         pc_axi_rdata   => axilReadSlave_i.rdata,
         pc_axi_rresp   => axilReadSlave_i.rresp,
         pc_axi_rvalid  => axilReadSlave_i.rvalid,
         pc_axi_rready  => axilReadMaster_o.rready
      ); 
   
   U_axiReadChecker : entity work.axiReadChecker
      port map (
         pc_status       => open,
         pc_asserted     => open,
         aclk            => clk_i,
         aresetn         => not rst_i,
         pc_axi_arid     => axiReadMaster_i.arid(3 downto 0),
         pc_axi_araddr   => axiReadMaster_i.araddr(31 downto 0),
         pc_axi_arlen    => axiReadMaster_i.arlen(7 downto 0),
         pc_axi_arsize   => axiReadMaster_i.arsize(2 downto 0),
         pc_axi_arburst  => axiReadMaster_i.arburst(1 downto 0),
         pc_axi_arlock   => axiReadMaster_i.arlock(0 downto 0),
         pc_axi_arcache  => axiReadMaster_i.arcache(3 downto 0),
         pc_axi_arprot   => axiReadMaster_i.arprot(2 downto 0),
         pc_axi_arqos    => axiReadMaster_i.arqos(3 downto 0),
         pc_axi_arregion => axiReadMaster_i.arregion(3 downto 0),
         pc_axi_arvalid  => axiReadMaster_i.arvalid,
         pc_axi_arready  => axiReadSlave_o.arready,
         pc_axi_rid      => axiReadSlave_o.rid(3 downto 0),
         pc_axi_rlast    => axiReadSlave_o.rlast,
         pc_axi_rdata    => axiReadSlave_o.rdata(31 downto 0),
         pc_axi_rresp    => axiReadSlave_o.rresp,
         pc_axi_rvalid   => axiReadSlave_o.rvalid,
         pc_axi_rready   => axiReadMaster_i.rready
      ); 
   
   --! @brief Clock procces 
   --! @details Clock signal generator
   --! @param[in]  T_C
   --! @param[out] clk_i
   p_SyncClkGen : process
   begin
      clk_i <= '1'; 
      wait for T_C/2; 
      clk_i <= '0'; 
      wait for T_C/2; 
   end process; 
   
   --! @brief 
   --! @details 
   --! @param[in] 
   --! @param[out]
   p_Sim : process
      variable vIline_v  : line; 
      variable ramVect_v : std_logic_vector(31 downto 0); 
   begin
      
      file_open(fileVectors, "coefsTb.coe", read_mode); 
      readline(fileVectors, vIline_v); --read info line
      readline(fileVectors, vIline_v); --read info line
      
      for I in 0 to 255 loop
         if(not endfile(fileVectors))then
            readline(fileVectors, vIline_v); 
            hread(vIline_v, ramVect_v); 
            ramData(I) <= ramVect_v; 
            wait for T_C * 1; 
         end if; 
      end loop; 
      
      -- input signals
      rst_i <= '0'; 
      wait for T_C * 10; 
      
      -- reset
      rst_i <= '1'; 
      wait for T_C * 10; 
      rst_i <= '0'; 
      wait for T_C * 10; 
      
      -- simulate--
      --! 256 reads
      tbAxiRead32(clk_i, TPD_C, axiReadMaster_i, axiReadSlave_o, x"0000_0000", x"ff",rdData_o); 
      if(ramData(0 to 255)/=rdData_o(0 to 255))then --Check correctnes
         printf(":: Data does not match!  \@"); 
         assert false severity failure; 
      end if; 
      rdData_o <= (others => (others => '0')); 
      
      --! 16 reads
      tbAxiRead32(clk_i, TPD_C, axiReadMaster_i, axiReadSlave_o, x"0000_0000", x"0f",rdData_o); 
      if(ramData(0 to 15)/=rdData_o(0 to 15))then --Check correctnes
         printf(":: Data does not match!  \@"); 
         assert false severity failure; 
      end if; 
      rdData_o <= (others => (others => '0')); 
      
      --! 1 read
      tbAxiRead32(clk_i, TPD_C, axiReadMaster_i, axiReadSlave_o, x"0000_0000", x"00",rdData_o); 
      if(ramData(0)/=rdData_o(0))then --Check correctnes
         printf(":: Data does not match!  \@"); 
         assert false severity failure; 
      end if; 
      rdData_o <= (others => (others => '0')); 
      
      wait for T_C * 50; 
      
      file_close(fileVectors); 
      
      -- Stop simulation
      assert false report "SIMULATION COMPLEATED" severity failure; 
      
   end process; 
end; 
---------------------------------------------------------------------------------------------------
