---------------------------------------------------------------------------------------------------
--! @brief  CslAxilRegisterMap testbench 
--! @details Testbench test all the functionalities of CslAxilRegisterMap component. User can select the number of implemented
--!          registers, addressing allignment (1 or 32 bit using BIT_ALIGN_32_C) and R/W  priority (using READ_FIRST_C)
--!
--! @author Jost Reberc (jost.reberc@cosylab.com)
--!
--! @date April 10 2019 created
--!
--! @version 
--!
--!
--! @file CslAxilRegisterMapTb.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslRegisterMapPkg.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.CslTbPkg.all;
use work.CslPrintfPkg.all;

--! @brief
--! @details
--! @author
---------------------------------------------------------------------------------------------------
entity CslAxilRegisterMapTb is
end CslAxilRegisterMapTb;
---------------------------------------------------------------------------------------------------
architecture behavior of CslAxilRegisterMapTb is

   --! NOTE: setting BIT_ALIGN_32_C to false produces the following error since protocol checker doesnt support unaligned addressing:
   --! ERROR : AXI_ERRM_WSTRB: Write strobes must only be asserted for the correct byte lanes as determined from start address, transfer size and beat number. Spec: section A3.4.3.

   --Constants
   constant TPD_C          : time    := 1 ns;
   constant BIT_ALIGN_32_C : boolean := false;
   constant READ_FIRST_C   : boolean := true;
   constant NUM_OF_REGS_C  : integer := 100;

   --Inputs
   signal clk_i : sl := '0'; --! internal clock signal bus
   signal rst_i : sl := '0'; --! internal reset signal bus

   --Outputs
   signal regStatus_i       : Slv32Array(0 to NUM_OF_REGS_C - 1) := (others => (others => '0'));
   signal regData_o         : Slv32Array(0 to NUM_OF_REGS_C - 1);
   signal axilReadMaster_i  : AxiLiteReadMasterType  := AXI_LITE_READ_MASTER_INIT_C;
   signal axilWriteMaster_i : AxiLiteWriteMasterType := AXI_LITE_WRITE_MASTER_INIT_C;
   signal axilWriteSlave_o  : AxiLiteWriteSlaveType;
   signal axilReadSlave_o   : AxiLiteReadSlaveType;
   signal tmpVect           : slv(31 downto 0);
   signal addrVect          : slv(31 downto 0);
   signal rdData            : slv(31 downto 0);
   signal wrData            : slv(31 downto 0);
   signal rdEcpected        : slv(31 downto 0);
   signal clrVct            : slv(31 downto 0);
   signal regConfig         : dataRegArrayType (0 to NUM_OF_REGS_C - 1);

   -- Clock period definitions 
   constant T_C         : time             := 10.0 ns; --! Clock period constant
   constant ONE_VECT_C  : slv(31 downto 0) := (others => '1');
   constant ZERO_VECT_C : slv(31 downto 0) := (others => '0');

begin

   --! Instantiate the Unit Under Test (UUT)
   uut_CslRegisterMap_1 : entity work.CslAxilRegisterMap
      generic map (
         TPD_G          => TPD_C,
         BIT_ALIGN_32_G => BIT_ALIGN_32_C,
         READ_FIRST_G   => READ_FIRST_C,
         NUM_OF_REGS_G  => NUM_OF_REGS_C
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i, 
         init_i            => regConfig,
         status_i          => regStatus_i,
         config_o          => regData_o,
         axilReadMaster_i  => axilReadMaster_i,
         axilWriteMaster_i => axilWriteMaster_i,
         axilWriteSlave_o  => axilWriteSlave_o,
         axilReadSlave_o   => axilReadSlave_o
      );

   U_axiLiteCheck : entity work.axiLiteChecker
      port map (
         pc_status      => open,
         pc_asserted    => open,
         aclk           => clk_i,
         aresetn        => not rst_i,
         pc_axi_awaddr  => axilWriteMaster_i.awaddr,
         pc_axi_awprot  => axilWriteMaster_i.awprot,
         pc_axi_awvalid => axilWriteMaster_i.awvalid,
         pc_axi_awready => axilWriteSlave_o.awready,
         pc_axi_wdata   => axilWriteMaster_i.wdata,
         pc_axi_wstrb   => axilWriteMaster_i.wstrb,
         pc_axi_wvalid  => axilWriteMaster_i.wvalid,
         pc_axi_wready  => axilWriteSlave_o.wready,
         pc_axi_bresp   => axilWriteSlave_o.bresp,
         pc_axi_bvalid  => axilWriteSlave_o.bvalid,
         pc_axi_bready  => axilWriteMaster_i.bready,
         pc_axi_araddr  => axilReadMaster_i.araddr,
         pc_axi_arprot  => ZERO_VECT_C(2 downto 0),
         pc_axi_arvalid => axilReadMaster_i.arvalid,
         pc_axi_arready => axilReadSlave_o.arready,
         pc_axi_rdata   => axilReadSlave_o.rdata,
         pc_axi_rresp   => axilReadSlave_o.rresp,
         pc_axi_rvalid  => axilReadSlave_o.rvalid,
         pc_axi_rready  => axilReadMaster_i.rready
      );

   --! @brief Clock procces 
   --! @details Clock signal generator
   --! @param[in]  T_C
   --! @param[out] clk_i
   p_SyncClkGen : process
   begin
      clk_i <= '0';
      wait for T_C/2;
      clk_i <= '1';
      wait for T_C/2;
   end process;

   --! @brief 
   --! @details 
   --! @param[in] 
   --! @param[out]
   p_Sim : process
   begin

      --Setup configuration of each register based on its address
      for a in 0 to NUM_OF_REGS_C-1 loop
         tmpVect <= (others => '0');
         tmpVect <= std_logic_vector(to_unsigned(a, tmpVect'length));
         wait for T_C * 10;
         regStatus_i(a) <= tmpVect;
         regConfig(a)   <= (data => tmpVect, addr => a, autoClr => ("0"&tmpVect(tmpVect'left downto 1)), readOnly => tmpVect(0));
      end loop;

      wait until clk_i = '1';
      wait for TPD_C;

      -- input signals
      rst_i <= '0';
      wait for T_C * 10;

      -- reset
      rst_i <= '1';
      wait for T_C * 10;

      rst_i <= '0';
      wait for T_C * 10;

      -- simulate
      wait for T_C * 10;

      -- Read all registers to cheeck if initial configuration is correct
      for ii in 0 to NUM_OF_REGS_C-1 loop
         addrVect <= (others => '0');
         addrVect <= std_logic_vector(to_unsigned(ii,32));
         wait for T_C * 1;
         if(BIT_ALIGN_32_C = true)then
            tbAxilRead(clk_i, TPD_C, axilReadMaster_i, axilReadSlave_o,rdData, (addrVect(addrVect'left-2 downto 0)&"00"));
         else
            tbAxilRead(clk_i, TPD_C, axilReadMaster_i, axilReadSlave_o,rdData, addrVect);
         end if;
         if(addrVect(0) = '1')then
            tbCheckSlv(rdData,addrVect,"Read of read only register does not match");
         else
            clrVct <= ("0"&addrVect(addrVect'left downto 1));
            wait for T_C * 1;
            for i in 0 to 31 loop
               if(clrVct(i) = '1')then
                  rdEcpected(i) <= '0';
                  wait for T_C * 1;
               else
                  rdEcpected(i) <= addrVect(i);
                  wait for T_C * 1;
               end if;
            end loop;
            tbCheckSlv(rdData,rdEcpected,"Read does not match");
         end if;
      end loop;

      -- Write some data to each register
      for ii in 0 to NUM_OF_REGS_C-1 loop
         addrVect <= std_logic_vector(to_unsigned(ii,32));
         wrData   <= std_logic_vector(to_unsigned(ii+3,32));
         wait for T_C * 1;

         if(BIT_ALIGN_32_C = true)then
            tbAxilWrite(clk_i, TPD_C, axilWriteMaster_i, axilWriteSlave_o, (addrVect(addrVect'left-2 downto 0)&"00"), wrData);
         else
            tbAxilWrite(clk_i, TPD_C, axilWriteMaster_i, axilWriteSlave_o, addrVect, wrData);
         end if;
      end loop;

      -- Read all registers to cheeck if configuration after writing is correct
      for ii in 0 to NUM_OF_REGS_C-1 loop
         addrVect <= std_logic_vector(to_unsigned(ii,32));
         wrData   <= std_logic_vector(to_unsigned(ii+3,32));
         wait for T_C * 1;

         if(BIT_ALIGN_32_C = true)then
            tbAxilRead(clk_i, TPD_C, axilReadMaster_i, axilReadSlave_o,rdData, (addrVect(addrVect'left-2 downto 0)&"00"));
         else
            tbAxilRead(clk_i, TPD_C, axilReadMaster_i, axilReadSlave_o,rdData, addrVect);
         end if;
         if(addrVect(0) = '1')then
            tbCheckSlv(rdData,addrVect,"Read of read only register does not match");
         else
            clrVct <= ("0"&addrVect(addrVect'left downto 1));
            wait for T_C * 1;
            for i in 0 to 31 loop
               if(clrVct(i) = '1')then
                  rdEcpected(i) <= '0';
                  wait for T_C * 1;
               else
                  rdEcpected(i) <= wrData(i);
                  wait for T_C * 1;
               end if;
            end loop;
            tbCheckSlv(rdData,rdEcpected,"Read does not match");
         end if;
      end loop;

      wait for T_C * 10;

      -- Test simultaneous R/W acces try (READ_FIRST_G ): this test can cause one of the following
      -- two errors when successful (from axi Lite Checker):
      -- ERROR : AXI_ERRM_AWVALID_STABLE: Once AWVALID is asserted, it must remain asserted until AWREADY is high. Spec: section A3.2.2.
      -- ERROR : AXI_ERRM_ARVALID_STABLE: Once ARVALID is asserted, it must remain asserted until ARREADY is high. Spec: section A3.2.1.
      axilReadMaster_i.arvalid  <= '1';
      axilReadMaster_i.araddr   <= (others => '0');
      axilWriteMaster_i.awvalid <= '1';
      axilWriteMaster_i.awaddr  <= (others => '0');
      wait for T_C;
      if(READ_FIRST_C = true)then --on simultaneous access read has priority
         if(axilReadSlave_o.arready = '0')then
            wait until axilReadSlave_o.arready = '1';
         end if;
         if( axilWriteSlave_o.awready = '1')then
            assert false report "READ_FIRST_G error" severity failure;
         end if;
         wait for T_C;
         axilReadMaster_i.arvalid  <= '0';
         axilWriteMaster_i.awvalid <= '0';
         axilReadMaster_i.rready   <= '1'; --end transaction
         wait for T_C;
      else
         if(axilWriteSlave_o.awready = '0')then
            wait until axilWriteSlave_o.awready = '1';
         end if;
         if(axilReadSlave_o.arready = '1' )then
            assert false report "READ_FIRST_G error" severity failure;
         end if;
         wait for T_C;
         --finish transfer
         axilWriteMaster_i.awvalid <= '0';
         axilReadMaster_i.arvalid  <= '0';
         axilWriteMaster_i.wvalid  <= '1';
         if(axilWriteSlave_o.wready = '0')then
            wait until axilWriteSlave_o.wready = '1';
         end if;
         wait for T_C;
         axilWriteMaster_i.wvalid <= '0';
         axilWriteMaster_i.bready <= '1';
         wait for T_C;
         if(axilWriteSlave_o.bvalid = '0')then
            wait until axilWriteSlave_o.bvalid = '1';
         end if;
         axilWriteMaster_i.bready <= '0';
         wait for T_C;
      end if;

      wait for T_C * 5;

      -- test address guard (axilReadSlave_o.rresp should be equal to AXI_RESP_DECERR_C)
      tbAxilRead(clk_i, TPD_C, axilReadMaster_i, axilReadSlave_o,rdData, x"00FF0000");

      -- Stop simulation
      assert false report "SIMULATION COMPLEATED" severity failure;

   end process;

end;
---------------------------------------------------------------------------------------------------