---------------------------------------------------------------------------------------------------
--! @brief  CslAxilWrite.vhd testbench
--! @details  Testbench employs BRAM with axiLite interface which is written to using CslAxilWrite.vhd component. The tesbench performs 
--!           NUM_OF_WRITES_C writes of data to RAM, reads it back (read using CslAxilRead.vhd) and compares it. If data matches, transaction was successfull. 
--!           Axi Lite interface is connected to Xilinx axi protocol checker to ensure it works according to the axi specifications.
--!
--! @author Jost Reberc(jost.reebrc@cosylab.com)
--!
--! @date March 15 2019 created
--!
--! @version 
--!
--! @file CslAxilWriteTb.vhd
---------------------------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.numeric_std.all; 
use ieee.std_logic_unsigned.all; 
use work.CslStdRtlPkg.all; 
use work.CslAxiPkg.all; 
use work.CslTbPkg.all; 
use work.CslPrintfPkg.all; 
---------------------------------------------------------------------------------------------------
entity CslAxilWriteTb is
end CslAxilWriteTb; 
---------------------------------------------------------------------------------------------------
architecture behavior of CslAxilWriteTb is 
   
   --Inputs
   signal clk_i            : sl := '0'; --! internal clock signal bus
   signal rst_i            : sl := '0'; --! internal reset signal bus
   signal rdValid_i        : sl; 
   signal wrValid_i        : sl; 
   signal rdAddr_i         : slv(31 downto 0); 
   signal wrAddr_i         : slv(31 downto 0); 
   signal wrData_i         : slv(31 downto 0); 
   signal axilReadSlave_i  : AxiLiteReadSlaveType; 
   signal axilWriteSlave_i : AxiLiteWriteSlaveType; 
   
   --Outputs
   signal rdValid_o         : sl; 
   signal rdData_o          : slv(31 downto 0); 
   signal rdRready_o        : sl; 
   signal wrRready_o        : sl; 
   signal wrDone_o          : sl; 
   signal rdErr_o           : sl; 
   signal wrErr_o           : sl; 
   signal axilReadMaster_o  : AxiLiteReadMasterType; 
   signal axilWriteMaster_o : AxiLiteWriteMasterType; 
   
   -- Clock period definitions 
   constant T_C             : time             := 10.0 ns; --! Clock period constant
   constant TPD_C           : time             := 1.0 ns;  --! Propagation delay
   constant NUM_OF_WRITES_C : integer          := 10; 
   constant ONE_VECT_C      : slv(31 downto 0) := (others => '1'); 
   constant ZERO_VECT_C     : slv(31 downto 0) := (others => '0'); 
   
---------------------------------------------------------------------------------------------------
begin
   
   --! Instantiate the Unit Under Test (UUT)
   U_axilRead : entity work.CslAxilRead
      generic map(
         --! Propagation delay
         TPD_G       => TPD_C,
         VALID_RED_G => true
      )
      port map( 
         clk_i            => clk_i,
         rst_i            => rst_i,            --: in  sl;
         valid_i          => rdValid_i,        --: in  sl;
         valid_o          => rdValid_o,        --: out sl;
         data_o           => rdData_o,         --: out slv(31 downto 0);
         addr_i           => rdAddr_i,         --: in  slv(31 downto 0);
         ready_o          => rdRready_o,       --: out sl;
         err_o            => rdErr_o,          --: out sl;
         axilReadMaster_o => axilReadMaster_o, --: out AxiLiteReadMasterType;
         axilReadSlave_i  => axilReadSlave_i   --: in  AxiLiteReadSlaveType
      ); 
   
   uut : entity work.CslAxilWrite 
      generic map(
         --! Propagation delay
         TPD_G       => TPD_C,
         VALID_RED_G => true
      )
      port map( 
         clk_i             => clk_i,
         rst_i             => rst_i,     --: in  sl;
         valid_i           => wrValid_i, --: in  sl;
         data_i            => wrData_i,
         addr_i            => wrAddr_i,
         ready_o           => wrRready_o,
         wrDone_o          => wrDone_o,
         err_o             => wrErr_o,
         axilWriteSlave_i  => axilWriteSlave_i,
         axilWriteMaster_o => axilWriteMaster_o
      ); 
   
   U_axiSlave : entity work.axiLiteRam
      port map (
         s_aclk        => clk_i,
         s_aresetn     => not rst_i,
         s_axi_awaddr  => axilWriteMaster_o.awaddr,  --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_awvalid => axilWriteMaster_o.awvalid, --: IN STD_LOGIC;
         s_axi_awready => axilWriteSlave_i.awready,  --: OUT STD_LOGIC;
         s_axi_wdata   => axilWriteMaster_o.wdata,   --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_wstrb   => axilWriteMaster_o.wstrb,   --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_wvalid  => axilWriteMaster_o.wvalid,  --: IN STD_LOGIC;
         s_axi_wready  => axilWriteSlave_i.wready,   --: OUT STD_LOGIC;
         s_axi_bresp   => axilWriteSlave_i.bresp,    --: OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_bvalid  => axilWriteSlave_i.bvalid,   -- : OUT STD_LOGIC;
         s_axi_bready  => axilWriteMaster_o.bready,  --: IN STD_LOGIC;
         s_axi_araddr  => axilReadMaster_o.araddr,   --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_arvalid => axilReadMaster_o.arvalid,  --: IN STD_LOGIC;
         s_axi_arready => axilReadSlave_i.arready,   --: OUT STD_LOGIC;
         s_axi_rdata   => axilReadSlave_i.rdata,     --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_rresp   => axilReadSlave_i.rresp,     -- : OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_rvalid  => axilReadSlave_i.rvalid,    --: OUT STD_LOGIC;
         s_axi_rready  => axilReadMaster_o.rready    --: IN STD_LOGIC
      ); 
   
   U_axiLiteCheck : entity work.axiLiteChecker
      port map (
         pc_status      => open,
         pc_asserted    => open,
         aclk           => clk_i,
         aresetn        => not rst_i,
         pc_axi_awaddr  => axilWriteMaster_o.awaddr,
         pc_axi_awprot  => axilWriteMaster_o.awprot,
         pc_axi_awvalid => axilWriteMaster_o.awvalid,
         pc_axi_awready => axilWriteSlave_i.awready,
         pc_axi_wdata   => axilWriteMaster_o.wdata,
         pc_axi_wstrb   => axilWriteMaster_o.wstrb,
         pc_axi_wvalid  => axilWriteMaster_o.wvalid,
         pc_axi_wready  => axilWriteSlave_i.wready,
         pc_axi_bresp   => axilWriteSlave_i.bresp,
         pc_axi_bvalid  => axilWriteSlave_i.bvalid,
         pc_axi_bready  => axilWriteMaster_o.bready,
         pc_axi_araddr  => axilReadMaster_o.araddr,
         pc_axi_arprot  => ZERO_VECT_C(2 downto 0),
         pc_axi_arvalid => axilReadMaster_o.arvalid,
         pc_axi_arready => axilReadSlave_i.arready,
         pc_axi_rdata   => axilReadSlave_i.rdata,
         pc_axi_rresp   => axilReadSlave_i.rresp,
         pc_axi_rvalid  => axilReadSlave_i.rvalid,
         pc_axi_rready  => axilReadMaster_o.rready
      ); 
   
   --! @brief Clock procces 
   --! @details Clock signal generator
   --! @param[in]  T_C
   --! @param[out] clk_i
   p_SyncClkGen : process
   begin
      clk_i <= '1'; 
      wait for T_C/2; 
      clk_i <= '0'; 
      wait for T_C/2; 
   end process; 
   
   --! @brief 
   --! @details 
   --! @param[in] 
   --! @param[out]
   p_Sim : process
   begin
      
      -- input signals
      rdValid_i <= '0'; 
      wrValid_i <= '0'; 
      rst_i     <= '0'; 
      wait for T_C * 10; 
      
      -- reset
      rst_i <= '1'; 
      wait for T_C * 10; 
      rst_i <= '0'; 
      wait for T_C * 10; 
      
      -- simulate--
      wrAddr_i <= (others => '0'); --first address
      rdAddr_i <= (others => '0'); --first address
      wait for T_C * 10; 
      
      --Write addresses to the ram
      for I in 0 to NUM_OF_WRITES_C-1 loop --make NUM_OF_WRITES_C reads
         
         wrValid_i <= '1'; 
         wrData_i  <= wrAddr_i; 
         wait for T_C * 1; 
         if(wrRready_o = '0')then --check if component is ready
            wait until wrRready_o = '1'; 
         end if; 
         wrValid_i <= '0'; 
         wait until wrDone_o = '1'; 
         if(rdErr_o = '0')then
            wrAddr_i <= wrAddr_i+4; --addres increase
            wait for T_C * 1; 
         else
            printf(":: Error detected!  \@"); 
            assert false severity failure; 
         end if; 
         
      end loop; 
      
      --Read back and confirm written data matches
      for I in 0 to NUM_OF_WRITES_C-1 loop --make NUM_OF_WRITES_C reads
         rdValid_i <= '1'; 
         if(rdRready_o = '0')then --check if component is ready
            wait until rdRready_o = '1'; 
         end if; 
         rdValid_i <= '0'; 
         wait until rdValid_o = '1'; --wait data
         if(rdErr_o = '0')then
            tbCheckSlv(rdData_o,rdAddr_i,"Data does not match! ",T_C); 
         else
            printf(":: Error detected!  \@"); 
            assert false severity failure; 
         end if; 
         rdAddr_i <= rdAddr_i+4; --addres increase
         wait for T_C * 1; 
      end loop; 
      
      wait for T_C * 10; 
      
      -- Stop simulation
      assert false report "SIMULATION COMPLEATED" severity failure; 
      
   end process; 
end; 
---------------------------------------------------------------------------------------------------
