---------------------------------------------------------------------------------------------------
--! @brief  Read pointer module
--! @details The module generates read pointer in grey code. With synchronised write pointer to 
--! read clock domain asserts empty signal if read pointer cathces write pointer. Lower AWIDTH_G -1
--! bits from read pointer are for FIFO memory address read.
--! 
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Jan 17 2018 created
--! @date Feb 22 2019 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, Jan 17 2018: Created
--! jkokalj, Feb 22 2019: design iteration
--!
--! @file CslRptrEmpty.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Read pointer module
--! @details The module generates read pointer in grey code. With synchronised write pointer to 
--! read clock domain asserts empty signal if read pointer cathces write pointer. Lower AWIDTH_G -1
--! bits from read pointer are for FIFO memory address read.
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
entity CslRptrEmpty is
   generic(
      TPD_G          : time   := 1 ns;
      AWIDTH_G       : positive:= 4                   --! Address width
   );
   Port ( 
      rclk_i         : in  sl;                        --!read local clock bus
      rrst_i         : in  sl;                        --!read reset signal bus  
      
      rinc_i         : in  sl;                        --!read pointer increment 
      rsync_wptr_i   : in  slv(AWIDTH_G  downto 0);   --! synchronised write pointer to read domain
      
      rptr_o         : out slv(AWIDTH_G  downto 0);   --! read pointer in grey output
      raddr_o        : out slv(AWIDTH_G -1 downto 0); --! read address signal bus
      rempty_o       : out sl;                        --! signal FIFO is empty => RD pointer caches WR pointer
      rcnt_o         : out slv(AWIDTH_G downto 0);    --! fifo fullness counter
      unf_o          : out sl);                       --! underflow flag => pulse when reading out of empty fifo
   end CslRptrEmpty;
---------------------------------------------------------------------------------------------------
architecture rtl of CslRptrEmpty is

   --! Record containing all register elements
   type reg_type is record
      unf         : sl;                            --! underflow flag signal
      empty       : sl;                            --! FIFO empty signal bus
      rcnt        : unsigned(AWIDTH_G downto 0);   --! FIFO words counter
      rbin        : unsigned(AWIDTH_G downto 0);   --! binary pointer signal bus
      rptr        : unsigned(AWIDTH_G downto 0);   --! grey pointer signal bus
   end record reg_type;
   
   --! Output of registers
   signal r: reg_type;
   
   --! Combinatorial input to registers
   signal rin: reg_type;
   
   --! Initial and reset values for all register elements
   constant REG_INIT_C : reg_type := (
      unf         => '0',
      empty       => '1',
      rcnt        => (others => '0'),
      rbin        => (others => '0'),
      rptr        => (others => '0'));      
      
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   --! @param[in]  rrst_i, rinc_i, rsync_wptr_i, r
   --! @param[out] rin, rempty_o, raddr_o, rptr_o
   comb : process(rrst_i, rinc_i, rsync_wptr_i, r) -- combinational process
   variable v : reg_type;
   begin
      v                 := r; -- default assignment
      v.unf             := '0';
      
      if rinc_i ='1' then
         if r.empty = '0' then
            --! pointer increment
            v.rbin      := r.rbin + 1;  
         else
            v.rbin      := r.rbin;
            v.unf       := '1';        --! read pointer increment on empty fifo => underflow
         end if;
      end if;
      
      v.rptr            := grayEncode(v.rbin); --! function to convert binary => grey
      
      v.rcnt            := unsigned(grayDecode(rsync_wptr_i)) - r.rbin; --! fifo fullness counter
      
      if rsync_wptr_i = slv(v.rptr) then
         --! next read pointer is equal to write pointer => fifo is empty
         v.empty        := '1';
      else
         v.empty        := '0';
      end if;
            
      if rrst_i = '1' then -- reset condition
         v              := REG_INIT_C;
      end if;
      
      rin               <= v; -- drive register inputs
      -- drive outputs
      rempty_o          <= r.empty;
      unf_o             <= r.unf;
      rcnt_o            <= slv(v.rcnt);
      raddr_o           <= slv(r.rbin(AWIDTH_G -1 downto 0)); 
      rptr_o            <= slv(r.rptr);
   end process comb;
 
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   --! @param[in]  rin, clk_i
   --! @param[out] r 
   seq : process(rclk_i) -- sequential process
   begin
      if rising_edge(rclk_i) then 
         r <= rin after TPD_G; 
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------