---------------------------------------------------------------------------------------------------
--! @brief  Shift register module test bench
--! @details   The test bench simulates the module with various input signals value,
--! lenghts and timing.
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date January 11 2018 created
--! @date March 28 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, January 11 2018: Created
--! jkokalj, March 28 2018: design iteration
--!
--! @file CslShiftRegTb.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Shift register module test bench
--! @details   The test bench simulates the module with various input signals value,
--! lenghts and timing. 
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
ENTITY CslAsyncFIFOTb IS
   generic(
      TPD_G          : time     := 1 ns;
      AWIDTH_G       : positive := 4;     --! Address width
      DWIDTH_G       : positive := 8;     --! Data width
      PIPELINE_G     : natural  := 1;     --! delay in clock cycles of output data
      WRITE_FIRST_G  : boolean  := false;
      BRAM_G         : boolean  := false   --! false => LUT RAM, true => BRAM
   );
END CslAsyncFIFOTb;

ARCHITECTURE behavioral of CslAsyncFIFOTb is
   --Inputs
   signal wclk_i : std_logic := '0';
   signal wrst_i : std_logic := '0';
   signal winc_i : std_logic := '0';
   signal wdata_i : std_logic_vector(7 downto 0) := (others => '0');
   signal rclk_i : std_logic := '0';
   signal rrst_i : std_logic := '0';
   signal rinc_i : std_logic := '0';

 	--Outputs
   signal wfull_o : std_logic;
   signal rdata_o : std_logic_vector(7 downto 0);
   signal rempty_o : std_logic;
 
   constant wclk_period_C : time := 10 ns;
   constant rclk_period_C : time := 13 ns;
 
BEGIN
   -- Instantiate the Unit Under Test (UUT)
   uut:entity work.CslAsyncFIFO 
      generic map(
         TPD_G          => TPD_G,
         AWIDTH_G       => AWIDTH_G,
         DWIDTH_G       => DWIDTH_G,
         PIPELINE_G     => PIPELINE_G,
         WRITE_FIRST_G  => WRITE_FIRST_G,
         BRAM_G         => BRAM_G
      )
      PORT MAP (
         wclk_i => wclk_i,
         wrst_i => wrst_i,
         winc_i => winc_i,
         wdata_i => wdata_i,
         wfull_o => wfull_o,
         rclk_i => rclk_i,
         rrst_i => rrst_i,
         rinc_i => rinc_i,
         rdata_o => rdata_o,
         rempty_o => rempty_o
      );

   --! @brief Write Clock procces 
   --! @details Clock signal generator
   --! @param[in]  clk_i_period_C
   --! @param[out] clk_i
   p_ClkWGen :process
   begin
		wclk_i <= '0';
		wait for wclk_period_C/2;
		wclk_i <= '1';
		wait for wclk_period_C/2;
   end process;
   
   --! @brief Reset procces 
   --! @details Reset signal generator
   --! @param[in]  clk_i_period_C
   --! @param[out] rst_i
   p_RstRGen :process
   begin
		rrst_i <= '1';
      --rrst_T <= '0';
		wait for rclk_period_C*500;
		rrst_i <= '0';
      --rrst_T <= '1';
		wait for rclk_period_C*1500;
   end process;
   
   --! @brief Clock procces 
   --! @details Clock signal generator
   --! @param[in]  clk_i_period_C
   --! @param[out] clk_i
   p_ClkRGen :process
   begin
		rclk_i <= '0';
		wait for rclk_period_C/2;
		rclk_i <= '1';
		wait for rclk_period_C/2;
   end process;
   
   --! @brief Reset procces 
   --! @details Reset signal generator
   --! @param[in]  clk_i_period_C
   --! @param[out] rst_i
   p_RstWGen :process
   begin
		wrst_i <= '1';
      --wrst_T <= '0';
		wait for wclk_period_C*500;
		wrst_i <= '0';
      --wrst_T <= '1';
		wait for wclk_period_C*1500;
   end process;   
   
   --! @brief Simulation procces 
   --! @details Simulation with various signals value, length and timings
   --! @param[in] 
   --! @param[out] data_i, load_i 
   p_SimR: process
   begin		
      wait for 4*rclk_period_C;
      rinc_i <= '1';
      wait for 2*rclk_period_C ;	
      rinc_i <= '0';
      wait for 4*rclk_period_C ;
      rinc_i <= '1';
      wait for 8*rclk_period_C ;
      rinc_i <= '0';
      wait for 6*rclk_period_C;	
      rinc_i <= '1';
      wait for 4*rclk_period_C ;
      rinc_i <= '0';
      wait for 2*rclk_period_C ;
      rinc_i <= '1';
      wait for 2*rclk_period_C;	
      rinc_i <= '0';
      wait for 2*rclk_period_C ;
      rinc_i <= '1';
      wait for 8*rclk_period_C;	
      rinc_i <= '0';
   end process;
   
   --! @brief Simulation procces 
   --! @details Simulation with various signals value, length and timings
   --! @param[in] 
   --! @param[out] data_i, load_i 
   p_SimW: process
   begin		
      wait for 4*wclk_period_C ;
      wdata_i <= "11110000";
      winc_i <= '1';
      wait for 1*wclk_period_C ;	
      winc_i <= '0';
      wait for 1*wclk_period_C ;
      wdata_i <= "11010000";      
      winc_i <= '1';
      wait for 1*wclk_period_C ;
      wdata_i <= "11010110";      
      winc_i <= '0';
      wait for 1*wclk_period_C ;	
      winc_i <= '1';
      wait for 1*wclk_period_C ;	
      winc_i <= '0';
      wait for 1*wclk_period_C ;
      wdata_i <= "11010100";
      winc_i <= '1';
      wait for 1*wclk_period_C ;	
      winc_i <= '0';
      wait for 1*wclk_period_C ;
      wdata_i <= "10101000";      
      winc_i <= '1';
      wait for 1*wclk_period_C ;
      wdata_i <= "11011110";      
      winc_i <= '0';
      wait for 1*wclk_period_C ;	
      winc_i <= '1';
      wait for 1*wclk_period_C ;	
      winc_i <= '0';
   end process;

END;
