---------------------------------------------------------------------------------------------------
--! @brief  Falling/Rising Edge Detection module
--! @details The module is detecting Rising or Falling edge of signal with no aditional 
--! clock cylces delay. When direction generic DIR_G = FALSE, then rising edge detection is selected. 
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date January 8 2018 created
--! @date March 28 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, January 8 2018: Created
--! jkokalj, January 9 2018: direction with boolean generic
--! jkokalj, March 28 2018: design iteration
--!
--! @file CslEdgeDet.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Falling/Rising Edge Detection module
--! @details The module is detecting Rising or Falling edge of signal with no aditional 
--! clock cylces delay. When direction generic DIR_G = FALSE, then rising edge detection is selected.
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
entity CslEdgeDet is
   generic
   (
      TPD_G             : time     := 1 ns;
      -- set the direction
      --! FALSE(default) = rising edge detection, 
      --! TRUE = falling edge detection
      FALLING_EDGE_G : boolean   := FALSE
   );
   port ( 
      -- global input signals
      clk_i : in  sl;   --! input clock bus
      rst_i : in  sl;   --! Input reset bus
      sig_i : in  sl;   --! Input signal bus
      -- global output signal
      sig_o : out  sl); --! Output signal bus
end CslEdgeDet;
---------------------------------------------------------------------------------------------------
architecture rtl of CslEdgeDet is 
   --! internal signal
   signal dff : sl;
---------------------------------------------------------------------------------------------------
begin

   --! @brief Edge detection process
   --! @details Falling/rising edge detection with synchronouse reset.
   --! @param[in]  clk_i, rst_i, dff , sig_i
   --! @param[out] sig_o 
   p_EdgeDet:  process(clk_i)
   begin
      if rising_edge(clk_i) then
         if rst_i = '1' then 
            --synchronous output reset
            sig_o <= '0' after TPD_G;
         else
            -- normal operation
            if FALLING_EDGE_G = true then
               sig_o <= dff and not(sig_i) after TPD_G; -- detection of falling edge
            else
               sig_o <= sig_i and not(dff) after TPD_G; -- detection of rising edge
            end if;
         end if;
      end if;
   end process p_EdgeDet;
   
   --! @brief D flip flop procces 
   --! @details Shifting input signal for one clock with D flip flop
   --! @param[in]  clk_i, sig_i
   --! @param[out] dff 
   p_Dff:  process(clk_i)
   begin
      if rising_edge(clk_i) then -- synchronous process
         dff <= sig_i after TPD_G;
      end if;
   end process p_Dff;

end rtl;
---------------------------------------------------------------------------------------------------
