---------------------------------------------------------------------------------------------------
--! @brief  Frequency divider module
--! @details The module is dividing input frequency with counter. Output frequency is input frequency 
--! divided by integer freqDiv_i. On odd integers value the generic ODD_HIGH_LONGER_G = true  
--! makes high polarity of the output clock longer else th low polarity of the clock is longer.
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date March 3 2018 created
--! @date March 28 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, March 3 2018: Created
--! jkokalj, March 28 2018 last modify
--!
--! @file CslFreqDiv.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use ieee.std_logic_arith.all;
use work.CslStdRtlPkg.all;

--! @brief  Frequency divider module
--! @details The module is dividing input frequency with counter. Output frequency is input frequency 
--! divided by integer freqDiv_i. On odd integers value the generic ODD_HIGH_LONGER_G = true  
--! makes high polarity of the output clock longer else the low polarity of the clock is longer.
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
entity CslFreqDiv is
   generic (
      TPD_G             : time      := 1 ns;
      ODD_HIGH_LONGER_G : boolean   := false;
      WIDTH_G           : positive  := 32
      );
   port(
      clk_i          : in  sl;   --! input clock bus
      rst_i          : in  sl;   --! input reset bus
      freqDiv_i      : in  slv(WIDTH_G-1 downto 0); --! input vector for frequency divide / period extend
      locClk_o       : out sl;   --! output clock bus
      locClkPulse_o  : out sl    --! output clock pulse bus
   );
end entity CslFreqDiv;

architecture rtl of CslFreqDiv is
   
   --! Record containing all register elements
   type RegType is record
      cnt         : slv(WIDTH_G-1 downto 0);
      locClk      : sl;
   end record RegType;
   
   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      cnt      => (others => '0'),
      locClk   => '0');
   
   --! Output of registers
   signal r          : RegType := REG_INIT_C;
   --! Combinatorial input to registers
   signal rin        : RegType;
   
   --! internal signal
   signal locClk_pi  : sl;
   
begin

   -- Edge detection component for pulse generation
   U_pulse : entity work.CslEdgeDet 
      generic map(
         TPD_G => TPD_G,
         FALLING_EDGE_G => FALSE
      )
      port map(
         clk_i => clk_i,
         rst_i => rst_i,
         sig_i => locClk_pi,
         sig_o => locClkPulse_o
      );
    
   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   --! @param[in]   rst_i, r, per_i
   --! @param[out] rin, locClk_o, pulseClk
   comb : process (r, rst_i, freqDiv_i, clk_i) is -- combinational process
      variable v : RegType;
   begin
      v := r; -- default assignment
            
      v.cnt := r.cnt + 1;
            
      if (r.cnt = freqDiv_i(WIDTH_G-1 downto 1)-1) then 
         if ODD_HIGH_LONGER_G = true then
            v.locClk := '1';
         else
            v.locClk := '0';
         end if;
      elsif (r.cnt >= freqDiv_i(WIDTH_G-1 downto 0) - 1) then 
         v.cnt    := (others => '0');
         v.locClk := not(r.locClk);-- invert output clock register on period/2
      end if;

      if (rst_i = '1') then -- reset condition
         v := REG_INIT_C;
      end if;

      rin         <= v; -- drive register inputs
      
      -- drive outputs
      if freqDiv_i < slv(to_unsigned(2,32)) then
         locClk_o    <= clk_i;
         locClk_pi   <= clk_i;
      else
         locClk_o    <= r.locClk;
         locClk_pi   <= r.locClk;
      end if;
      
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   --! @param[in]  rin, clk_i
   --! @param[out] r 
   seq : process (clk_i) is -- sequential process
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end architecture rtl;
---------------------------------------------------------------------------------------------------