---------------------------------------------------------------------------------------------------
--! @brief  Clock correction module
--! @details
--!   The module is inserting clock correction symbols on given period. The symbols are generic and
--!   clock correction period is real time configurable. Clock correction symbols are inserted 
--!   if the fiberLink is in Idle. 
--!   Modification: 
--!   If the link is not in Idle the clock correction character is 
--!   injected as soon as the link is Idle again. 
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Dec 27 2018 created
--! @date Dec 19 2019 last modify
--!
--! @version v0.2
--!
--! @file ClkCorrTx.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use work.CslStdRtlPkg.all;
use work.FiberPkg.all;
use ieee.numeric_std.all;
library UNISIM;
use UNISIM.VComponents.all;
---------------------------------------------------------------------------------------------------
entity ClkCorrTx is
   generic (
      TPD_G          : time            := 1 ns;
      CLK_CORR_2_G   : slv(8 downto 0) := '1' & x"FD"; --! clock correction symbol 2
      CLK_CORR_1_G   : slv(8 downto 0) := '1' & x"BC"; --! clock correction symbol 1
      CLK_CORR_PER_G : positive        := 2500         --! clock correction period
   );
   port (
      clk_i        : in sl;                  --! clock signal bus
      rst_i        : in sl;                  --! reset signal bus
      en_i         : in sl;                  --! module enable signal 
      idle_i       : in sl;                  --! idle signal bus
      clkCorrPer_i : in slv(15 downto 0);    --! period vector signal bus
      gtTx_i       : in GtType := GT_INIT_C; --! input data

      gtTx_o : out GtType := GT_INIT_C --! output data
   );
end ClkCorrTx;

---------------------------------------------------------------------------------------------------
architecture rtl of ClkCorrTx is

   --! Record containing all register elements
   type RegType is record
      clkCorCnt : unsigned(15 downto 0); --! clock correction period counter
      gtTx      : GtType;                --! pipelined data
      insert    : sl;                    --! The SR register that indicates that the character has to be inserted  
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
         clkCorCnt => (others => '0'),
         gtTx      => GT_INIT_C,
         insert    => '0'
      );

   --! internal signals
   signal r   : RegType := REG_INIT_C;
   signal rin : RegType;
---------------------------------------------------------------------------------------------------
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, gtTx_i, en_i, clkCorrPer_i, idle_i) is
      variable v : RegType;
   begin
      v := r;

      --! registered input data
      v.gtTx := gtTx_i;

      if r.clkCorCnt > 0 then
         --! down counter
         v.clkCorCnt := r.clkCorCnt - 1;
      else
         --! reseting counter to value on input port if reached zero
         v.clkCorCnt := unsigned(clkCorrPer_i);
      end if;

      if r.clkCorCnt = 0 then
         --! counter has reached zero => inserting clock correction symbols 
         v.insert := '1';
      end if;

      --! Insert the clock correction characters if the stream is IDLE
      --! If not IDLE we wait for the end of the data packet and inset then
      --! TODO: Check if we need to sync the counter as well
      if idle_i = '1' and r.insert = '1' then
         v.gtTx.data := CLK_CORR_2_G(7 downto 0) & CLK_CORR_1_G(7 downto 0);
         v.gtTx.char := CLK_CORR_2_G(8) & CLK_CORR_1_G(8);
         v.insert    := '0';
      end if;

      --! Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      rin <= v;

      --! Output assignment
      if(en_i = '1')then
         gtTx_o <= r.gtTx;
      else
         gtTx_o <= gtTx_i;
      end if;
   end process comb;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end;
---------------------------------------------------------------------------------------------------