---------------------------------------------------------------------------------------------------
--! @brief     Fiber Link Map receiver
--! @details   
--!    MAP Stands for MedAustron Piped 
--!    The protocol features one real-time high priority pipe and N number of low priority pipes.
--!    The high priority and low priority pipes are completely separated into two independent 8-bit 
--!    communication data channels. The low priority data (LPD) is transferred over the LSB 8-bits 
--!    and the high priority data (HPD) over the MSB 8-bits, of the 16-bit GT interface. 
--!    The low-level communication modules FiberLinkRx are identical for HPD and LPD.  
--!    
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 1 2018 created
--! @date Aug 27 2019 last modify
--!
--! @version v0.1
--!
--! @file FiberLinkMapRx.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
---------------------------------------------------------------------------------------------------
entity FiberLinkMapRx is
   generic (
      TPD_G              : time     := 1 ns;               --! Simulation delta time offset
      --! HPD Generics
      HPD_DATA_WIDTH_G   : positive  := 4;                 --! Input axi stream width number of bytes (1,2,4,8,16)
      HPD_IDLE_G         : slv(8 downto 0) := '0' & x"00"; --! Sent out at when IDLE_G
      HPD_SOF_CHAR_G     : slv(7 downto 0) := x"1C";       --! Start of frame character
      HPD_EOF_CHAR_G     : slv(7 downto 0) := x"FE";       --! End of frame character
      HPD_MAX_SIZE_G     : positive := 8;                  --! Maximum number of input axis data words
      HPD_TWO_CLOCKS_G   : boolean  := false;              --! Enable optional secondary clock and reset inputs
      
      --! HPD Generics
      LPD_DATA_WIDTH_G   : positive  := 4;                 --! Input axi stream width number of bytes (1,2,4,8,16)
      LPD_PIPES_G        : integer range 1 to 255 := 1;    --! Number of priority pipes (For HPD set to 1)        
      LPD_IDLE_G         : slv(8 downto 0) := '1' & x"BC"; --! Sent out at when IDLE_G
      LPD_SOF_CHAR_G     : slv(7 downto 0) := x"1C";       --! Start of frame character
      LPD_EOF_CHAR_G     : slv(7 downto 0) := x"FE";       --! End of frame character
      LPD_MAX_SIZE_G     : positive := 8;                  --! Maximum number of input axis data words 
      LPD_TWO_CLOCKS_G   : boolean  := false               --! Enable optional secondary clock and reset inputs
   );
   port (
      clk_i         : in  sl;
      rst_i         : in  sl;
      
      --! Fiber side interface connect directly to GT
      gtDataRx_i    : in slv(15 downto 0);
      gtCharRx_i    : in slv(1 downto 0);
      link_i        : in sl; 
      
      --! User side interface
      
      --! HPD pipe
      hpdRxAxisMaster_o : out AxiStreamMasterType;
      hpdRxAxisSlave_i  : in  AxiStreamSlaveType;      
      hpdRxPktEmpty_o   : out sl;
      hpdRxDrop_o       : out sl;
      hpdSysClk_i       : in  sl := '0';    --! optional secondary clock used for clock domain crossing
      hpdSysRst_i       : in  sl := '0';    --! optional secondary reset used for clock domain crossing

      --! HPD pipe
      lpdRxAxisMaster_o : out AxiStreamMasterArray(LPD_PIPES_G-1 downto 0);
      lpdRxAxisSlave_i  : in  AxiStreamSlaveArray(LPD_PIPES_G-1 downto 0);      
      lpdRxPktEmpty_o   : out slv(LPD_PIPES_G-1 downto 0);
      lpdRxDrop_o       : out sl;
      lpdSysClk_i       : in  sl := '0';    --! optional secondary clock used for clock domain crossing
      lpdSysRst_i       : in  sl := '0'     --! optional secondary reset used for clock domain crossing
   );
end FiberLinkMapRx;
---------------------------------------------------------------------------------------------------
architecture rtl of FiberLinkMapRx is
---------------------------------------------------------------------------------------------------
begin

   --! High priority data pipe
   u_HpdFiberLinkRx: entity work.FiberLinkRx
      generic map (
         TPD_G        => TPD_G,
         DATA_WIDTH_G => HPD_DATA_WIDTH_G,
         PIPES_G      => 1,
         IDLE_G       => HPD_IDLE_G,
         SOF_CHAR_G   => HPD_SOF_CHAR_G,
         EOF_CHAR_G   => HPD_EOF_CHAR_G,
         MAX_SIZE_G   => HPD_MAX_SIZE_G,
         TWO_CLOCKS_G => HPD_TWO_CLOCKS_G)
      port map (
         clk_i           => clk_i,
         rst_i           => rst_i,
         sysClk_i        => hpdSysClk_i,
         sysRst_i        => hpdSysRst_i,
         dataRx_i        => gtDataRx_i(15 downto 8),
         charRx_i        => gtCharRx_i(1),
         link_i          => link_i,
         axisMaster_o(0) => hpdRxAxisMaster_o,
         axisSlave_i(0)  => hpdRxAxisSlave_i,
         pktEmpty_o(0)   => hpdRxPktEmpty_o,
         drop_o          => hpdRxDrop_o);

   --! Low priority data pipes
   u_LpdFiberLinkRx: entity work.FiberLinkRx
      generic map (
         TPD_G        => TPD_G,
         DATA_WIDTH_G => LPD_DATA_WIDTH_G,
         PIPES_G      => LPD_PIPES_G,
         IDLE_G       => LPD_IDLE_G,
         SOF_CHAR_G   => LPD_SOF_CHAR_G,
         EOF_CHAR_G   => LPD_EOF_CHAR_G,
         MAX_SIZE_G   => LPD_MAX_SIZE_G,
         TWO_CLOCKS_G => LPD_TWO_CLOCKS_G)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         sysClk_i     => lpdSysClk_i,
         sysRst_i     => lpdSysRst_i,
         dataRx_i     => gtDataRx_i(7 downto 0),
         charRx_i     => gtCharRx_i(0),
         link_i       => link_i,
         axisMaster_o => lpdRxAxisMaster_o,
         axisSlave_i  => lpdRxAxisSlave_i,
         pktEmpty_o   => lpdRxPktEmpty_o,
         drop_o       => lpdRxDrop_o);

end rtl;
---------------------------------------------------------------------------------------------------