---------------------------------------------------------------------------------------------------
--! @brief     Fiber Link Map transmitter
--! @details   
--!    MAP Stands for MedAustron Piped 
--!    The protocol features one real-time high priority pipe and N number of low priority pipes.
--!    The high priority and low priority pipes are completely separated into two independent 8-bit 
--!    communication data channels. The low priority data (LPD) is transferred over the LSB 8-bits 
--!    and the high priority data (HPD) over the MSB 8-bits, of the 16-bit GT interface. 
--!    The low-level communication modules FiberLinkTx are identical for HPD and LPD. 
--!
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Aug 27 2019 last modify
--!
--! @version v0.1
--!
--! @file FiberLinkMapTx.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
---------------------------------------------------------------------------------------------------
entity FiberLinkMapTx is
   generic (
      TPD_G              : time     := 1 ns;                    --! Simulation delta time offset
      --! HPD Generics 
      HPD_DATA_WIDTH_G   : positive := 4;                       --! Input axi stream width number of bytes (1,2,4,8,16)
      HPD_IDLE_G         : slv(8 downto 0) := '0' & x"00";      --! Sent out at when IDLE_G
      HPD_SOF_CHAR_G     : slv(7 downto 0) := x"1C";            --! Start of frame character
      HPD_EOF_CHAR_G     : slv(7 downto 0) := x"FE";            --! End of frame character
      HPD_MAX_SIZE_G     : positive := 8;                       --! Maximum number of input axis data words 
      HPD_TWO_CLOCKS_G   : boolean  := false;                   --! Enable optional secondary clock and reset inputs
                                                                
      --! LPD Generics
      LPD_DATA_WIDTH_G   : positive := 4;                       --! Input axi stream width number of bytes (1,2,4,8,16)
      LPD_PIPES_G        : positive := 1;                       --! Number of priority pipes (For HPD this is set to 1)
      LPD_IDLE_G         : slv(8 downto 0) := '1' & x"BC";      --! Sent out at when IDLE_G
      LPD_SOF_CHAR_G     : slv(7 downto 0) := x"1C";            --! Start of frame character
      LPD_EOF_CHAR_G     : slv(7 downto 0) := x"FE";            --! End of frame character
      LPD_MAX_SIZE_G     : positive := 8;                       --! Maximum number of input axis data words
      LPD_TWO_CLOCKS_G   : boolean  := false                    --! Enable optional secondary clock and reset inputs
   );
   port (
      clk_i         : in  sl;
      rst_i         : in  sl;

      --! User side interface
      
      --! HPD pipe
      hpdTxAxisMaster_i : in  AxiStreamMasterType;
      hpdTxAxisSlave_o  : out AxiStreamSlaveType;
      hpdTxPktFull_o    : out sl;
      hpdTxErr_o        : out slv(1 downto 0);
      hpdIdle_o         : out sl;
      hpdSysClk_i       : in  sl := '0';    --! optional secondary clock used for clock domain crossing
      hpdSysRst_i       : in  sl := '0';    --! optional secondary reset used for clock domain crossing
            
      -- LPD pipes
      lpdTxAxisMaster_i : in  AxiStreamMasterArray(LPD_PIPES_G-1 downto 0);
      lpdTxAxisSlave_o  : out AxiStreamSlaveArray(LPD_PIPES_G-1 downto 0);      
      lpdTxPktFull_o    : out slv(LPD_PIPES_G-1 downto 0);
      lpdTxErr_o        : out Slv2Array(LPD_PIPES_G-1 downto 0);
      lpdIdle_o         : out sl;
      lpdSysClk_i       : in  sl := '0';    --! optional secondary clock used for clock domain crossing
      lpdSysRst_i       : in  sl := '0';    --! optional secondary reset used for clock domain crossing
      
      --! Fiber side interface
      gtDataTx_o    : out slv(15 downto 0);
      gtCharTx_o    : out slv(1 downto 0)
   );
end FiberLinkMapTx;
---------------------------------------------------------------------------------------------------
architecture rtl of FiberLinkMapTx is
---------------------------------------------------------------------------------------------------
begin

   --! High priority data pipe
   u_HpdFiberLinkTx: entity work.FiberLinkTx
      generic map (
         TPD_G        => TPD_G,
         DATA_WIDTH_G => HPD_DATA_WIDTH_G,
         PIPES_G      => 1,              --! Fixed to 1 because of high priority
         IDLE_G       => HPD_IDLE_G,
         SOF_CHAR_G   => HPD_SOF_CHAR_G,
         EOF_CHAR_G   => HPD_EOF_CHAR_G,
         MAX_SIZE_G   => HPD_MAX_SIZE_G,
         TWO_CLOCKS_G => HPD_TWO_CLOCKS_G)
      port map (
         clk_i           => clk_i,
         rst_i           => rst_i,
         sysClk_i        => hpdSysClk_i,
         sysRst_i        => hpdSysRst_i,
         axisMaster_i(0) => hpdTxAxisMaster_i,
         axisSlave_o(0)  => hpdTxAxisSlave_o,
         txPktFull_o(0)  => hpdTxPktFull_o,
         txErr_o(0)      => hpdTxErr_o,
         dataTx_o        => gtDataTx_o(15 downto 8),
         charTx_o        => gtCharTx_o(1),
         idle_o          => hpdIdle_o);

   --! Low priority data pipes
   u_LpdFiberLinkTx: entity work.FiberLinkTx
      generic map (
         TPD_G        => TPD_G,
         DATA_WIDTH_G => LPD_DATA_WIDTH_G,
         PIPES_G      => LPD_PIPES_G,
         IDLE_G       => LPD_IDLE_G,
         SOF_CHAR_G   => LPD_SOF_CHAR_G,
         EOF_CHAR_G   => LPD_EOF_CHAR_G,
         MAX_SIZE_G   => LPD_MAX_SIZE_G,
         TWO_CLOCKS_G => LPD_TWO_CLOCKS_G)
      port map (
         clk_i          => clk_i,
         rst_i          => rst_i,
         sysClk_i       => lpdSysClk_i,
         sysRst_i       => lpdSysRst_i,
         axisMaster_i   => lpdTxAxisMaster_i,
         axisSlave_o    => lpdTxAxisSlave_o,
         txPktFull_o    => lpdTxPktFull_o,
         txErr_o        => lpdTxErr_o,
         dataTx_o       => gtDataTx_o(7 downto 0),
         charTx_o       => gtCharTx_o(0),
         idle_o         => lpdIdle_o);

end rtl;
---------------------------------------------------------------------------------------------------