---------------------------------------------------------------------------------------------------
--! @brief  Fiber Link transmitter module 
--! @details
--!   The transmitter module accepts packets on PIPES_G number of AXI4-Streams
--!   and converts them to fiber messages. The AXI4-Stream packets from PIPES_G number 
--!   of pipes are received by PIPES_G number of AXI4-Stream FIFOs. 
--!   The packets are held within the FIFOs until the control logic txFsm schedules them for transmission. 
--!   Within the FIFO the packet is converted from DATA_WIDTH to 8-bit wide AXI4-Stream. 
--!   The control logic TxFsm reads the scheduled packet from the FIFO, 
--!   adds the packet delimiter characters, pipe number, and CRC, and transmits the message to the fiber.
--!   
--!   txPktFull_o - The user can observe the pktFull_o for simplified handshaking. 
--!                 The pktFull_o signals the user in the same way as the normal FIFO almost full signal.
--!                 Since the pktFull_o is ‘Low’ the user can be confident that the FIFO has space for at 
--!                 least one packet of MAX_SIZE before the tReady drops.
--!   txErr_o(0)  - urnErr - Underrun error indicates that the transmission from the user is too slow. 
--!                 If the tValid drops before the tLast is received.
--!   txErr_o(1)  - lenErr - Length error indicates if the packet is longer than the MAX_SIZE.
--!
--!   Two different clocks can be used to drive the CslAxisFifo for the purpose of clock domain crossing
--!   synchronization. This feature is enabled by setting TWO_CLOCKS_G to "true". If enabled, clk_i and 
--!   rst_i are used for the output side of the fifo and sysClk_i and sysRst_i are used for the input side.
--!   If this feature is disabled, clk_i and rst_i are used for driving both sides of the fifo.
--!     
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Aug 27 2019 last modify
--!
--! @version v0.1
--!
--! @file FiberLinkTx.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
---------------------------------------------------------------------------------------------------
entity FiberLinkTx is
   generic (
      TPD_G        : time                   := 1 ns;        --! Simulation delta time offset
      DATA_WIDTH_G : natural                := 4;           --! Input axi stream width number of bytes (1,2,4,8,16)
      PIPES_G      : integer range 1 to 255 := 1;           --! Number of priority pipes (For HPD set to 1)        
      IDLE_G       : slv(8 downto 0)        := '1' & x"BC"; --! Sent out at when IDLE_G
      SOF_CHAR_G   : slv(7 downto 0)        := x"1C";       --! Start of frame character
      EOF_CHAR_G   : slv(7 downto 0)        := x"FE";       --! End of frame character
      MAX_SIZE_G   : positive               := 8;           --! Maximum number of input axis data words 
      TWO_CLOCKS_G   : boolean  := false                    --! Enable optional secondary clock and reset inputs
   );
   port (
      clk_i         : in sl;
      rst_i         : in sl;
      sysClk_i      : in  sl := '0';    --! optional secondary clock used for clock domain crossing
      sysRst_i      : in  sl := '0';    --! optional secondary reset used for clock domain crossing

      --! User side interface
      -- Tx pipes
      axisMaster_i : in  AxiStreamMasterArray(PIPES_G-1 downto 0);
      axisSlave_o  : out AxiStreamSlaveArray(PIPES_G-1 downto 0);
      txPktFull_o  : out slv(PIPES_G-1 downto 0);
      txErr_o      : out Slv2Array(PIPES_G-1 downto 0);

      --! Fiber side interface connect directly to GT
      dataTx_o : out slv(7 downto 0);
      charTx_o : out sl;
      idle_o   : out sl
   );
end FiberLinkTx;
---------------------------------------------------------------------------------------------------
architecture rtl of FiberLinkTx is
   --! Calculation of the fifo parameters

   --! The fifo should have space for minimum of 4 (MAX_SIZE_G) packets 
   constant FIFO_ADDR_WIDTH_C : positive := ite((bitSize(MAX_SIZE_G) < 4), 5, bitSize(MAX_SIZE_G)+2);
   --! The fifo pause threshold is set to indicate that there is space for one packet   
   constant FIFO_PAUSE_THRESH_C : positive := ((2**FIFO_ADDR_WIDTH_C) - MAX_SIZE_G);
   --! Max size in bytes for the TX FSM
   constant MAX_BYTES_C : positive := DATA_WIDTH_G*MAX_SIZE_G;

   --! Pseudo output is defined because of endian swap    
   signal axisMaster_pi : AxiStreamMasterArray(PIPES_G-1 downto 0);     

   --! Axi stream internal interconnect
   signal intAxisMaster_pi : AxiStreamMasterArray(PIPES_G-1 downto 0);
   signal intAxisSlave_po  : AxiStreamSlaveArray(PIPES_G-1 downto 0);
   signal intAxisMaster_po : AxiStreamMasterArray(PIPES_G-1 downto 0);
   signal intAxisSlave_pi  : AxiStreamSlaveArray(PIPES_G-1 downto 0);

   signal intPause : slv(PIPES_G-1 downto 0);

   --! Secondary clock and reset signals
   signal secClk: sl;
   signal secRst: sl;

   attribute keep : string;
   attribute keep of intAxisMaster_po : signal is "true";
   attribute keep of intAxisSlave_pi  : signal is "true";

---------------------------------------------------------------------------------------------------
begin
   
   --! Wire either one of the clock and reset inputs to input side of fifo
   GEN_ONE_CLK : if (TWO_CLOCKS_G = false) generate
      secClk   <= clk_i;
      secRst   <= rst_i;
   end generate GEN_ONE_CLK;

   GEN_TWO_CLKS : if (TWO_CLOCKS_G = true) generate
      secClk   <= sysClk_i;
      secRst   <= sysRst_i;
   end generate GEN_TWO_CLKS;


   --! PIPES_G number of fifos
   GEN_SYNC_FIFO :
   for i in PIPES_G-1 downto 0 generate

      --! Swap endianness because the axiStreamFifo is Little endian but we need big endian.
      axisMaster_pi(i).tValid                             <= axisMaster_i(i).tValid;
      axisMaster_pi(i).tStrb                              <= axisMaster_i(i).tStrb;
      axisMaster_pi(i).tKeep                              <= axisMaster_i(i).tKeep;
      axisMaster_pi(i).tLast                              <= axisMaster_i(i).tLast;
      axisMaster_pi(i).tDest                              <= axisMaster_i(i).tDest;
      axisMaster_pi(i).tId                                <= axisMaster_i(i).tId;
      axisMaster_pi(i).tUser                              <= axisMaster_i(i).tUser;
      axisMaster_pi(i).tData(127 downto (DATA_WIDTH_G*8)) <= axisMaster_i(i).tData(127 downto (DATA_WIDTH_G*8));
      axisMaster_pi(i).tData((DATA_WIDTH_G*8)-1 downto 0) <= endianSwap(axisMaster_i(i).tData((DATA_WIDTH_G*8)-1 downto 0));

      --! CslAxisFifo module instatiation
      u_AxisFifo : entity work.CslAxisFifo
         generic map (
            TPD_G          => TPD_G,
            PIPELINE_G     => 2,
            BRAM_G         => false,
            AWIDTH_G       => FIFO_ADDR_WIDTH_C,
            THRESHOLD_G    => FIFO_PAUSE_THRESH_C,
            TUSER_NORMAL_G => false,
            TID_WIDTH_G    => 1,
            TDEST_WIDTH_G  => 1,
            TUSER_WIDTH_G  => 2,
            BYTES_SLV_G    => DATA_WIDTH_G,
            BYTES_MST_G    => 1
         )
         port map (
            sAxisClk_i    => secClk,
            sAxisRst_i    => secRst,
            sAxisMaster_i => axisMaster_pi(i),
            sAxisSlave_o  => axisSlave_o(i),
            mAxisClk_i    => clk_i,
            mAxisRst_i    => rst_i,
            mAxisMaster_o => intAxisMaster_po(i),
            mAxisSlave_i  => intAxisSlave_pi(i),
            wrCnt_o       => open,
            pause_o       => intPause(i)
         );
   end generate GEN_SYNC_FIFO;

   --! Axistream interconnect
   intAxisMaster_pi <= intAxisMaster_po;
   intAxisSlave_pi  <= intAxisSlave_po;

   --! TxFsm module instatiation
   u_TxFsm : entity work.TxFsm
      generic map (
         TPD_G      => TPD_G,
         PIPES_G    => PIPES_G,
         IDLE_G     => IDLE_G,
         SOF_CHAR_G => SOF_CHAR_G,
         EOF_CHAR_G => EOF_CHAR_G,
         MAX_SIZE_G => MAX_BYTES_C)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => intAxisMaster_pi,
         axisSlave_o  => intAxisSlave_po,
         err_o        => txErr_o,
         dataTx_o     => dataTx_o,
         charTx_o     => charTx_o,
         idle_o       => idle_o);

   --! Assign the fifo pause to full
   GEN_PIPES :
   for i in PIPES_G-1 downto 0 generate
      txPktFull_o(i) <= intPause(i);
   end generate GEN_PIPES;

end rtl;
---------------------------------------------------------------------------------------------------