---------------------------------------------------------------------------------------------------
--! @brief     Inject error into fiber data module    
--! @details    
--!   This module is for testing purposes. The module can inject CRC, ID and LEN errors into data.
--!    
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Oct 15 2018 created
--! @date Dec 28 2018 last modify
--!
--! @version v0.1
--!
--! @file InjectErrFiber.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
use work.FiberPkg.all;
library UNISIM;
use UNISIM.VComponents.all;
---------------------------------------------------------------------------------------------------
entity InjectErrFiber is
   generic (
      TPD_G             : time      := 1 ns;
      SOF_CHAR_G     : slv(7 downto 0) := x"1C";   --! Start of frame character
      EOF_CHAR_G     : slv(7 downto 0) := x"FE"    --! End of frame character
   );
   port (
      clk_i             : in sl;                   --! clock signal bus
      rst_i             : in sl;                   --! reset signal bus
      en_i              : in sl;                   --! enable module signal
      injId_i           : in sl;                   --! inject ID error
      id_i              : in slv(7 downto 0);      --! ID value to be injected
      injCrc_i          : in sl;                   --! inject CRC error
      injLen_i          : in sl;                   --! inject Length error
      gtTxData_i        : in slv(7 downto 0);      --! data input
      gtTxChar_i        : in sl;                   --! char input
      gtClkCorrEn_i     : in sl;                   --! enable clock correction input signal
      gtCorruptTxData_o : out slv(7 downto 0);     --! data output
      gtCorruptTxChar_o : out sl;                  --! char output
      gtClkCorrEn_o     : out sl                   --! enable clock correction output signal
   );
end InjectErrFiber;
---------------------------------------------------------------------------------------------------
architecture structure of InjectErrFiber is

   --! State type containing all states  
   type StateType is (
      IDLE_S,
      ID_INJ_S,
      CRC_INJ_S,
      LEN_INJ_S
   );
   
   --! Record containing all register elements
   type RegType is record    
      idInjEn     : sl;
      pipeId      : slv(7 downto 0);
      crcInjEn    : sl;
      lenInjEn    : sl;
      dataRx      : Slv8Array(3 downto 0);
      charRx      : slv(3 downto 0);
      clkCorrEn   : slv(3 downto 0);
      dataRxOut   : slv (7 downto 0);
      charRxOut   : sl;
      clkCorrEnOut: sl;
      state       : StateType;
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
      idInjEn     => '0',
      pipeId      => (others => '0'), 
      crcInjEn    => '0',
      lenInjEn    => '0',
      dataRx      => (others => x"00"),
      charRx      => (others => '0'),
      dataRxOut   => (others => '0'),
      charRxOut   => '0',
      clkCorrEnOut=> '0',
      clkCorrEn   => (others => '0'),
      state => IDLE_S
   );
   
   --! internal signals
   signal r      : RegType := REG_INIT_C;
   signal rin    : RegType;
---------------------------------------------------------------------------------------------------  
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, en_i, injId_i, id_i, gtClkCorrEn_i, injCrc_i, injLen_i, gtTxData_i, gtTxChar_i) is
      variable v       : RegType;
      variable vCrcErr : sl;
   begin
      v := r;
      
      --! enable error injection => state progres to error injection
      v.idInjEn   := injId_i;
      v.crcInjEn  := injCrc_i;
      v.lenInjEn  := injlen_i;
      
      --! pipeline input data, char and clock correction
      v.dataRxOut    := r.dataRx(1);
      v.charRxOut    := r.charRx(1);
      v.clkCorrEnOut := r.clkCorrEn(1);
      
      --! Data and char Rx pipeline
      v.dataRx(0) := gtTxData_i;
      v.dataRx(1) := r.dataRx(0);
      v.dataRx(2) := r.dataRx(1); 
      v.dataRx(3) := r.dataRx(2);
      v.charRx(0) := gtTxChar_i;
      v.charRx(1) := r.charRx(0);
      v.charRx(2) := r.charRx(1); 
      v.charRx(3) := r.charRx(2);
      v.clkCorrEn(0) := gtClkCorrEn_i;
      v.clkCorrEn(1) := r.clkCorrEn(0);
      v.clkCorrEn(2) := r.clkCorrEn(1); 
      v.clkCorrEn(3) := r.clkCorrEn(2);
      
      --! State Machine
      case r.state is
         ----------------------------------------------------------------------
         when IDLE_S =>
            --! Next state
            --! Look for the idInjEn signal. Proceed only if enable is ok.
            if (r.idInjEn = '1' and en_i = '1') then
               v.pipeId    := id_i;
               v.state     := ID_INJ_S;
            end if;
            
            --! Next state
            --! Look for the crcInjEn signal. Proceed only if enable is ok.
            if (r.crcInjEn = '1' and en_i = '1') then
               v.state := CRC_INJ_S;
            end if;
            
            --! Next state
            --! Look for the crcInjEn signal. Proceed only if enable is ok.
            if (r.lenInjEn = '1' and en_i = '1') then
               v.state := LEN_INJ_S;
            end if;
         ----------------------------------------------------------------------
         when ID_INJ_S =>         
            --! inject ID error
            if (r.dataRx(2) = SOF_CHAR_G and r.charRx(2) = '1') then
               v.dataRxOut := r.pipeId;
               v.charRxOut := r.charRx(1);
               v.state := IDLE_S;
            end if;
         ----------------------------------------------------------------------
         when CRC_INJ_S =>         
            --! inject CRC error
            if (r.dataRx(3) = SOF_CHAR_G and r.charRx(3) = '1') then
               v.dataRxOut := r.dataRx(1)(7 downto 1) & not(r.dataRx(1)(0));
               v.charRxOut := r.charRx(1);
               v.state := IDLE_S;
            end if;
         ----------------------------------------------------------------------
         when LEN_INJ_S =>         
            --! inject Length error
            if (r.dataRx(1) = EOF_CHAR_G and r.charRx(1) = '1') then
               v.dataRxOut := r.dataRx(2);
               v.charRxOut := r.charRx(2);
               v.state := IDLE_S;
            end if;
         ----------------------------------------------------------------------
         when others =>
            --! This state should not be reached
            --! Initialize
            v := REG_INIT_C;
         ----------------------------------------------------------------------      
      end case;
      
      -- Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      rin <= v;
      
      --! Output assignment
      gtCorruptTxData_o <= r.dataRxOut;
      gtCorruptTxChar_o <= r.charRxOut;
      gtClkCorrEn_o     <= r.clkCorrEnOut;
   -----------------------------------------------
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end structure;
---------------------------------------------------------------------------------------------------