---------------------------------------------------------------------------------------------------
--! @brief   Message receiver
--! @details
--!    Receives the message from AxiStream and stores it into 32-bit wide message memory
--!    strobe_o signal indicates that the message was received.
--!    crcErr_o and lenErr_o indicate the length and CRC errors at the message reception.
--!    
--!    Note: That the massage size has to be exactly SIZE_G words, otherwise the lenErr_o is indicated.
--!    Note: When strobe suppress generic STR_SUP_G is TRUE the strobe_o will not be asserted when errors occur.
--!    Note: The errors should be monitored by external rising edge triggered status counters   
--!    Note: Use 'true' to add a register if necessary that the output message is static until the next strobe.
--!          If the message is registered externally use 'false'.
--!
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Sep 20 2018 last modify
--!
--! @version v0.1
--!
--!
--! @file MsgStreamRx.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;

--! @brief  
--! @details 
---------------------------------------------------------------------------------------------------
entity MsgStreamRx is
generic (
      TPD_G      : time       := 1 ns;    --! Simulation delta time offset
      SIZE_G     : positive   := 8;       --! Message size in a number of 32-bit words
      STR_SUP_G  : boolean    := true;    --! Suppress the strobe when error
      REGISTER_G : boolean    := false    --! Register the output massage
   );
   port (
      clk_i         : in  sl;
      rst_i         : in  sl;

      --! Input stream (the stream is always ready)
      axisMaster_i : in  AxiStreamMasterType;
      axisSlave_o  : out AxiStreamSlaveType:= AXI_STREAM_SLAVE_FORCE_C;

      --! Parallel message output
      strobe_o     : out sl;
      msg32Array_o : out Slv32Array(SIZE_G-1 downto 0);
      crcErr_o     : out sl;
      lenErr_o     : out sl
   );
end MsgStreamRx;
---------------------------------------------------------------------------------------------------
architecture rtl of MsgStreamRx is

   --! State type containing all states  
   type StateType is (
      MOVE_S,
      LAST_S
   );
   
   --! Record containing all register elements
   type RegType is record
      cnt        : natural range 0 to SIZE_G;
      msg32Array : Slv32Array(SIZE_G-1 downto 0);
      lenErr       : sl;
      crcErr       : sl;
      strobe       : sl;
      --
      state      : StateType;
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
      cnt        => 0,
      msg32Array => (others => x"0000_0000"),
      lenErr     => '0',
      crcErr     => '0',
      strobe     => '0',
      --
      state      => MOVE_S
   );
   
   --! internal signals
   signal r      : RegType := REG_INIT_C;
   signal rin    : RegType;
   signal msgReg     : Slv32Array(SIZE_G-1 downto 0);
   signal strobeReg  : sl;
   signal crcReg     : sl;
   signal lenReg     : sl;
---------------------------------------------------------------------------------------------------
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, axisMaster_i) is
      variable v    : RegType;
   begin
      v := r;
      
      --! State Machine
      case r.state is
         ----------------------------------------------------------------------
         when MOVE_S =>
            --! Store the data at the indexed location
            --! It does not matter if tValid is not high because the location
            --! is held at the index address             
            v.msg32Array(r.cnt) := axisMaster_i.tData(31 downto 0);
            
            --! Reset errors
            v.lenErr := '0';
            v.crcErr := '0';
            v.strobe := '0'; 
            
            --! Move the data to the register at tValid
            --! Note that tReady is always high
            if (axisMaster_i.tValid = '1') then
               --! Increment the address
               v.cnt   := r.cnt + 1;
               
               --! Length error if the message is to short
               --! The strobe is generated and the counter is reset
               --! to wait for the next packet
               if (axisMaster_i.tLast = '1') then
                  --! Put the address to 0
                  v.cnt    := 0;
                  v.lenErr := '1';
                  v.crcErr := axisMaster_i.tUser(0);
               end if;
               
               --! Go to tLast
               if (r.cnt = SIZE_G-2) then
                  v.state := LAST_S;
               end if;               
            end if;
         ----------------------------------------------------------------------
         when LAST_S =>
            --! Store the data at the indexed location
            --! It does not matter if tValid is not high because the location
            --! is held at the index address             
            v.msg32Array(r.cnt) := axisMaster_i.tData(31 downto 0);
                        
            --! Move the data to the register at tValid
            --! Note that tReady is always high
            if (axisMaster_i.tValid = '1') then
               --! Put the address to 0
               v.cnt   := 0;
               
               --! Length error if the message is to short
               if (axisMaster_i.tLast = '0' or axisMaster_i.tUser(1) = '1') then
                  v.lenErr := '1';
               end if;
               
               v.crcErr := axisMaster_i.tUser(0);
               
               --! Strobe will be asserted only if the message is valid (STR_SUP_G= True)
               if STR_SUP_G= True then 
                  if (v.lenErr = '0' and v.crcErr = '0') then
                     v.strobe := '1'; 
                  end if;
               --! Strobe will be always asserted (STR_SUP_G= False)
               else
                  v.strobe := '1';               
               end if;
               
               --! Go back to MOVE to wait for the next packet
               v.state := MOVE_S;
            
            end if;
      end case;
     
      --! Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;
      
      --! Register the variable for next clock cycle
      rin <= v;
      
  -----------------------------------------------
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;
   
---------------------------------------------------------------------------------------------------
--! Register output accorfing to generic
---------------------------------------------------------------------------------------------------
   GEN_N_REG: if REGISTER_G = FALSE generate
      --! @brief Just pass on the message
      --! The message will be valid only on strobe
      strobe_o     <= r.strobe;
      msg32Array_o <= r.msg32Array;
      crcErr_o     <= r.crcErr;
      lenErr_o     <= r.lenErr; 
   end generate GEN_N_REG;

   GEN_REG: if REGISTER_G = TRUE generate 
      --! @brief Register the message on strobe
      p_reg: process (clk_i)
      begin
         if rising_edge(clk_i) then
            if rst_i = '1' then
               --! synchronous reset
               msgReg <= REG_INIT_C.msg32Array after TPD_G;
            elsif r.strobe = '1' then
               --! Update the register on strobe
               msgReg <= r.msg32Array after TPD_G;
            else
               --! Hold value
               msgReg <= msgReg after TPD_G;
            end if;
            --! Delay the strobe and errors
            strobeReg <= r.strobe after TPD_G;
            crcReg <= r.crcErr after TPD_G;
            lenReg <= r.lenErr after TPD_G;
         end if;
      end process p_reg;
      
      --! Assign outputs
      strobe_o     <= strobeReg;
      msg32Array_o <= msgReg;
      crcErr_o     <= crcReg;
      lenErr_o     <= lenReg;     
   end generate GEN_REG;

--------------------------------------------------------------------------------------------------
end rtl;
---------------------------------------------------------------------------------------------------