---------------------------------------------------------------------------------------------------
--! @brief   Message transmitter
--! @details
--!    Takes the parallel 32-bit wide message stream and sends it over the AxiStream
--!    strobe_i signal initiates the transfer
--!    full_i connects to FiberLinkTx TxPktFull_o.
--!           full_i prevents the data transmission.
--!           This signal is not required leave unconnected if not used 
--!           (the tReady handshaking will be used if FIFO fills up).
--!    drop_i occurs if the strobe_i  is requested while the packet is busy.
--!    
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Sep 20 2018 last modify
--!
--! @version v0.1
--!
--!
--! @file MsgStreamTx.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;

--! @brief  
--! @details 
---------------------------------------------------------------------------------------------------
entity MsgStreamTx is
generic (
      TPD_G          : time  := 1 ns;
      SIZE_G     : positive  := 8
   );
   port (
      clk_i         : in  sl;
      rst_i         : in  sl;

      --! Parallel message input
      strobe_i     : in sl;
      msg32Array_i : in Slv32Array(SIZE_G-1 downto 0);
      --! Connect TxPktFull_o from FiberLinkTx
      full_i       : in sl := '0';  
      
      --! Output axi stream
      axisMaster_o : out  AxiStreamMasterType;
      axisSlave_i  : in   AxiStreamSlaveType;      
      drop_o       : out sl  
   );
end MsgStreamTx;
---------------------------------------------------------------------------------------------------
architecture rtl of MsgStreamTx is

   --! State type containing all states  
   type StateType is (
      IDLE_S,
      MOVE_S,
      LAST_S
   );
   
   --! Record containing all register elements
   type RegType is record
      cnt        : natural range 0 to SIZE_G;
      msg32Array : Slv32Array(SIZE_G-1 downto 0);
      axisMaster : AxiStreamMasterType;
      drop       : sl;
      --
      state      : StateType;
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
      cnt        => 0,
      msg32Array => (others => x"0000_0000"),
      axisMaster => AXI_STREAM_MASTER_INIT_C,
      drop       => '0',
      --
      state      => IDLE_S
   );
   
   --! internal signals
   signal r      : RegType := REG_INIT_C;
   signal rin    : RegType;
   
---------------------------------------------------------------------------------------------------
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, axisSlave_i, strobe_i, msg32Array_i, full_i) is
      variable v    : RegType;
   begin
      v := r;
      
      --! Reset the flags
      if axisSlave_i.tReady = '1' then
         v.axisMaster.tValid := '0';
         v.axisMaster.tLast  := '0';
      end if;
      
      --! State Machine
      case r.state is
         ----------------------------------------------------------------------
         when IDLE_S =>
            --! Reset the counter
            v.cnt := 0;
            --! Wait for the message
            v.axisMaster := AXI_STREAM_MASTER_INIT_C;
                        
            --! Send the message on strobe
            if (strobe_i = '1' and full_i = '0') then
               --! Save the message
               v.msg32Array      := msg32Array_i;
               -- Next state
               v.state           := MOVE_S;
            end if;
         ----------------------------------------------------------------------
         when MOVE_S =>
            --! Check if ready to move data
            if v.axisMaster.tValid = '0' then
               --
               v.axisMaster.tValid             := '1';
               v.axisMaster.tData(31 downto 0) := r.msg32Array(r.cnt);
               
               --! Increment the counter
               v.cnt := r.cnt + 1;
               
               -- Next state
               if (r.cnt = SIZE_G-2) then             
                  v.state := LAST_S;
               end if;
            end if;
         ----------------------------------------------------------------------
         when LAST_S =>
            --! Check if ready to move data
            if v.axisMaster.tValid = '0' then
               -- Assert tLast
               v.axisMaster.tValid              := '1';
               v.axisMaster.tLast               := '1';
               v.axisMaster.tData(31 downto 0)  := r.msg32Array(r.cnt);
               --
               v.state := IDLE_S;
            end if;
      end case;
      
      --! Check if packets are dropped
      v.drop := '0';
      if (strobe_i = '1' and r.state /= IDLE_S) then
         v.drop := '1';
      end if;
      if (strobe_i = '1' and r.state = IDLE_S and full_i = '1') then
         v.drop := '1';
      end if;      
      
      --! Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;
      
      --! Register the variable for next clock cycle
      rin <= v;
      
      -- Outputs        
      axisMaster_o <= r.axisMaster;
      drop_o       <= r.drop;
                            
   -----------------------------------------------
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;
---------------------------------------------------------------------------------------------------
end rtl;
---------------------------------------------------------------------------------------------------