---------------------------------------------------------------------------------------------------
--! @brief     Status monitor module
--! @details 
--!   Status signals are inputed to the module throug statusBits input vector signal. These signals
--!   are wired to rising edge detection. Status monitor counter is increased when corresponding 
--!   edge is detected on any of the statusBits. Output signal err_o is high when not all of the
--!   monitoring counters are zero. Counters are outputed and ilk_o is pulsed when any of the 
--!   monitoring counter is increased.
--!    
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Oct 15 2018 created
--! @date Dec 28 2018 last modify
--!
--! @version v0.1
--!
--! @file StatusMon.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use work.CslStdRtlPkg.all;
use ieee.numeric_std.all;
library UNISIM;
use UNISIM.VComponents.all;
---------------------------------------------------------------------------------------------------
entity StatusMon is
   generic (
      TPD_G             : time      := 1 ns;
      SIZE_G            : positive  := 4                       --! number of status bits & counters
   );
   port (
      clk_i             : in sl;                               --! clock signal bus
      rst_i             : in sl;                               --! reset signal bus
      clear_i           : in sl;                               --! clear counters signal bus
      statusBits        : in slv(SIZE_G-1 downto 0);           --! input status bits vector signal
   
      ilk_o             : out sl;                              --! interlock output signal => asserted when counters are incresed
      err_o             : out sl;                              --! error output signal => asserted when counters are not zero
      sts32arrayReg_o   : out Slv32Array(SIZE_G -1 downto 0);   --! status counters
      sts32array_o      : out Slv32Array(SIZE_G -1 downto 0) 
   );
end StatusMon;
---------------------------------------------------------------------------------------------------
architecture structure of StatusMon is

   --! internal signals
   signal statusCntInc  : slv(SIZE_G - 1 downto 0);
   signal sts32arrayErr : slv(SIZE_G - 1 downto 0);
   signal counterReg    : Slv32Array(SIZE_G - 1 downto 0) := (others => (others => '0'));
   --! Record containing all register elements
   type RegType is record
      counter        : Slv32Array(SIZE_G - 1 downto 0);      
      sts32arrayErr  : slv(SIZE_G - 1 downto 0);
   end record RegType;
 
   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      counter        => (others => (others => '0')),      
      sts32arrayErr  => (others => '0'));
   
   --! Output of registers. (The Q output)
   signal r : RegType := REG_INIT_C;
   signal rReg   : Slv32Array(SIZE_G - 1 downto 0);
   --! Combinatorial input to registers (The D input)
   signal rin : RegType;
   signal rinReg    : Slv32Array(SIZE_G - 1 downto 0);
   --! 
   signal statNotClear : slv(SIZE_G - 1 downto 0);   
---------------------------------------------------------------------------------------------------  
begin
   --! The status bits are put to '0' for the duration of clear
   --! This enables that the rising edge of the persisting status bit will be detected again after clear
   ST_GEN : for i in SIZE_G - 1 downto 0 generate   
      statNotClear(i) <= statusBits(i) and not clear_i;
   end generate ST_GEN;
   
   --! Edge detector module instatiation
   u0_CslEdgeDetVec : entity work.CslEdgeDetVec
      generic map (
         TPD_G          => TPD_G,
         FALLING_EDGE_G => false,
         WIDTH_G        => SIZE_G
      )
      port map (
         clk_i => clk_i,
         rst_i => rst_i,
         sig_i => statNotClear,
         sig_o => statusCntInc
      );

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   p_Comb : process (r, statusCntInc, clear_i, rst_i) is
      variable v : RegType;
   begin
      --! Initialize v with current value of all registers
      v        := r;
      
      for i in SIZE_G - 1 downto 0 loop
         if statusCntInc(i) = '1' then
            --! edge on status bit detected
             if r.counter(i) < x"FFFFFFFF" then
               --! counter is increased if not staurated to maximum value
             v.counter(i) := slv(unsigned(r.counter(i)) + to_unsigned(1, 32));
             counterReg(i) <= slv(unsigned(counterReg(i)) + to_unsigned(1, 32));      
             end if;
         end if;
         if r.counter(i) > x"00000000" then
            --! counter is not zero => error
            v.sts32arrayErr(i) := '1';
         else  
            --! counter is zero => no error
            v.sts32arrayErr(i) := '0';
         end if;
      end loop;
                              
      if (clear_i = '1') then
         --! clear counters
         v := REG_INIT_C;
      end if;
      
      --! Synchronous reset
      --! Override all above assignments and apply init values
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      --! Assign final state of local variable to rin signal
      --! Registers will assume these values on next rising edge
      rin <= v;
      rinReg <= counterReg;
      
      --! assign outputs
      ilk_o          <= uOr(statusCntInc);
      sts32array_o   <= r.counter;
      sts32arrayReg_o   <= rReg;
      err_o          <= uOr(r.sts32arrayErr);
      
   end process p_Comb;
 
   --! Boilerplate sequential process
   --! Assign rin to r on rising edge of clk to create registers
   p_Seq : process (clk_i) is begin
       if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
         rReg <= rinReg after TPD_G;
       end if;
   end process p_Seq;
   
end structure;
---------------------------------------------------------------------------------------------------