---------------------------------------------------------------------------------------------------
--! @brief     Watchdog module
--! @details      
--!   The watchdog generates output high signal if the module receives no ping in dedicated time.
--!   The time for countdown counter is real time configurable through input signal vector port.
--!   The module can also be used as a link OK module. If used in this scenario, output of high state
--!   holds the information of error free design.
--!    
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Oct 15 2018 created
--! @date Dec 28 2018 last modify
--!
--! @version v0.1
--!
--! @file Watchdog.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
library UNISIM;
use UNISIM.VComponents.all;
---------------------------------------------------------------------------------------------------
entity Watchdog is
   generic (
      TPD_G             : time      := 1 ns;
      WIDTH_G           : positive  := 4                 --! width of vector for counters& time threshold
   );
   port (
      clk_i             : in sl;                         --! clock signal bus
      rst_i             : in sl;                         --! reset signal bus
      en_i              : in sl;                         --! module enable signal
      wdRst_i           : in sl;                         --! watcdog ping/reset signal
      time_i            : in slv(WIDTH_G -1 downto 0);   --! watchdog timer threshold value
         
      tout_o            : out sl                         --! output signal
   );
end Watchdog;
---------------------------------------------------------------------------------------------------
architecture structure of Watchdog is

   --! Record containing all register elements
   type RegType is record
      counter     : unsigned(WIDTH_G - 1 downto 0);
   end record RegType;
 
   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      counter     => (others => '0'));
   
   --! Output of registers. (The Q output)
   signal r : RegType := REG_INIT_C;
   
   --! Combinatorial input to registers (The D input)
   signal rin : RegType;
---------------------------------------------------------------------------------------------------  
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   p_Comb : process (r, wdRst_i, time_i, en_i, rst_i) is
      variable v : RegType;
   begin
      --! Initialize v with current value of all registers
      v        := r;
      
      
      if en_i =  '1' then
         --! the module is enabled
         if r.counter > 0 then
            --! downcount to zero
            v.counter := r.counter - 1;
         end if;
      end if;
      
      if wdRst_i =  '1' and en_i = '1' then
         --! received watchdog reset/ping from outside => reset counter
         v.counter := unsigned(time_i);
      end if;
                  
      --! Synchronous reset
      --! Override all above assignments and apply init values
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;
      
      --! Assign final state of local variable to rin signal
      --! Registers will assume these values on next rising edge
      rin <= v;
      
      --! assign outputs
      if r.counter = 0 then
         --! watchdog was not reseted/pinged in time, ouptut is high
         tout_o   <= '1';
      else
         --! watchdog was reseted/pinged in time, ouptut is low
         tout_o   <= '0';
      end if;
   end process p_Comb;
 
   --! Boilerplate sequential process
   --! Assign rin to r on rising edge of clk to create registers
   p_Seq : process (clk_i) is begin
       if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
       end if;
   end process p_Seq;

end structure;
---------------------------------------------------------------------------------------------------