---------------------------------------------------------------------------------------------------
--! @brief Config Ctrl test bench 
--! @details
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Dec 12 2018 created
--! @date Aug 29 2018 last modify
--!
--! @version v0.1
--!
--! @file ConfigCtrlTb.vhd
---------------------------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.ItuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity ConfigCtrlTb is
end entity ;
--------------------------------------------------------------------------------
architecture Bhv of ConfigCtrlTb is
   -----------------------------
   -- Port Signals 
   -----------------------------
   -- Clock reset
   constant CLK_PERIOD_C      : time    := 10 ns;
   constant CLK_DELAY_C       : time    := 0 ns;
   constant RST_START_DELAY_C : time    := 1 ns;
   constant RST_HOLD_TIME_C   : time    := 100 ns;
   constant SYNC_RESET_C      : boolean := true;
   --
   constant TPD_C          : time                   := 1 ns;
   constant DATA_WIDTH_C   : natural                := 4;
   constant PIPES_C        : integer range 1 to 255 := 1;
   constant IDLE_C         : slv(8 downto 0)        := '1' & x"BC";
   constant SOF_CHAR_C     : slv(7 downto 0)        := x"1C";
   constant EOF_CHAR_C     : slv(7 downto 0)        := x"FE";
   constant HPD_MAX_SIZE_C : positive               := 4;
   constant LPD_MAX_SIZE_C : positive               := 24;

   constant DEVICE_ADDR_C : natural  := 0;
   constant TOUT_C        : positive := 1000;
   constant BURST_SIZE_C  : positive := 16;
   constant WIDTH_C       : natural  := 32;
   --
   signal clk_i : sl := '0';
   signal rst_i : sl := '1';

   signal testRegWriteAllow : sl := '1';
   signal axiReadMaster     : AxiLiteReadMasterType;
   signal axiReadSlave      : AxiLiteReadSlaveType;
   signal axiWriteMaster    : AxiLiteWriteMasterType;
   signal axiWriteSlave     : AxiLiteWriteSlaveType;

   signal testReg_o : Slv32Array(15 downto 0);
   signal cfgMsg_i  : CfgMsgType;
   signal cfgMsg_o  : CfgMsgType;
   signal dropMsg_o : sl;

   --! Input message
   signal strobe_i          : slv(PIPES_C downto 0)                 := (others => '0');
   signal Msg32Array_i      : Slv32Array(15 downto 0)               := (others => x"01234567");
   signal hpdTxMsg32Array_i : Slv32Array(HPD_MAX_SIZE_C-1 downto 0) := (others => x"01234567");
   signal lpdTxMsg32Array_i : Slv32Array(LPD_MAX_SIZE_C-1 downto 0) := (others => x"01234567");
   type lpdMsg32Array2D is array (PIPES_C-1 downto 0) of Slv32Array(LPD_MAX_SIZE_C-1 downto 0);
   signal lpdRxMsg32Array2D_o : lpdMsg32Array2D;
   signal hpdRxMsg32array_o   : Slv32Array(HPD_MAX_SIZE_C-1 downto 0);
   --! TX 
   signal hpdTxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdTxAxisSlave  : AxiStreamSlaveType;
   signal hpdTxPktFull_o  : sl;
   signal hpdTxErr_o      : slv(1 downto 0);
   signal lpdTxAxisMaster : AxiStreamMasterArray(PIPES_C-1 downto 0) := (others => AXI_STREAM_MASTER_INIT_C);
   signal lpdTxAxisSlave  : AxiStreamSlaveArray(PIPES_C-1 downto 0);
   signal lpdTxPktFull_o  : slv(PIPES_C-1 downto 0);
   signal lpdTxErr_o      : Slv2Array(PIPES_C-1 downto 0);

   --! GT Data
   signal gtData : slv(15 downto 0);
   signal gtChar : slv(1 downto 0);

   signal link_i          : sl                  := '1';
   signal hpdRxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdRxAxisSlave  : AxiStreamSlaveType;
   signal hpdRxPktEmpty_o : sl;
   signal hpdRxDrop_o     : sl;
   signal lpdRxAxisMaster : AxiStreamMasterArray(PIPES_C-1 downto 0) := (others => AXI_STREAM_MASTER_INIT_C);
   signal lpdRxAxisSlave  : AxiStreamSlaveArray(PIPES_C-1 downto 0);
   signal lpdRxPktEmpty_o : slv(PIPES_C-1 downto 0);
   signal lpdRxDrop_o     : sl;
   --! Output message
   signal strobe_o : slv(PIPES_C downto 0);
   signal drop_o   : slv(PIPES_C downto 0);
   signal crcErr_o : slv(PIPES_C downto 0);
   signal lenErr_o : slv(PIPES_C downto 0);

   signal testReg_i : Slv32Array(15 downto 0) := (others => (others => '0'));


begin -- architecture Bhv
      -----------------------------
      -- component instantiation 
      -----------------------------

   --! HPD Message transmitter   
   u_MsgStreamTx : entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_C,
         SIZE_G => HPD_MAX_SIZE_C)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => strobe_i(PIPES_C),
         msg32Array_i => hpdTxMsg32Array_i,
         full_i       => hpdTxPktFull_o,
         axisMaster_o => hpdTxAxisMaster,
         axisSlave_i  => hpdTxAxisSlave,
         drop_o       => drop_o(PIPES_C));

   --! LPD Message transmitters
   GEN_PIPES_I :
   for i in PIPES_C-1 downto 0 generate
      u_MsgStreamTx : entity work.MsgStreamTx
         generic map (
            TPD_G  => TPD_C,
            SIZE_G => LPD_MAX_SIZE_C)
         port map (
            clk_i        => clk_i,
            rst_i        => rst_i,
            strobe_i     => strobe_i(i),
            msg32Array_i => lpdTxMsg32Array_i,
            full_i       => lpdTxPktFull_o(i),
            axisMaster_o => lpdTxAxisMaster(i),
            axisSlave_i  => lpdTxAxisSlave(i),
            drop_o       => drop_o(i));
   end generate GEN_PIPES_I;

   u_FiberLinkMapTx : entity work.FiberLinkMapTx
      generic map (
         TPD_G            => TPD_C,
         HPD_DATA_WIDTH_G => DATA_WIDTH_C,
         HPD_IDLE_G       => IDLE_C,
         HPD_SOF_CHAR_G   => SOF_CHAR_C,
         HPD_EOF_CHAR_G   => EOF_CHAR_C,
         HPD_MAX_SIZE_G   => HPD_MAX_SIZE_C,
         LPD_DATA_WIDTH_G => DATA_WIDTH_C,
         LPD_PIPES_G      => PIPES_C,
         LPD_IDLE_G       => IDLE_C,
         LPD_SOF_CHAR_G   => SOF_CHAR_C,
         LPD_EOF_CHAR_G   => EOF_CHAR_C,
         LPD_MAX_SIZE_G   => LPD_MAX_SIZE_C)
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         hpdTxAxisMaster_i => hpdTxAxisMaster,
         hpdTxAxisSlave_o  => hpdTxAxisSlave,
         hpdTxPktFull_o    => hpdTxPktFull_o,
         hpdTxErr_o        => hpdTxErr_o,
         lpdTxAxisMaster_i => lpdTxAxisMaster,
         lpdTxAxisSlave_o  => lpdTxAxisSlave,
         lpdTxPktFull_o    => lpdTxPktFull_o,
         lpdTxErr_o        => lpdTxErr_o,
         gtDataTx_o        => gtData,
         gtCharTx_o        => gtChar);



   u_FiberLinkMapRx : entity work.FiberLinkMapRx
      generic map (
         TPD_G            => TPD_C,
         HPD_DATA_WIDTH_G => DATA_WIDTH_C,
         HPD_IDLE_G       => IDLE_C,
         HPD_SOF_CHAR_G   => SOF_CHAR_C,
         HPD_EOF_CHAR_G   => EOF_CHAR_C,
         HPD_MAX_SIZE_G   => HPD_MAX_SIZE_C,
         LPD_DATA_WIDTH_G => DATA_WIDTH_C,
         LPD_PIPES_G      => PIPES_C,
         LPD_IDLE_G       => IDLE_C,
         LPD_SOF_CHAR_G   => SOF_CHAR_C,
         LPD_EOF_CHAR_G   => EOF_CHAR_C,
         LPD_MAX_SIZE_G   => LPD_MAX_SIZE_C)
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         gtDataRx_i        => gtData,
         gtCharRx_i        => gtChar,
         link_i            => link_i,
         hpdRxAxisMaster_o => hpdRxAxisMaster,
         hpdRxAxisSlave_i  => hpdRxAxisSlave,
         hpdRxPktEmpty_o   => hpdRxPktEmpty_o,
         hpdRxDrop_o       => hpdRxDrop_o,
         lpdRxAxisMaster_o => lpdRxAxisMaster,
         lpdRxAxisSlave_i  => lpdRxAxisSlave,
         lpdRxPktEmpty_o   => lpdRxPktEmpty_o,
         lpdRxDrop_o       => lpdRxDrop_o);

   u_MsgStreamRx : entity work.MsgStreamRx
      generic map (
         TPD_G      => TPD_C,
         SIZE_G     => HPD_MAX_SIZE_C,
         REGISTER_G => true)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => hpdRxAxisMaster,
         axisSlave_o  => hpdRxAxisSlave,
         strobe_o     => strobe_o(PIPES_C),
         msg32Array_o => hpdRxMsg32array_o,
         crcErr_o     => crcErr_o(PIPES_C),
         lenErr_o     => lenErr_o(PIPES_C));

   GEN_PIPES_O :
   for i in PIPES_C-1 downto 0 generate
      u_MsgStreamRx : entity work.MsgStreamRx
         generic map (
            TPD_G      => TPD_C,
            SIZE_G     => LPD_MAX_SIZE_C,
            REGISTER_G => true)
         port map (
            clk_i        => clk_i,
            rst_i        => rst_i,
            axisMaster_i => lpdRxAxisMaster(i),
            axisSlave_o  => lpdRxAxisSlave(i),
            strobe_o     => strobe_o(i),
            msg32Array_o => lpdRxMsg32Array2D_o(i),
            crcErr_o     => crcErr_o(i),
            lenErr_o     => lenErr_o(i));
   end generate GEN_PIPES_O;

   cfgMsg_i <= encCfgFec(lpdRxMsg32Array2D_o(PIPES_C-1), strobe_o(PIPES_C-1));

   u0_ConfigCtrl : entity work.ConfigCtrl
      generic map (
         TPD_G         => TPD_C,
         DEVICE_ADDR_G => DEVICE_ADDR_C,
         TOUT_G        => TOUT_C,
         BURST_SIZE_G  => BURST_SIZE_C
      )
      port map (
         clk_i         => clk_i,
         rst_i         => rst_i,
         crcErr_i      => crcErr_o(PIPES_C),
         lenErr_i      => lenErr_o(PIPES_C),
         cfgMsg_i      => cfgMsg_i,
         parErr_i      => '0',
         writeMaster_o => axiWriteMaster,
         writeSlave_i  => axiWriteSlave,
         readMaster_o  => axiReadMaster,
         readSlave_i   => axiReadSlave,
         cfgMsg_o      => cfgMsg_o,
         drop_o        => dropMsg_o
      );

   u0_TestReg : entity work.TestReg
      generic map (
         TPD_G   => TPD_C,
         WIDTH_G => WIDTH_C
      )
      port map (
         axiClk         => clk_i,
         axiRst         => rst_i,
         axiWriteAllow  => testRegWriteAllow,
         axiReadMaster  => axiReadMaster,
         axiReadSlave   => axiReadSlave,
         axiWriteMaster => axiWriteMaster,
         axiWriteSlave  => axiWriteSlave,
         testReg_i      => testReg_i,
         testReg_o      => testReg_o
      );

   -- Clock
   process is
   begin
      wait for CLK_DELAY_C;
      while (true) loop
         clk_i <= not clk_i;
         wait for CLK_PERIOD_C/2.0;
      end loop;
   end process;

   -- Proces for generatig input array message
   process(clk_i,rst_i)
   begin
      if rising_edge(clk_i) then
         if rst_i = '1' then
            Msg32Array_i <= (others => x"00000000");
         else
            for i in 0 to 15 loop
               Msg32Array_i(i) <= std_logic_vector(unsigned(Msg32Array_i(i)) + to_unsigned((i+1)*10,32));
            end loop;
         end if;
      end if;
   end process;

   -- Reset
   process is
   begin
      rst_i <= '1';
      wait for RST_START_DELAY_C;
      if (SYNC_RESET_C) then
         wait until clk_i = '1';
      end if;
      rst_i <= '1';
      wait for RST_HOLD_TIME_C;
      if (SYNC_RESET_C) then
         wait until clk_i = '1';
      end if;
      rst_i <= '0';
      wait;
   end process;

   lpdTxMsg32Array_i <= (0 => x"01000002",
         1  => x"00000000",
         2  => x"00000000",
         3  => x"0000000F",
         4  => x"00000000",
         5  => x"00000000",
         6  => x"00000000",
         7  => x"00000000",
         8  => Msg32Array_i(0),
         9  => Msg32Array_i(1),
         10 => Msg32Array_i(2),
         11 => Msg32Array_i(3),
         12 => Msg32Array_i(4),
         13 => Msg32Array_i(5),
         14 => Msg32Array_i(6),
         15 => Msg32Array_i(7),
         16 => Msg32Array_i(8),
         17 => Msg32Array_i(9),
         18 => Msg32Array_i(10),
         19 => Msg32Array_i(11),
         20 => Msg32Array_i(12),
         21 => Msg32Array_i(13),
         22 => Msg32Array_i(14),
         23 => Msg32Array_i(15)
   );

   hpdTxMsg32Array_i <= lpdTxMsg32Array_i(23 downto 20);

   p_StimuliProcess : process
   begin
      -- wait until rst_i = '0';

      wait for 10 * RST_HOLD_TIME_C;
      wait for 5*CLK_PERIOD_C;
      wait until clk_i = '1';
      strobe_i <= '1'& '0';
      wait until clk_i = '1';
      strobe_i <= '0'& '0';
      wait for RST_HOLD_TIME_C;

      wait for 3*CLK_PERIOD_C;
      wait until clk_i = '1';
      strobe_i <= '1'& '1';
      wait until clk_i = '1';
      strobe_i <= '0'& '0';
   --wait;
   -- wait for 10*CLK_PERIOD_C;     
   end process p_StimuliProcess;

   -- write allow
   process is
   begin
      wait for 1000*CLK_PERIOD_C;
      while (true) loop
         testRegWriteAllow <= not testRegWriteAllow;
         wait for 1000*CLK_PERIOD_C;
      end loop;
   end process;

end architecture Bhv;