---------------------------------------------------------------------------------------------------
--! @brief Fiber Link transmitter test bench 
--! @details
--!
--!
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Sep 20 2018 last modify
--!
--! @version v0.1
--!
--!
--! @file FiberLinkMapTb.vhd
---------------------------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

use work.CslStdRtlPkg.all;
use work.AxiStreamPkg.all;

entity  FiberLinkMapTb is

end entity ;
--------------------------------------------------------------------------------


architecture Bhv of FiberLinkMapTb is
  -----------------------------
  -- Port Signals 
  -----------------------------
   -- Clock reset
   constant CLK_PERIOD_C      : time    := 10 ns;
   constant CLK_DELAY_C       : time    := 0 ns;
   constant RST_START_DELAY_C : time    := 1 ns; 
   constant RST_HOLD_TIME_C   : time    := 100 ns;
   constant SYNC_RESET_C      : boolean := true;
   --
   constant TPD_C        : time                    := 1 ns;
   constant DATA_WIDTH_C : natural                 := 4;
   constant PIPES_C      : integer range 1 to 255  := 4;
   constant IDLE_C       : slv(8 downto 0)         := '1' & x"BC";
   constant SOF_CHAR_C   : slv(7 downto 0)         := x"1C";  
   constant EOF_CHAR_C   : slv(7 downto 0)         := x"FE";  
   constant MAX_SIZE_C   : positive                := 8;
   --
   signal   clk_i        : sl:= '0';
   signal   rst_i        : sl:= '1';
   
   --! Input message
   signal   strobe_i          : slv(PIPES_C downto 0)            := '0'& x"0";  
   signal   msg32Array_i      : Slv32Array(MAX_SIZE_C-1 downto 0):= (others=> x"01234567");
   type msg32Array2D is array (PIPES_C downto 0) of Slv32Array(MAX_SIZE_C-1 downto 0);
   signal   msg32array2D_o    : msg32Array2D;
   --! TX 
   signal   hpdTxAxisMaster   : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal   hpdTxAxisSlave    : AxiStreamSlaveType;
   signal   hpdTxPktFull_o    : sl;
   signal   hpdTxErr_o        : slv(1 downto 0);
   signal   lpdTxAxisMaster   : AxiStreamMasterArray(PIPES_C-1 downto 0):=(others => AXI_STREAM_MASTER_INIT_C);
   signal   lpdTxAxisSlave    : AxiStreamSlaveArray(PIPES_C-1 downto 0);
   signal   lpdTxPktFull_o    : slv(PIPES_C-1 downto 0);
   signal   lpdTxErr_o        : Slv2Array(PIPES_C-1 downto 0);
   --! GT Data
   signal   gtData            : slv(15 downto 0);
   signal   gtChar            : slv(1 downto 0);
   
   signal   gtDataCorrupt     : slv(15 downto 0);
   signal   gtCharCorrupt     : slv(1 downto 0);
   signal   injId_i           : sl  := '0';
   signal   injCrc_i          : sl  := '0';
   signal   injLen_i          : sl  := '0';
   signal   id_i              : slv(2 downto 0) := (others => '0');
   --! RX
   signal   link_i            : sl:='1';
   signal   hpdRxAxisMaster   : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal   hpdRxAxisSlave    : AxiStreamSlaveType;
   signal   hpdRxPktEmpty_o   : sl;
   signal   hpdRxDrop_o       : sl;
   signal   lpdRxAxisMaster   : AxiStreamMasterArray(PIPES_C-1 downto 0):=(others => AXI_STREAM_MASTER_INIT_C);
   signal   lpdRxAxisSlave    : AxiStreamSlaveArray(PIPES_C-1 downto 0);
   signal   lpdRxPktEmpty_o   : slv(PIPES_C-1 downto 0);
   signal   lpdRxDrop_o       : sl;
   --! Output message
   signal   strobe_o          : slv(PIPES_C downto 0);
   signal   drop_o            : slv(PIPES_C downto 0);
   signal   crcErr_o          : slv(PIPES_C downto 0);
   signal   lenErr_o          : slv(PIPES_C downto 0);
   --! This is commented out because the port is open and the output message should be observed inside the HDL sim 
   -- signal   msg32Array_o : Slv32Array(MAX_SIZE_C-1 downto 0);
   
   
begin  -- architecture Bhv

   -----------------------------
   -- component instantiation 
   -----------------------------

   --! HPD Message transmitter   
   u_MsgStreamTx: entity work.MsgStreamTx
   generic map (
      TPD_G  => TPD_C,
      SIZE_G => MAX_SIZE_C)
   port map (
      clk_i        => clk_i,
      rst_i        => rst_i,
      strobe_i     => strobe_i(PIPES_C),
      msg32Array_i => msg32Array_i,
      full_i       => hpdTxPktFull_o,
      axisMaster_o => hpdTxAxisMaster,
      axisSlave_i  => hpdTxAxisSlave,
      drop_o       => drop_o(PIPES_C));
  
   --! LPD Message transmitters
   GEN_PIPES_I :
   for i in PIPES_C-1 downto 0 generate
      u_MsgStreamTx: entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_C,
         SIZE_G => MAX_SIZE_C)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => strobe_i(i),
         msg32Array_i => msg32Array_i,
         full_i       => lpdTxPktFull_o(i),
         axisMaster_o => lpdTxAxisMaster(i),
         axisSlave_i  => lpdTxAxisSlave(i),
         drop_o       => drop_o(i));
   end generate GEN_PIPES_I;
   
   u_FiberLinkMapTx: entity work.FiberLinkMapTx
   generic map (
      TPD_G            => TPD_C,
      HPD_DATA_WIDTH_G => DATA_WIDTH_C,
      HPD_IDLE_G       => IDLE_C,
      HPD_SOF_CHAR_G   => SOF_CHAR_C,
      HPD_EOF_CHAR_G   => EOF_CHAR_C,
      HPD_MAX_SIZE_G   => MAX_SIZE_C,
      LPD_DATA_WIDTH_G => DATA_WIDTH_C,
      LPD_PIPES_G      => PIPES_C,
      LPD_IDLE_G       => IDLE_C,
      LPD_SOF_CHAR_G   => SOF_CHAR_C,
      LPD_EOF_CHAR_G   => EOF_CHAR_C,
      LPD_MAX_SIZE_G   => MAX_SIZE_C)
   port map (
      clk_i             => clk_i,
      rst_i             => rst_i,
      hpdTxAxisMaster_i => hpdTxAxisMaster,
      hpdTxAxisSlave_o  => hpdTxAxisSlave,
      hpdTxPktFull_o    => hpdTxPktFull_o,
      hpdTxErr_o        => hpdTxErr_o,
      lpdTxAxisMaster_i => lpdTxAxisMaster,
      lpdTxAxisSlave_o  => lpdTxAxisSlave,
      lpdTxPktFull_o    => lpdTxPktFull_o,
      lpdTxErr_o        => lpdTxErr_o,
      gtDataTx_o        => gtData,
      gtCharTx_o        => gtChar);

      
   u0_InjectErrFiber : entity work.InjectErrFiber
      generic map (
         TPD_G      => TPD_C,
         SOF_CHAR_G => SOF_CHAR_C,
         EOF_CHAR_G => EOF_CHAR_C
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         en_i              => '1',
         injId_i           => '0',
         id_i              => (others => '0'),
         injCrc_i          => injCrc_i,
         injLen_i          => injLen_i,
         gtTxData_i        => gtData(15 downto 8),
         gtTxChar_i        => gtChar(1),
         gtCorruptTxData_o => gtDataCorrupt(15 downto 8),
         gtCorruptTxChar_o => gtCharCorrupt(1)
      );
      
   u1_InjectErrFiber : entity work.InjectErrFiber
      generic map (
         TPD_G      => TPD_C,
         SOF_CHAR_G => SOF_CHAR_C,
         EOF_CHAR_G => EOF_CHAR_C
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         en_i              => '1',
         injId_i           => injId_i,
         id_i(7 downto 3)  => (others => '0'),
         id_i(2 downto 0)  => id_i,
         injCrc_i          => injCrc_i,
         injLen_i          => injLen_i,
         gtTxData_i        => gtData(7 downto 0),
         gtTxChar_i        => gtChar(0),
         gtCorruptTxData_o => gtDataCorrupt(7 downto 0),
         gtCorruptTxChar_o => gtCharCorrupt(0)
      );
         
   u_FiberLinkMapRx: entity work.FiberLinkMapRx
   generic map (
      TPD_G            => TPD_C,
      HPD_DATA_WIDTH_G => DATA_WIDTH_C,
      HPD_IDLE_G       => IDLE_C,
      HPD_SOF_CHAR_G   => SOF_CHAR_C,
      HPD_EOF_CHAR_G   => EOF_CHAR_C,
      HPD_MAX_SIZE_G   => MAX_SIZE_C,
      LPD_DATA_WIDTH_G => DATA_WIDTH_C,
      LPD_PIPES_G      => PIPES_C,
      LPD_IDLE_G       => IDLE_C,
      LPD_SOF_CHAR_G   => SOF_CHAR_C,
      LPD_EOF_CHAR_G   => EOF_CHAR_C,
      LPD_MAX_SIZE_G   => MAX_SIZE_C)
   port map (
      clk_i             => clk_i,
      rst_i             => rst_i,
      gtDataRx_i        => gtDataCorrupt,
      gtCharRx_i        => gtCharCorrupt,
      link_i            => link_i,
      hpdRxAxisMaster_o => hpdRxAxisMaster,
      hpdRxAxisSlave_i  => hpdRxAxisSlave,
      hpdRxPktEmpty_o   => hpdRxPktEmpty_o,
      hpdRxDrop_o       => hpdRxDrop_o,
      lpdRxAxisMaster_o => lpdRxAxisMaster,
      lpdRxAxisSlave_i  => lpdRxAxisSlave,
      lpdRxPktEmpty_o   => lpdRxPktEmpty_o,
      lpdRxDrop_o       => lpdRxDrop_o);
   
   u_MsgStreamRx: entity work.MsgStreamRx
   generic map (
      TPD_G      => TPD_C,
      SIZE_G     => MAX_SIZE_C,
      REGISTER_G => true)
   port map (
      clk_i        => clk_i,
      rst_i        => rst_i,
      axisMaster_i => hpdRxAxisMaster,
      axisSlave_o  => hpdRxAxisSlave,
      strobe_o     => strobe_o(PIPES_C),
      msg32Array_o => msg32array2D_o(PIPES_C),
      crcErr_o     => crcErr_o(PIPES_C),
      lenErr_o     => lenErr_o(PIPES_C));

   GEN_PIPES_O :
   for i in PIPES_C-1 downto 0 generate
      u_MsgStreamRx: entity work.MsgStreamRx
      generic map (
         TPD_G      => TPD_C,
         SIZE_G     => MAX_SIZE_C,
         REGISTER_G => true)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => lpdRxAxisMaster(i),
         axisSlave_o  => lpdRxAxisSlave(i),
         strobe_o     => strobe_o(i),
         msg32Array_o => msg32array2D_o(i),
         crcErr_o     => crcErr_o(i),
         lenErr_o     => lenErr_o(i));
   end generate GEN_PIPES_O;
  
   -- Clock
	process is
   begin
      wait for CLK_DELAY_C;
      while (true) loop
         clk_i <= not clk_i;
         wait for CLK_PERIOD_C/2.0;
      end loop;
   end process;
   
   -- inject crc error
	process is
   begin
      wait for 1000*CLK_PERIOD_C/2.0;
      injCrc_i <= not injCrc_i;
      wait for CLK_PERIOD_C;
      injCrc_i <= not injCrc_i;
   end process;
   
   -- inject len error
	process is
   begin
      wait for 2000*CLK_PERIOD_C/2.0;
      injLen_i <= not injLen_i;
      wait for CLK_PERIOD_C;
      injLen_i <= not injLen_i;
   end process;
   
   -- inject len error
	process is
   begin
      wait for 4000*CLK_PERIOD_C/2.0;
      injId_i <= not injId_i;
      wait for CLK_PERIOD_C;
      injId_i <= not injId_i;
   end process;
   
   -- Reset
   process is
   begin
      rst_i <= '1';
      wait for RST_START_DELAY_C;
      if (SYNC_RESET_C) then
         wait until clk_i = '1';
      end if;
         rst_i <= '1';
         wait for RST_HOLD_TIME_C;
      if (SYNC_RESET_C) then
         wait until clk_i = '1';
      end if;
         rst_i <= '0';
      wait;
   end process;
   
   process (clk_i) is
   begin
      if rising_edge(clk_i) then
         id_i <= std_logic_vector(unsigned(id_i) +to_unsigned(1, 3));
      end if;
   end process;
   --
   p_StimuliProcess : process
   begin
     -- wait until rst_i = '0';
      msg32Array_i <= (
         0 => x"01234567",
         1 => x"89ABCDEF",      
         2 => x"01234567",
         3 => x"89ABCDEF",
         4 => x"01234567",
         5 => x"89ABCDEF",      
         6 => x"01234567",
         7 => x"89ABCDEF"
       --  8 => x"DEADBEEF"
      );
         
      wait for 10 * RST_HOLD_TIME_C;
      wait for 5*CLK_PERIOD_C;
      wait until clk_i = '1';
      strobe_i <= '1'& x"9";
      wait until clk_i = '1';
      strobe_i <= '0'& x"0";
      wait for RST_HOLD_TIME_C;
      
      msg32Array_i <= (
         0 => x"ABCDABCD",
         1 => x"DEADBEEF",      
         2 => x"BA5EBA11",
         3 => x"1CEB00DA",
         4 => x"01234567",
         5 => x"89ABCDEF",      
         6 => x"01234567",
         7 => x"89ABCDEF"
       --  8 => x"DEADBEEF"
      );
      
      
      wait for 3*CLK_PERIOD_C;
      wait until clk_i = '1';
      strobe_i <= '1'& x"e";
      wait until clk_i = '1';
      strobe_i <= '0'& x"0";
      --wait;
     -- wait for 10*CLK_PERIOD_C;     
   end process p_StimuliProcess;
  
end architecture Bhv;